/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MolInfo.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1997/03/23 07:27:52 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   This is a helper function for the 'molinfo # get ...' command in
 * TclCommands.h .  It is used to access the information known about
 * molecules.
 *
 ***************************************************************************/

#include <stdlib.h> 
#include "Global.h"
#include "SymbolTable.h"
#include "MoleculeList.h"
#include "MoleculeFile.h"
#include "tcl.h"
#include "Timestep.h"
#include "TclCommands.h" // for my own, external function definitions

SymbolTable molInfoTable;

static int molinfo_get_index;
static int molinfo_get_id;
static int molinfo_frame_number;
static char *molinfo_get_string;

#define generic_molinfo_data(fctnname, fromwhere) \
static int fctnname(int, int *data, int *)        \
{                                                 \
  data[0] = fromwhere;                            \
  return 1;                                       \
}
#define generic_molinfo_fdata(fctnname, fromwhere) \
static int fctnname(int, double *data, int *)        \
{                                                 \
  data[0] = fromwhere;                            \
  return 1;                                       \
}

#define generic_molinfo_sdata(fctnname, fromwhere) \
static int fctnname(int, GString *data, int *)     \
{                                                  \
  data[0] = fromwhere;                             \
  return 1;                                        \
}

#define generic_molinfo_set_data(fctnname, fromwhere) \
static int fctnname(int, int *data, int *)        \
{                                                 \
  fromwhere;                                      \
  return 1;                                       \
}
#define generic_molinfo_set_fdata(fctnname, fromwhere) \
static int fctnname(int, double *data, int *)        \
{                                                 \
  fromwhere;                                      \
  return 1;                                       \
}

// return the id
generic_molinfo_data(molinfo_id, molinfo_get_id);

// return index
generic_molinfo_data(molinfo_index, molinfo_get_index);

// return the number of atoms
generic_molinfo_data(molinfo_numatoms, moleculeList->
		     molecule(molinfo_get_index)->nAtoms);

// return or set the alpha/beta/gamma and a/b/c information
generic_molinfo_fdata(molinfo_alpha, moleculeList->
		     molecule(molinfo_get_index)->alpha);
generic_molinfo_fdata(molinfo_beta, moleculeList->
		     molecule(molinfo_get_index)->beta);
generic_molinfo_fdata(molinfo_gamma, moleculeList->
		     molecule(molinfo_get_index)->gamma);
generic_molinfo_fdata(molinfo_a, moleculeList->
		     molecule(molinfo_get_index)->a_length);
generic_molinfo_fdata(molinfo_b, moleculeList->
		     molecule(molinfo_get_index)->b_length);
generic_molinfo_fdata(molinfo_c, moleculeList->
		     molecule(molinfo_get_index)->c_length);

generic_molinfo_set_fdata(molinfo_set_alpha, moleculeList->
			 molecule(molinfo_get_index)->alpha = data[0]);
generic_molinfo_set_fdata(molinfo_set_beta, moleculeList->
			 molecule(molinfo_get_index)->beta = data[0]);
generic_molinfo_set_fdata(molinfo_set_gamma, moleculeList->
			 molecule(molinfo_get_index)->gamma = data[0]);
generic_molinfo_set_fdata(molinfo_set_a, moleculeList->
			 molecule(molinfo_get_index)->a_length = data[0]);
generic_molinfo_set_fdata(molinfo_set_b, moleculeList->
			 molecule(molinfo_get_index)->b_length = data[0]);
generic_molinfo_set_fdata(molinfo_set_c, moleculeList->
			 molecule(molinfo_get_index)->c_length = data[0]);



// where does this come from (data file, remote connection, etc.)
generic_molinfo_sdata(molinfo_source,
     data[0] = moleculeList -> molecule(molinfo_get_index)->source);

// what file/computer does this come from?
static int molinfo_name(int, GString *data, int *)
{
  Molecule *mol = moleculeList -> molecule(molinfo_get_index);
  if (!strcmp(mol -> source, "File") ||
      !strcmp(mol -> source, "Graphics")) {
    data[0] = ((MoleculeFile *)mol)->file_name();
  } else {
    data[0] = mol -> source;
  }
  return 1;
}

// return the full path+filename
static int molinfo_filename(int, GString *data, int *)
{
  Molecule *mol = moleculeList -> molecule(molinfo_get_index);
  if (!strcmp(mol -> source, "File")) {
    data[0] = ((MoleculeFile *)mol)->file_path();
    data[0] += ((MoleculeFile *)mol)->file_name();
  } else {
    data[0] = "";
  }
  return 1;
}

// return the filetype
static int molinfo_filetype(int, GString *data, int *)
{
  Molecule *mol = moleculeList -> molecule(molinfo_get_index);
  if (!strcmp(mol -> source, "File")) {
    // from the type, lookup the name
    data[0] = structureFileTypeNames[mol -> filetype];
  } else {
    data [0] = "";
  }
  return 1;
}

// return the full path+filename of the coordinate file
// (if there was one in addition to the topology file)
static int molinfo_filename2(int, GString *data, int *)
{
  Molecule *mol = moleculeList -> molecule(molinfo_get_index);
  if (!strcmp(mol -> source, "File") && mol -> source2) {
    data[0] = mol -> source2;
  } else {
    data[0] = "";
  }
  return 1;
}

// return the filetype of the secondary (coordinate) file
static int molinfo_filetype2(int, GString *data, int *)
{
  Molecule *mol = moleculeList -> molecule(molinfo_get_index);
  if (!strcmp(mol -> source, "File") && mol -> source2) {
    // from the type, lookup the name
    data[0] = CoorFileSuffix[mol -> filetype2];
  } else {
    data [0] = "";
  }
  return 1;
}


//// set/return the status information
#define generic_molinfo_setlist(fctnname, fctn1, fctn2) \
static int fctnname(int, int *data, int *)        \
{                                                 \
  if (data[0]) {fctn1;} else {fctn2;}             \
  return 1;                                       \
}
// active
generic_molinfo_data(molinfo_isactive,
  moleculeList->active(molinfo_get_index));
generic_molinfo_setlist(molinfo_setactive,
  moleculeList->activate(molinfo_get_index),
  moleculeList->inactivate(molinfo_get_index));

// drawn
generic_molinfo_data(molinfo_isdrawn,
  moleculeList->displayed(molinfo_get_index));
generic_molinfo_setlist(molinfo_setdrawn,
  moleculeList->show(molinfo_get_index),
  moleculeList->hide(molinfo_get_index));

// fixed
generic_molinfo_data(molinfo_isfixed,
  moleculeList->fixed(molinfo_get_index));
generic_molinfo_setlist(molinfo_setfixed,
  moleculeList->fix(molinfo_get_index),
  moleculeList->unfix(molinfo_get_index));

// top
generic_molinfo_data(molinfo_istop,
  moleculeList->is_top(molinfo_get_index));
generic_molinfo_setlist(molinfo_settop,
  moleculeList->make_top(molinfo_get_index),
  0);


// center {x y z}
static int molinfo_center(int, GString *data, int *)
{
  // compute the center location
  float pos[4] = {0,0,0,1}, offset[4];
  (moleculeList->molecule(molinfo_get_index)->centm).multpoint4d(
		pos, offset);
  char s[50];
  sprintf(s, "%f %f %f", -offset[0], -offset[1], -offset[2]);
  data[0] = s;
  return 1;
}
static int molinfo_setcenter(int, GString *data, int *)
{
  // check that there are three elements
  float x, y, z, w;
  int num;
  num = sscanf( (const char *) data[0], "%f %f %f %f", &x, &y, &z, &w);
  if (num != 3) {
    msgErr << "molinfo: set center: must have three position elements"
      << sendmsg;
  } else {
    moleculeList->molecule(molinfo_get_index)->change_center(x, y, z);
  }
  return 1;
}

// get/set the matrix values
#define get_matrix_macro(name, matrix)                                      \
static int name(int, GString *data, int *) {                                \
  char s[16*25];                                                            \
  Matrix4 *m = &(moleculeList->molecule(molinfo_get_index)->matrix);        \
  sprintf(s, "{%f %f %f %f} {%f %f %f %f} {%f %f %f %f} {%f %f %f %f}",     \
	  m->mat[0][0], m->mat[1][0], m->mat[2][0], m->mat[3][0],           \
	  m->mat[0][1], m->mat[1][1], m->mat[2][1], m->mat[3][1],           \
	  m->mat[0][2], m->mat[1][2], m->mat[2][2], m->mat[3][2],           \
	  m->mat[0][3], m->mat[1][3], m->mat[2][3], m->mat[3][3]);          \
  data[0] = s;                                                              \
  return 1;                                                                 \
}
#define set_matrix_macro(name, matrix)                                    \
static int name(int, GString *data, int *) {                              \
  /* This is a very loose way to get the 16 values (eg, ignores 17th) */  \
  Matrix4 tmp;                                                            \
  int num = sscanf( (const char *) data[0],                               \
    " { %f %f %f %f } { %f %f %f %f } { %f %f %f %f } { %f %f %f %f }",   \
    &tmp.mat[0][0], &tmp.mat[1][0], &tmp.mat[2][0], &tmp.mat[3][0],       \
    &tmp.mat[0][1], &tmp.mat[1][1], &tmp.mat[2][1], &tmp.mat[3][1],       \
    &tmp.mat[0][2], &tmp.mat[1][2], &tmp.mat[2][2], &tmp.mat[3][2],       \
    &tmp.mat[0][3], &tmp.mat[1][3], &tmp.mat[2][3], &tmp.mat[3][3]);      \
  if (num == 16) {                                                        \
    moleculeList->molecule(molinfo_get_index)->matrix(tmp);               \
  } else {                                                                \
    msgErr << "There should be 16 elements in the matrix" << sendmsg;     \
  }                                                                       \
  return 1;                                                               \
}

get_matrix_macro(molinfo_centermat, centm);
get_matrix_macro(molinfo_rotatemat, rotm);
get_matrix_macro(molinfo_scalemat , scalem);
get_matrix_macro(molinfo_globalmat, globm);
get_matrix_macro(molinfo_viewmat, tm);
set_matrix_macro(molinfo_set_centermat, set_cent_trans);
set_matrix_macro(molinfo_set_rotatemat, set_rot);
set_matrix_macro(molinfo_set_scalemat , set_scale);
set_matrix_macro(molinfo_set_globalmat, set_glob_trans);

// get the number of representations
generic_molinfo_data(molinfo_numreps, 
  moleculeList->molecule(molinfo_get_index)->components());

// get a specific rep. of a component
static int molinfo_rep(int, GString *data, int *)
{
  // first get the rep number; this is of the form " *rep +[0-9]+ *"
  // so it has to be good (though perhaps out of bounds)
  char *tmp = molinfo_get_string;
  while (*tmp++ == ' ');
  while (*tmp++ != ' ');
  while (*tmp++ == ' ');
  int repnum = atoi(tmp-1);

  if (repnum >= moleculeList->molecule(molinfo_get_index)->components()) {
    // some sort of error message, and return a ""
    data[0] = "Out of range";
    return 0;
  }
  data[0] = moleculeList->molecule(molinfo_get_index)->component(repnum)->
    atomRep -> cmdStr; // phew
  return 1;
}
// get a specific selection of a component
static int molinfo_selection(int, GString *data, int *)
{
  // this is of the form " *selection +[0-9]+ *"
  char *tmp = molinfo_get_string;
  while (*tmp++ == ' ');
  while (*tmp++ != ' ');
  while (*tmp++ == ' ');
  int repnum = atoi(tmp-1);
  //  msgInfo << "repnum is " << repnum << sendmsg;

  if (repnum >= moleculeList->molecule(molinfo_get_index)->components()) {
    // some sort of error message, and return a ""
    data[0] = "Out of range";
    return 0;
  }
  data[0] = moleculeList->molecule(molinfo_get_index)->component(repnum)->
    atomSel -> cmdStr; // phew
  return 1;
}
// get a specific color of a component
static int molinfo_color(int, GString *data, int *)
{
  // this is of the form " *colou?r +[0-9]+ *"
  char *tmp = molinfo_get_string;
  while (*tmp++ == ' ');
  while (*tmp++ != ' ');
  while (*tmp++ == ' ');
  int repnum = atoi(tmp-1);
  //  msgInfo << "repnum is " << repnum << sendmsg;

  if (repnum >= moleculeList->molecule(molinfo_get_index)->components()) {
    // some sort of error message, and return a ""
    data[0] = "Out of range";
    return 0;
  }
  data[0] = moleculeList->molecule(molinfo_get_index)->component(repnum)->
    atomColor -> cmdStr; // phew
  return 1;
}


/// ANIMATIONS
//get the number of animation frames
generic_molinfo_data(molinfo_numframes,
  moleculeList->molecule(molinfo_get_index)->num());

// get the current frame number
generic_molinfo_data(molinfo_frame, 
  moleculeList->molecule(molinfo_get_index)->frame());
// set the frame number
generic_molinfo_set_data(molinfo_set_frame,
  moleculeList->molecule(molinfo_get_index)->override_current_frame(data[0]));

//// simulation
// Is this a remote simulation?
static int molinfo_remote(int, int *data, int *)
{
  // there are two ways to do this.  The other is to check the remote list
  // and see if this molecule is on it
  if (!strcmp(moleculeList->molecule(molinfo_get_index)->source, "File")) {
    data[0] = 0;
  } else {
    data[0] = 1;
  }
  return 1;
}

//////// Get and set simulation information (energy and temperature)
#define generic_molinfo_simulation(name,term)                             \
static int name(int, double *data, int *)                                 \
{                                                                         \
  /* get the right time step */                                           \
  Timestep *ts = moleculeList->molecule(molinfo_get_index) ->             \
					item(molinfo_frame_number);       \
  if (!ts) {                                                              \
    data[0] = 0;                                                          \
    return 0;                                                             \
  }                                                                       \
  data[0] = ts -> energy[term];                                           \
  return 1;                                                               \
}
#define generic_molinfo_set_simulation(name,term)                         \
static int name(int, double *data, int *)                                 \
{                                                                         \
  /* get the right time step */                                           \
  Timestep *ts = moleculeList->molecule(molinfo_get_index) ->             \
					item(molinfo_frame_number);       \
  if (!ts) {                                                              \
    return 0;                                                             \
  }                                                                       \
  ts -> energy[term] = data[0];                                           \
  return 1;                                                               \
}

generic_molinfo_simulation(molinfo_bond, TSE_BOND);
generic_molinfo_simulation(molinfo_angle, TSE_ANGLE);
generic_molinfo_simulation(molinfo_dihedral, TSE_DIHE);
generic_molinfo_simulation(molinfo_improper, TSE_IMPR);
generic_molinfo_simulation(molinfo_vdw, TSE_VDW);
generic_molinfo_simulation(molinfo_electrostatic, TSE_COUL);
generic_molinfo_simulation(molinfo_hbond, TSE_HBOND);
generic_molinfo_simulation(molinfo_kinetic, TSE_KE);
generic_molinfo_simulation(molinfo_potential, TSE_PE);
generic_molinfo_simulation(molinfo_temperature, TSE_TEMP);
generic_molinfo_simulation(molinfo_energy, TSE_TOTAL);
generic_molinfo_simulation(molinfo_volume, TSE_VOLUME);
generic_molinfo_simulation(molinfo_pressure, TSE_PRESSURE);
generic_molinfo_simulation(molinfo_efield, TSE_EFIELD);

generic_molinfo_set_simulation(molinfo_set_bond, TSE_BOND);
generic_molinfo_set_simulation(molinfo_set_angle, TSE_ANGLE);
generic_molinfo_set_simulation(molinfo_set_dihedral, TSE_DIHE);
generic_molinfo_set_simulation(molinfo_set_improper, TSE_IMPR);
generic_molinfo_set_simulation(molinfo_set_vdw, TSE_VDW);
generic_molinfo_set_simulation(molinfo_set_electrostatic, TSE_COUL);
generic_molinfo_set_simulation(molinfo_set_hbond, TSE_HBOND);
generic_molinfo_set_simulation(molinfo_set_kinetic, TSE_KE);
generic_molinfo_set_simulation(molinfo_set_potential, TSE_PE);
generic_molinfo_set_simulation(molinfo_set_temperature, TSE_TEMP);
generic_molinfo_set_simulation(molinfo_set_energy, TSE_TOTAL);
generic_molinfo_set_simulation(molinfo_set_volume, TSE_VOLUME);
generic_molinfo_set_simulation(molinfo_set_pressure, TSE_PRESSURE);
generic_molinfo_set_simulation(molinfo_set_efield, TSE_EFIELD);


/////////////////////////////////////////////////
void molInfoTable_init(void)
{
  // the " *" are to allow spaces

  molInfoTable.add_keyword(" *index *", "index", molinfo_index, NULL);
  molInfoTable.add_keyword(" *id *", "id", molinfo_id, NULL);
  molInfoTable.add_keyword(" *numatoms *", "numatoms", molinfo_numatoms, NULL);
  molInfoTable.add_keyword(" *source *", "source", molinfo_source, NULL);
  molInfoTable.add_keyword(" *name *", "name", molinfo_name, NULL);
  molInfoTable.add_keyword(" *filename *", "filename", molinfo_filename, NULL);
  molInfoTable.add_keyword(" *filetype *", "filetype", molinfo_filetype, NULL);
  molInfoTable.add_keyword(" *filename2 *", "filename2", molinfo_filename2,
			   NULL);
  molInfoTable.add_keyword(" *filetype2 *", "filetype2", molinfo_filetype2,
			   NULL);

  // Status flags
  molInfoTable.add_keyword(" *active *", "active", molinfo_isactive, 
			   molinfo_setactive);
  molInfoTable.add_keyword(" *\\(drawn\\|displayed\\) *", "drawn", 
			   molinfo_isdrawn, molinfo_setdrawn);
  molInfoTable.add_keyword(" *fixed *", "fixed", molinfo_isfixed, 
			   molinfo_setfixed);
  molInfoTable.add_keyword(" *top *", "top", molinfo_istop, molinfo_settop);


  // matrix data
  molInfoTable.add_keyword(" *center *", "center", molinfo_center, 
			   molinfo_setcenter);

  molInfoTable.add_keyword(" *center_matrix *", "center_matrix",
			   molinfo_centermat, molinfo_set_centermat);
  molInfoTable.add_keyword(" *rotate_matrix *", "rotate_matrix",
			   molinfo_rotatemat, molinfo_set_rotatemat);
  molInfoTable.add_keyword(" *scale_matrix *", "scale_matrix",
			   molinfo_scalemat, molinfo_set_scalemat);
  molInfoTable.add_keyword(" *global_matrix *", "global_matrix",
			   molinfo_globalmat, molinfo_set_globalmat);
  molInfoTable.add_keyword(" *view_matrix *", "view_matrix",
			   molinfo_viewmat, NULL);

  // Representations
  molInfoTable.add_keyword(" *numreps *", "numreps", molinfo_numreps, NULL);
  molInfoTable.add_keyword(" *rep +[0-9]+ *", "rep", molinfo_rep, NULL);
  molInfoTable.add_keyword(" *selection +[0-9]+ *", "selection", 
			   molinfo_selection, NULL);
  molInfoTable.add_keyword(" *colou?r +[0-9]+ *", "color", 
			   molinfo_color, NULL);

  // Animation
  molInfoTable.add_keyword(" *numframes *", "numframes", 
			   molinfo_numframes, NULL);
  molInfoTable.add_keyword(" *frame *", "frame", molinfo_frame, 
			   molinfo_set_frame);


  // Simulation info
  molInfoTable.add_keyword(" *remote *", "remote", molinfo_remote, NULL);
  molInfoTable.add_keyword(" *bond *", "bond", molinfo_bond, 
			   molinfo_set_bond);
  molInfoTable.add_keyword(" *angle *", "angle", molinfo_angle, 
			   molinfo_set_angle);
  molInfoTable.add_keyword(" *dihedral *", "dihedral", 
			   molinfo_dihedral, molinfo_set_dihedral);
  molInfoTable.add_keyword(" *improper *", "improper", 
			   molinfo_improper, molinfo_set_improper);
  molInfoTable.add_keyword(" *vdw *", "vdw", molinfo_vdw, 
			   molinfo_set_vdw);
  molInfoTable.add_keyword(" *elec\\(trostatic\\)? *", "electrostatic", 
			   molinfo_electrostatic, molinfo_set_electrostatic);
  molInfoTable.add_keyword(" *hbond *", "hbond", molinfo_hbond, 
			   molinfo_set_hbond);
  molInfoTable.add_keyword(" *kinetic *", "kinetic", molinfo_kinetic, 
			   molinfo_set_kinetic);
  molInfoTable.add_keyword(" *potential *", "potential", molinfo_potential, 
			   molinfo_set_potential);
  molInfoTable.add_keyword(" *temp\\(erature\\)? *", "temperature", 
			   molinfo_temperature, molinfo_set_temperature);
  molInfoTable.add_keyword(" *energy *", "energy", molinfo_energy, 
			   molinfo_set_energy);
  molInfoTable.add_keyword(" *volume *", "volume", 
			   molinfo_volume, molinfo_set_volume);
  molInfoTable.add_keyword(" *pressure *", "pressure", 
			   molinfo_pressure, molinfo_set_pressure);
  molInfoTable.add_keyword(" *efield *", "efield", 
			   molinfo_efield, molinfo_set_efield);

  // crystallographic information
  molInfoTable.add_keyword(" *alpha *", "alpha", molinfo_alpha,
			   molinfo_set_alpha);
  molInfoTable.add_keyword(" *beta *", "beta", molinfo_beta, 
			   molinfo_set_beta);
  molInfoTable.add_keyword(" *gamma *", "gamma", molinfo_gamma, 
			   molinfo_set_gamma);
  molInfoTable.add_keyword(" *a *", "a", molinfo_a, molinfo_set_a);
  molInfoTable.add_keyword(" *b *", "b", molinfo_b, molinfo_set_b);
  molInfoTable.add_keyword(" *c *", "c", molinfo_c, molinfo_set_c);
}

// This trick causes the compiler to instantiate a MolInfoTable_helper,
// which then initializes the molInfoTable
class MolInfoTable_helper {
public:
  MolInfoTable_helper(void) {
    molInfoTable_init();
  }
};
MolInfoTable_helper molInfoTable_helper; // just to init the symboltable


// given the list of strings, return the data
int molinfo_get(int molid, int argc, char *argv[], Tcl_Interp *interp,
		int frame_num)
{
  // does the molecule exist?
  molinfo_get_id = molid;
  molinfo_get_index = moleculeList->mol_index_from_id(molid);

  if (molinfo_get_index == -1) {
    char s[10];
    sprintf(s, "%d", molid);
    Tcl_AppendResult(interp, "molinfo: set: no molecule exists with id ",
		     s, NULL);
    return TCL_ERROR;
  }
  // get the right frame number
  switch (frame_num) {
  case AtomSel::TS_NOW: 
    molinfo_frame_number = moleculeList->molecule(molinfo_get_index) ->
      frame();
    break;
  case AtomSel::TS_LAST:
    molinfo_frame_number = moleculeList->molecule(molinfo_get_index) ->
      num();
    break;
  default:      
    molinfo_frame_number = frame_num;
  }

  // now see if all the terms exist
  int *mapping = new int[argc], index;
  for (int i=0; i<argc; i++) {
    mapping[i] = -1;
  }
  int term;
  for(term=0; term<argc; term++) {
    // find the mapping from name to function index
    index = molInfoTable.find_attribute(argv[term]);
    if (index < 0) {
      Tcl_AppendResult(interp, "molinfo: cannot find molinfo attribute '",
		       argv[term], "'", NULL);
      delete [] mapping;
      return TCL_ERROR;
    }
    mapping[term] = index;
  }
  // every term on the options list exists, so get the data
  int fake_list = 1;
  GString *attribs_data = new GString;
  for (term=0; term<argc; term++) {
    molinfo_get_string = argv[term];
    molInfoTable.extract_keyword_info(mapping[term], 1, attribs_data, 
				      &fake_list);
    // append the data to the interpreter return
    Tcl_AppendElement(interp, (char *) (const char *) attribs_data[0]);
  }
  delete [] mapping;
  delete attribs_data;
  return TCL_OK;
}


// given the list of strings and data, set the right values
int molinfo_set(int molid, int argc, char *argv[], char *data[],
		Tcl_Interp *interp, int frame_num)
{
  // does the molecule exist?
  molinfo_get_id = molid;
  molinfo_get_index = moleculeList->mol_index_from_id(molid);

  if (molinfo_get_index == -1) {
    char s[10];
    sprintf(s, "%d", molid);
    Tcl_AppendResult(interp, "molinfo: set: no molecule exists with id ",
		     s, NULL);
    return TCL_ERROR;
  }

  // get the right frame number
  switch (frame_num) {
  case AtomSel::TS_NOW: 
    molinfo_frame_number = moleculeList->molecule(molinfo_get_index) ->
      frame();
    break;
  case AtomSel::TS_LAST:
    molinfo_frame_number = moleculeList->molecule(molinfo_get_index) ->
      num();
    break;
  default:      
    molinfo_frame_number = frame_num;
  }
  
  // now see if all the terms exist
  int *mapping = new int[argc], index;
  for (int i=0; i<argc; i++) {
    mapping[i] = -1;
  }
  int term;
  for(term=0; term<argc; term++) {
    // find the mapping from name to function index
    index = molInfoTable.find_attribute(argv[term]);
    if (index < 0) {
      Tcl_AppendResult(interp, "molinfo: cannot find molinfo attribute '",
		       argv[term], "'", NULL);
      delete [] mapping;
      return TCL_ERROR;
    }
    mapping[term] = index;
  }
  // every term on the options list exists, so set the data
  int fake_list = 1;
  GString *attribs_data = new GString;
  for (term = 0; term<argc; term++) {
    molinfo_get_string = argv[term];
    *attribs_data = data[term];
    molInfoTable.set_keyword_info(mapping[term], 1, attribs_data,
				  &fake_list);
  }
  delete [] mapping;
  delete attribs_data;
  return TCL_OK;
}

