/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeFilePDB.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1997/03/20 20:31:46 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Read in a PDB file, and tries to make the topology
 *
 ***************************************************************************/

#include "MoleculeFilePDB.h"
#include "ReadPDB.h"

MoleculeFilePDB::MoleculeFilePDB(char *filename, Scene *sc,
	 float cutoff) : MoleculeFile(filename, sc)
{
   real_filename = NULL;
   pdbCutoff = cutoff;
}
MoleculeFilePDB::MoleculeFilePDB(char *realfilename, char *filename,
				 Displayable *disp, float cutoff)
: MoleculeFile(filename, disp)
{
   real_filename = stringdup(realfilename);
   pdbCutoff = cutoff;
}

MoleculeFilePDB::MoleculeFilePDB(char *filename, Displayable *disp,
	float cutoff) : MoleculeFile(filename, disp)
{
   real_filename = NULL;
   pdbCutoff = cutoff;
}

MoleculeFilePDB::~MoleculeFilePDB(void) {
}

int MoleculeFilePDB::create(void){
   if (real_filename) {
      int i = createFromPDB(real_filename, pdbCutoff);
      delete [] real_filename;
      real_filename = NULL;
      if (!i) {
	 return FALSE;
      }
   } else if (!createFromPDB(strFile, pdbCutoff)) {
      return FALSE;
   }

   return(MoleculeFile::create());
}

// create the molecular structure based on the given PDB file.  This will
// 	read in the data for each atom, and then construct the connectivity
//	based on a distance search.
int MoleculeFilePDB::createFromPDB(char *pdbfile, float cutoff) {

  FILE *pdb;
  char pdbrec[PDB_RECORD_LENGTH + 1];
  int natoms, i, j, rectype;
  char names[8], rnames[8], segnames[8], ridstr[8];
  char chains[4];
  float newpos[ATOMCOORDS], newdata[ATOMEXTRA_DYNAMIC + ATOMEXTRA_STATIC];

  // open the file, if possible
  if(!(pdb = fopen(pdbfile,"r"))) {
    msgErr << "Cannot open file '" << pdbfile << "'." << sendmsg;
    return FALSE;
  }

  MSGDEBUG(2,"MoleculeFile: Creating structure from .pdb" << sendmsg);
  
  // read the file once to find the total number of atoms
  MSGDEBUG(3,"   Scanning pdb file once to find # of atoms ..." << sendmsg);
  natoms = 0;
  do {
    rectype = read_pdb_record(pdb, pdbrec);
    if(rectype == PDB_ATOM)
      natoms++;
  } while (rectype != PDB_END && rectype != PDB_EOF);
  rewind(pdb);
  MSGDEBUG(3,"   ==> " << natoms << " found in initial scan." << sendmsg);

  // now scan through again, extracting structural data
  MSGDEBUG(3,"   Scanning pdb file again for structural data ... " << sendmsg);
  init_atoms(natoms);
  do {
    rectype = read_pdb_record(pdb, pdbrec);
    switch (rectype) {
    case PDB_ATOM:
      get_pdb_fields(pdbrec, names, rnames, chains, segnames, ridstr, newpos,
      	newpos + 1, newpos + 2, newdata + ATOMOCCUP, newdata + ATOMBETA);
      newdata[ATOMCHARGE] = default_charge(names);
      newdata[ATOMMASS] = default_mass(names);
      newdata[ATOMRAD] = default_radius(names);
      add_atom(names, names, rnames, ridstr, chains, segnames, newpos, 
	       newdata);
      break;
    case PDB_CRYST1:
      get_pdb_cryst1(pdbrec, &alpha, &beta, &gamma,
		     &a_length, &b_length, &c_length);
      break;
    default:
      // ignore
      break;
    }
  } while (rectype != PDB_END && rectype != PDB_EOF);

  // done with PDB file ... close it
  fclose(pdb);

  MSGDEBUG(2,"Finished reading .pdb" << sendmsg);

  // determine bonds from position of atoms just read
  find_bonds_from_coordinates(NULL, cutoff);

  // append coordinates as first timestep
  Timestep *ts = new Timestep(natoms, 0.0);
  float *pos = ts->pos;
  float *extrad = ts->data;
  MolAtom *atm;
  for(i=0; i < natoms; i++) {
    atm = atom(i);
    for(j=0; j < ATOMCOORDS; j++)
      *(pos++) = atm->pos[j];
    for(j=0; j < ATOMEXTRA_DYNAMIC; j++)
      *(extrad++) = atm->extra[j];
  }
  append_frame(ts);

  return TRUE;
}


// macro used to get pointer to coordinates for next routine
#define ACOOR(ts,n)	(ts ? ts->pos + 3*n : atom(n)->pos)

// determine bonds from position of atoms previously read.
// If the specified Timestep is NULL, then the coordinates stored in the
// Atom records will be used.
// If cutoff < 0, use vdw radius to determine if bonded.
int MoleculeFilePDB::find_bonds_from_coordinates(Timestep *ts, float cutoff) {
  float cut, dist, rad1, rad2, *loc, *loc1, *loc2, dr[3];
  int i,j, k;
  float xmin, xmax, ymin, ymax, zmin, zmax;
  float boxsizex, boxsizey, boxsizez;
  int xb, yb, zb, xytotb, totb, axb, ayb, azb, aindex, bindex, starti;
  int xi, yi, zi, *inbox, *maxinbox, *tmpbox, *nbrbox, **boxatom;
  MolAtom *atom1, *atom2;

  if(!nAtoms || cutoff == 0.0)
    return 0;

  msgInfo << "   Determining bond structure from distance search ...";
  msgInfo << sendmsg;
  
  if(cutoff > 0.0)
    cut = cutoff * cutoff;	// compare to dist^2, to elim need for sqrt

  // find min/max bounds of molecule's coordinates
  for(i=0; i < nAtoms; i++) {
    loc = ACOOR(ts, i);
    if(i==0) {
      xmin = xmax = loc[0];
      ymin = ymax = loc[1];
      zmin = zmax = loc[2];
    } else {
      if(loc[0] < xmin)
        xmin = loc[0];
      else if(loc[0] > xmax)
        xmax = loc[0];
      if(loc[1] < ymin)
        ymin = loc[1];
      else if(loc[1] > ymax)
        ymax = loc[1];
      if(loc[2] < zmin)
        zmin = loc[2];
      else if(loc[2] > zmax)
        zmax = loc[2];
    }
  }
  
  // from size of molecule, break up space into boxes of side length=4 A
  // (or, if the molecule size is too large in any direction, the boxes are
  // increased so that a maximum of 50 boxes are used in any direction)
  boxsizex = (xmax - xmin >= 200.0 ? (xmax - xmin)/50.0 : 4.0);
  boxsizey = (ymax - ymin >= 200.0 ? (ymax - ymin)/50.0 : 4.0);
  boxsizez = (zmax - zmin >= 200.0 ? (zmax - zmin)/50.0 : 4.0);
  xb = (int)((xmax - xmin) / boxsizex) + 1;
  yb = (int)((ymax - ymin) / boxsizey) + 1;
  zb = (int)((zmax - zmin) / boxsizez) + 1;
  xytotb = yb * xb;
  totb = xytotb * zb;
  inbox = new int[totb];
  maxinbox = new int[totb];
  boxatom = new int *[totb];
  for(i=0; i < totb; i++) {
    maxinbox[i] = 0;
    inbox[i] = 0;
    boxatom[i] = NULL;
  }

  // for all the atoms, put them in their particular box
  for(i=0; i < nAtoms; i++) {
    loc1 = ACOOR(ts, i);
    axb = (int)((loc1[0] - xmin) / boxsizex);
    ayb = (int)((loc1[1] - ymin) / boxsizey);
    azb = (int)((loc1[2] - zmin) / boxsizez);
    aindex = azb * xytotb + ayb * xb + axb;
    if(boxatom[aindex] == NULL) {
      maxinbox[aindex] = 10;
      boxatom[aindex] = new int[10];
    } else if(inbox[aindex] == maxinbox[aindex]) {
      tmpbox = new int[2*inbox[aindex]];
      for(j=0; j < inbox[aindex]; j++)
        tmpbox[j] = boxatom[aindex][j];
      delete [] (boxatom[aindex]);
      boxatom[aindex] = tmpbox;
      maxinbox[aindex] *= 2;
    }
    boxatom[aindex][inbox[aindex]++] = i;
  }
  
  // now, for each box, check atoms in (positive) neighbor boxes for bonds
  aindex = 0;
  for(zi=0; zi < zb; zi++) {
    for(yi=0; yi < yb; yi++) {
      for(xi=0; xi < xb; xi++) {
        tmpbox = boxatom[aindex];
	for(j=0; j < 14; j++) {
	  bindex = (-1);
	  switch(j) {
	    case 0: bindex = aindex; break;				// same
	    case 1: if(xi < (xb - 1)) bindex = aindex + 1; break;	// X
	    case 2: if(yi < (yb - 1)) bindex = aindex + xb; break;	// Y
            case 3: if(zi < (zb - 1)) bindex = aindex + xytotb; break;	// Z		
	    case 4: if(xi < (xb - 1) && yi < (yb - 1))
	              bindex = aindex + xb + 1;
		    break;						// XY
	    case 5: if(xi < (xb - 1) && zi < (zb - 1))
	              bindex = aindex + xytotb + 1;
		    break;						// XZ
	    case 6: if(yi < (yb - 1) && zi < (zb - 1))
	              bindex = aindex + xytotb + xb;
		    break;						// YZ
	    case 7: if(xi < (xb - 1) && yi > 0)
	              bindex = aindex - xb + 1;
		    break;						// X-Y
	    case 8: if(xi > 0 && zi < (zb - 1))
	              bindex = aindex + xytotb - 1;
		    break;						// -XZ
	    case 9: if(yi > 0 && zi < (zb - 1))
	              bindex = aindex + xytotb - xb;
		    break;						// -YZ
	    case 10: if(xi < (xb - 1) && yi < (yb - 1) && zi < (zb - 1))
	              bindex = aindex + xytotb + xb + 1;
		    break;						// XYZ
	    case 11: if(xi > 0 && yi < (yb - 1) && zi < (zb - 1))
	              bindex = aindex + xytotb + xb - 1;
		    break;						// -XYZ
	    case 12: if(xi < (xb - 1) && yi > 0 && zi < (zb - 1))
	              bindex = aindex + xytotb - xb + 1;
		    break;						// X-YZ
	    case 13: if(xi > 0 && yi > 0 && zi < (zb - 1))
	              bindex = aindex + xytotb - xb - 1;
		    break;						//-X-YZ

	  }
	  if(bindex >= 0) {
	    nbrbox = boxatom[bindex];
            for(i=0; i < inbox[aindex]; i++) {
	      atom1 = atom(tmpbox[i]);
              loc1 = ACOOR(ts, tmpbox[i]);
              if(cutoff < 0.0)
	        rad1 = atom1->extra[ATOMRAD];
	      if(bindex == aindex)
	        starti = i + 1;
	      else
	        starti = 0;
	      for(k=starti; k < inbox[bindex]; k++) {
	        atom2 = atom(nbrbox[k]);
	        loc2 = ACOOR(ts, nbrbox[k]);
                if(cutoff < 0.0) {
	          rad2 = atom2->extra[ATOMRAD];
		  cut = (rad1 + rad2)*0.6;
		  cut *= cut;
                }
                dr[0] = loc2[0] - loc1[0];
                dr[1] = loc2[1] - loc1[1];
                dr[2] = loc2[2] - loc1[2];
                dist = (dr[0] * dr[0]) + (dr[1] * dr[1]) + (dr[2] * dr[2]);
                if(dist <= cut) {	// possible new bond
		  // do NOT bond H ... H
		  if(atom1->namestr[0] != 'H' || atom2->namestr[0] != 'H') {
		    // add a new bond
		    add_bond(tmpbox[i], nbrbox[k]);
		  }
                }
	      } // loop over k
	    } // loop over i
	  } // if (bindex >= 0)
        } // loop over j
	aindex++;
      } // xi
    } // yi
  } // zi
  
  // free up the storage space allocted for the grid search
  for(i=0; i < totb; i++)
    if(boxatom[i])  delete [] boxatom[i];
  delete [] boxatom;
  delete [] maxinbox;
  delete [] inbox;
  
  return nBonds;
}

