/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeFileRaster3D.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.10 $	$Date: 1997/03/13 17:38:56 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   read in from the Raster3D file format
 *
 ***************************************************************************/

////  The header contains a lot of information which I ignore
// TITLE (80 chars)
// NTX, NTY (number of tiles -- ignored)
// NPX, NPY (number of points per tile -- ignored)
// SCHEME (antialiasing scheme -- ignored)
// BKGND (background color -- ignored)
// SHADOW (T/F -- ignored)
// IPHONG (phong power -- ignored)
// STRAIT (secondary light source -- ignored)
// AMBIEN (ambient illumination -- ignored)
// SPECLR ignored
// EYEPOS ignored
// SOURCE ignored
// TMAT 4x4 matrix used to view the system
// object mode (1 == triangle, 2 == spheres, or 3 ==  mixed)
// INFMTS  --  FORTRAN input field specifications
//   for input format specifiers 1/6, 2, and 3/5
//   triangle(1) or plane(6) (x1,y1,z1)-(x2,y2,z2)-(x3,y3,z3) (r,g,b)
//   sphere(2)      (x,y,z) r (r,g,b)
//   cylinder(3,5)  (x1,y1,z1) R1 - (x2,y2,z2) R2(is ignored) (r,g,b)
//                      except if one radius is 0, a cone is made
//   I ignore these lines and just read them in as "%f"

#include <stdio.h>
#include <math.h>
#include "MoleculeFileRaster3D.h"
#include "DispCmds.h"
#include "Inform.h"
#include "ColorList.h"
#include "Global.h"


MoleculeFileRaster3D::MoleculeFileRaster3D(char *filename, Scene *sc)
: MoleculeFile(filename, sc)
{
}
MoleculeFileRaster3D::MoleculeFileRaster3D(char *filename, Displayable *disp)
: MoleculeFile(filename, disp)
{
}

MoleculeFileRaster3D::~MoleculeFileRaster3D(void)
{
}

// read (x,y,z)[0], (x,y,z)[1], (x,y,z)[2], (r,g,b)
static int get_triangle(FILE *infile, float *arr)
{
   if( fscanf(infile, "%f %f %f %f %f %f %f %f %f %f %f %f\n",
	      arr  , arr+1, arr+2,
	      arr+3, arr+4, arr+5,
	      arr+6, arr+7, arr+8,
	      arr+9, arr+10, arr+11) < 12) { 
      msgErr << "Could not read a triangle -- skipping" << sendmsg;
      return FALSE;
   }
   return TRUE;
}

// read (x,y,z)[0], (x,y,z)[1], (x,y,z)[2]
static int get_normals(FILE *infile, float *arr)
{
   if( fscanf(infile, "%f %f %f %f %f %f %f %f %f\n",
	      arr  , arr+1, arr+2,
	      arr+3, arr+4, arr+5,
	      arr+6, arr+7, arr+8 ) < 9) { 
      msgErr << "Could not read the normals -- skipping" << sendmsg;
      return FALSE;
   }
   return TRUE;
}

// read (x,y,z), R, (r,g,b)
static int get_sphere(FILE *infile, float *arr)
{
   if (fscanf(infile, "%f %f %f %f %f %f %f\n",
		 arr, arr+1, arr+2,
		 arr+3,
		 arr+4, arr+5, arr+6) < 7) {
      msgErr << "Could not read a sphere -- skipping" << sendmsg;
      return FALSE;
   }
   return TRUE;
}

// read (x,y,z)[0], R[0], (x,y,z)[1], R[1], (r,g,b)
static int get_cylinder(FILE *infile, float *arr)
{
   if (fscanf(infile, "%f %f %f %f %f %f %f %f %f %f %f\n",
		 arr  , arr+1, arr+2, arr+3,
		 arr+4, arr+5, arr+6, arr+7,
		 arr+8, arr+9, arr+10) < 10) {
      msgErr << "Could not read a cylinder -- skipping" << sendmsg;
      return FALSE;
   }
   return TRUE;
}

// read the file and put the info into the drawing list
int MoleculeFileRaster3D::create(void)
{
   FILE *infile = fopen(strFile, "r");
   if (!infile) {
      msgErr << "Cannot open Raster3D file " << strFile << sendmsg;
      return FALSE;
   }

   Matrix4 mat;
   char header[100];
   if (!fgets(header, 99, infile)) {
      fclose(infile);
      return FALSE;
   }
   msgInfo << header << sendmsg;  // tell the user info about the file

   int count = 11;    // ignore the next 11 lines of text
   char buffer[200];
   while (count--) {
      if (!fgets(buffer, 99, infile)) {
	 fclose(infile);
	 msgErr << "Could not read Raster3D file (" << strFile << ")";
	 msgErr << " header information." << sendmsg;
	 return FALSE;
      }
   }
   // Now I have to get the matrix.  This is made nasty since
   // there could be extra text after the first four numbers on a line
   // as in: 1 0 0 This is an extra comment
   int i;
   for (i=0; i<4; i++) {
      int j;
      fgets(buffer, 199, infile);  // read the whole line into a string
      if (j=sscanf(buffer, "%f %f %f %f",
		 &mat.mat[i][0], &mat.mat[i][1], &mat.mat[i][2],
		 &mat.mat[i][3])<4) {
	 msgErr << "Could not read all of " << strFile << " line ";
	 msgErr << i+1 << " of the Raster3D viewing matrix." << sendmsg;
	 fclose(infile);
	 return FALSE;
      }
   }
   int inputtype;  // get the input type (1, 2, or 3)
   fgets(buffer, 199, infile);
   if (sscanf(buffer, "%d", &inputtype) < 1) {
      msgErr << "Could not read the object mode line in " << strFile;
      msgErr << " Raster3D file." << sendmsg;
      fclose(infile);
      return FALSE;
   }
   if (inputtype < 1 || inputtype > 3) {
      msgErr << "Unknown mode type " << inputtype << " in ";
      msgErr << strFile << sendmsg;
      fclose(infile);
      return FALSE;
   }
   
   // do top-level creation; remember return code
   int createRetVal = MoleculeFile::create();
   if(!createRetVal) {
     // error doing top-level create ... abort
     fclose(infile);
     return FALSE;
   }

   // since we only need to do this once, create display list here
   reset_disp_list();

//   DispCmdMult multmat(&mat);
//   multmat.put( this);
   DispCmdColorIndex col;
   DispCmdMaterials materials;
   DispCmdTriangle tri;
   DispCmdSphere sph;
   DispCmdCylinder cyl;
   DispCmdCone cone;
   float data[15];
   float normals[15];
   count=0;
   // HACK NOT IN USE ANY MORE -- VESTIGILE CODE!
   // we changed to a 2 sided lighting model
   //   int triangle_count = 0; // this is a hack to try and
   //   // get the triagulation for the ribbons from molscript, etc. to
   //   // look right.  I alternate ribbons front and back

   materials.putdata(TRUE, this);
   // note, I also have to get the min/max values to find the 
   // cov and scale parameters
   if (inputtype == 1) { // all triangles -- not yet tested
      fgets(buffer, 199, infile); // ignore the input specifier
      while (!feof(infile)) {
	 if (get_triangle(infile, data) ) {
	    col.putdata(::colors -> nearest_index(
		sqrt(data[9]), sqrt(data[10]), sqrt(data[11])), this);
	       //	    if (!triangle_count) {
	       tri.putdata(data, data+3, data+6, this);
	       //	       triangle_count = 1;
	       //	    } else {
	       //	       tri.putdata(data, data+6, data+3, this);
	       //	       triangle_count = 0;
	       //	    }
	    if (count++ == 0) {
	       minx = maxx = data[0];
	       miny = maxy = data[1];
	       minz = maxz = data[2];
	    }
	    for (i=0; i<3; i++) {
	       if (data[3*i+0] < minx) minx = data[3*i+0];
	       if (data[3*i+1] < miny) miny = data[3*i+1];
	       if (data[3*i+2] < minz) minz = data[3*i+2];
	       if (data[3*i+0] > maxx) maxx = data[3*i+0];
	       if (data[3*i+1] > maxy) maxy = data[3*i+1];
	       if (data[3*i+2] > maxz) maxz = data[3*i+2];
	    }
	 }
      }
   } else if (inputtype == 2) { // all spheres -- not tested
      fgets(buffer, 199, infile); // ignore the input specifier
      while (!feof(infile)) {
	 if (get_sphere(infile, data) ) {
	    col.putdata(::colors -> nearest_index(
		sqrt(data[4]), sqrt(data[5]), sqrt(data[6])), this);
	    sph.putdata(data, data[3], this);
	    if (count++ == 0) {
	       minx = maxx = data[0];
	       miny = maxy = data[1];
	       minz = maxz = data[2];
	    }
	    for (i=0; i<1; i++) {
	       if (data[3*i+0] < minx) minx = data[3*i+0];
	       if (data[3*i+1] < miny) miny = data[3*i+1];
	       if (data[3*i+2] < minz) minz = data[3*i+2];
	       if (data[3*i+0] > maxx) maxx = data[3*i+0];
	       if (data[3*i+1] > maxy) maxy = data[3*i+1];
	       if (data[3*i+2] > maxz) maxz = data[3*i+2];
	    }
	 }
      }
   } else { // mixed
      char buffer2[200]; // used when looking for triangle normals
      int use_buffer2 = 0;
      fgets(buffer, 199, infile); // ignore the input specifiers
      fgets(buffer, 199, infile);
      fgets(buffer, 199, infile);
      while (!feof(infile)) {
	 int objtype = -1;
	 int error_check;   // retrieve from the lookahead buffer needed
	 if (use_buffer2) { // for the explicit normals
	   strcpy(buffer, buffer2);
	   use_buffer2 = 0;
	 } else {
	   if (!fgets(buffer, 199, infile)) {
	     continue;
	   }
	 }
	 // I do some checking in case things go whacko and a line
	 // was dropped, for whatever reasons.  This will attempt
	 // to get back to a line w/ only one item
	 if (sscanf(buffer, "%d %d", &objtype, &error_check)==1) {
	    switch(objtype) {
	     case 1:
	       if (get_triangle(infile, data)) {
		  int next_val;

		  buffer2[0] = 0;   // check for explicit normals
		  fgets(buffer2, 199, infile);
		  sscanf(buffer2, "%d", &next_val);
		  if (next_val != 7) {
		    use_buffer2 = 1;
		  } else {
		    use_buffer2 = 0;
		    get_normals(infile, normals);
		  }

		  col.putdata(::colors -> nearest_index(
			sqrt(data[9]), sqrt(data[10]), sqrt(data[11])), this);
  // flip every other triangle to see ribbons correctly
  // otherwise I get a front/ back pattern.  This is a hack.
		  //		  if (!triangle_count) {
		     if (use_buffer2) {
		        tri.putdata(data, data+3, data+6, this);
		     } else {
		        tri.putdata(data, data+3, data+6,
				    normals, normals+3, normals+6, this);
		     }
		     //		     triangle_count = 1;
		     //		  } else {
		     //if (use_buffer2) {
		     //   tri.putdata(data, data+6, data+3, this);
		     //} else {
		     //   tri.putdata(data, data+6, data+3,
		     //		    normals, normals+6, normals+3, this);
		     //}
		     //triangle_count = 0;
		     //}
		  if (count++ == 0) {
		     minx = maxx = data[0];
		     miny = maxy = data[1];
		     minz = maxz = data[2];
		  }
		  for (i=0; i<3; i++) {
		     if (data[3*i+0] < minx) minx = data[3*i+0];
		     if (data[3*i+1] < miny) miny = data[3*i+1];
		     if (data[3*i+2] < minz) minz = data[3*i+2];
		     if (data[3*i+0] > maxx) maxx = data[3*i+0];
		     if (data[3*i+1] > maxy) maxy = data[3*i+1];
		     if (data[3*i+2] > maxz) maxz = data[3*i+2];
		  }
	       }
	       break;
	     case 2: 
	       //	       triangle_count = 0;
	       if (get_sphere(infile, data) ) {  
		  col.putdata(::colors->nearest_index(
			sqrt(data[4]), sqrt(data[5]), sqrt(data[6])), this);
		  sph.putdata(data, data[3], this);
		  if (count++ == 0) {
		     minx = maxx = data[0];
		     miny = maxy = data[1];
		     minz = maxz = data[2];
		  }
		  for (i=0; i<1; i++) {
		     if (data[3*i+0] < minx) minx = data[3*i+0];
		     if (data[3*i+1] < miny) miny = data[3*i+1];
		     if (data[3*i+2] < minz) minz = data[3*i+2];
		     if (data[3*i+0] > maxx) maxx = data[3*i+0];
		     if (data[3*i+1] > maxy) maxy = data[3*i+1];
		     if (data[3*i+2] > maxz) maxz = data[3*i+2];
		  }
	       }
	       break;
	     case 3:  // these are the same so far
	     case 5:
	       //	       triangle_count = 0;
	       if (get_cylinder(infile, data)) {
		  col.putdata(::colors->nearest_index(
			sqrt(data[8]), sqrt(data[9]), sqrt(data[10])), this);
		  if (data[3] == 0.0) {
		    cone.putdata(data + 4, data, data[7], 8, this);
		  } else if (data[7] == 0.0) {
		    cone.putdata(data, data + 4, data[3], 8, this);
		  } else {
		    cyl.putdata(data, data+4, data[3], 8, this);
		  }
		  // spheres removed for speed
//		  sph.putdata(data, data[3], this);
//		  sph.putdata(data+4, data[3], this);
		  if (count++ == 0) {
		     minx = maxx = data[0];
		     miny = maxy = data[1];
		     minz = maxz = data[2];
		  }
		  for (i=0; i<2; i++) {
		     if (data[3*i+0+i] < minx) minx = data[3*i+0+i];
		     if (data[3*i+1+i] < miny) miny = data[3*i+1+i];
		     if (data[3*i+2+i] < minz) minz = data[3*i+2+i];
		     if (data[3*i+0+i] > maxx) maxx = data[3*i+0+i];
		     if (data[3*i+1+i] > maxy) maxy = data[3*i+1+i];
		     if (data[3*i+2+i] > maxz) maxz = data[3*i+2+i];
		  }
	       }
	       break;
	     default:{
		 //	       triangle_count = 0;
		msgErr << "I don't (yet?) understand object type ";
		msgErr << objtype << " -- skipping it" << sendmsg;
		msgErr << "skipped: " << buffer << sendmsg;
		fgets(buffer, 199, infile);
		msgErr << "skipped: " << buffer << sendmsg;
		break;
	     }
	    } // end of switch
	 }  else { // end if "got an input"
	    msgErr << "Expected one element on the line, got: "<< sendmsg;
	    msgErr << " " << buffer << sendmsg;
	 }
      } // end of while
   } // end of type "mixed objects"
   materials.putdata(FALSE, this);
   fclose(infile);
   return createRetVal;
} // end of function


// center of volume
void MoleculeFileRaster3D::cov(float &x, float &y, float&z) {
   x = (minx + maxx)/2;
   y = (miny + maxy)/2;
   z = (minz + maxz)/2;
}

// get the size
float MoleculeFileRaster3D::scale_factor(void) {
   float wx = maxx - minx;
   float wy = maxy - miny;
   float wz = maxz - minz;
   if (wx > wy) {
      if (wx > wz) {
	 return 2.0/wx;  // the 2.0 is to fit the system in
      } else {           // a box from -1 to 1
	 return 2.0/wz;
      }
   } else {
      if (wy > wz) {
	 return 2.0/wy;
      } else {
	 return 2.0/wz;
      }
   }
}

