/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeList.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.24 $	$Date: 1996/02/21 16:43:11 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The MoleculeList class, which is a list of the molecules being displayed.
 * This is a Displayable object, where each molecule is a child Displayable.
 *
 ***************************************************************************/

#include <ctype.h>
#include "MoleculeList.h"
#include "ColorList.h"
#include "Inform.h"
#include "MolAction.h"
#include "Scene.h"


// static data for initialization of colors
#define AtomColors_num		7
static char *AtomColors_names[AtomColors_num] = 
  { "H", "O", "N", "C", "S", "P", "Z" };
static int AtomColors_colors[AtomColors_num] = 
  { REGWHITE, REGRED, REGBLUE, REGCYAN, REGYELLOW, REGTAN, REGSILVER };

#define ResNameColors_num	34
static char *ResNameColors_names[ResNameColors_num] = 
  { "ALA", "ARG", "ASN", "ASP", "CYS", "GLY", "GLU", "GLN", "HIS", "ILE",
    "LEU", "LYS", "MET", "PHE", "PRO", "SER", "THR", "TRP", "TYR", "VAL",
    "ADE", "CYT", "GUA", "THY", "URA", "TIP", "TIP3", "WAT", "SOL", "H2O",
    "LYR", "ZN",  "NA",  "CL" };
static int ResNameColors_colors[ResNameColors_num] = 
  { REGBLUE, REGWHITE, REGTAN, REGRED, REGYELLOW,
    REGWHITE, REGPINK, REGORANGE, REGCYAN, REGGREEN,
    REGPINK, REGCYAN, REGYELLOW, REGPURPLE, REGOCHRE,
    REGYELLOW, REGMAUVRE, REGSILVER, REGGREEN, REGTAN,
    REGBLUE, REGORANGE, REGYELLOW, REGPURPLE, REGGREEN,
    REGCYAN, REGCYAN, REGCYAN, REGCYAN, REGCYAN,
    REGPURPLE, REGSILVER, REGYELLOW, REGGREEN };

// mapping of resname --> restype
static int ResTypeMapping[ResNameColors_num] =
  { 6, 3, 5, 4, 6, 6, 4, 5, 3, 6,
    6, 3, 6, 6, 6, 5, 5, 6, 5, 6,
    2, 2, 2, 2, 2, 1, 1, 1, 1, 1,
    6, 7, 7, 7 };

#define ResTypeColors_num	8
static char *ResTypeColors_names[ResTypeColors_num] = 
  { "Unassigned", "Solvent", "Nucleic_Acid", "Basic", "Acidic",
    "Polar", "Nonpolar", "Ion" };

static int ResTypeColors_colors[ResTypeColors_num] =
  { REGCYAN,	REGYELLOW,	REGPURPLE,	REGBLUE,	REGRED,
    REGGREEN,	REGWHITE,	REGTAN };

///////////////////////  constructor  
MoleculeList::MoleculeList(Scene *scn) 
	: Displayable3D(MULT, "Molecule List", scn, 4), molList(8) {

  topMol = NULL;
  needTopMolReset = TRUE;

  currAtomRep = new AtomRep();
  currAtomSel = new AtomSel(this);
}

///////////////////////  destructor  
MoleculeList::~MoleculeList(void) {
  // delete all added molecules
  for(int i=0; i < num(); i++)
    delete molecule(i);

  delete currAtomColor;
  delete currAtomRep;
  delete currAtomSel;
}

///////////////////////  protected virtual routines

// do action when a new color list is provided
void MoleculeList::do_use_colors(void) {
  int i;

  // create atom color object
  currAtomColor = new AtomColor(this, colorList);
  
  // create new color name categories
  colorCatIndex[MLCAT_NAMES] = colorList->add_color_category("Name");
  colorCatIndex[MLCAT_TYPES] = colorList->add_color_category("Type");
  colorCatIndex[MLCAT_RESNAMES] = colorList->add_color_category("Resname");
  colorCatIndex[MLCAT_RESTYPES] = colorList->add_color_category("Restype");
  colorCatIndex[MLCAT_CHAINS] = colorList->add_color_category("Chain");
  colorCatIndex[MLCAT_SEGNAMES] = colorList->add_color_category("Segname");
  colorCatIndex[MLCAT_MOLECULES] = colorList->add_color_category("Molecule");
  colorCatIndex[MLCAT_SPECIAL] = colorList->add_color_category("Highlight");
  colorCatIndex[MLCAT_SSTRUCT] = colorList->add_color_category("Structure");
  
  // add initial definitions for some of these categories
  
  // special items
  (colorList->color_category(colorCatIndex[MLCAT_SPECIAL]))->add_name(
  	"Proback", REGGREEN);
  (colorList->color_category(colorCatIndex[MLCAT_SPECIAL]))->add_name(
  	"Nucback", REGYELLOW);
  (colorList->color_category(colorCatIndex[MLCAT_SPECIAL]))->add_name(
  	"Nonback", REGBLUE);

  // secondary structure classifications
  (colorList->color_category(colorCatIndex[MLCAT_SSTRUCT]))->add_name(
  	"Alpha Helix", REGPURPLE);
  (colorList->color_category(colorCatIndex[MLCAT_SSTRUCT]))->add_name(
  	"3_10_Helix", REGMAUVRE);
  (colorList->color_category(colorCatIndex[MLCAT_SSTRUCT]))->add_name(
  	"Pi_Helix", REGRED);
  (colorList->color_category(colorCatIndex[MLCAT_SSTRUCT]))->add_name(
  	"Extended_Beta", REGYELLOW);
  (colorList->color_category(colorCatIndex[MLCAT_SSTRUCT]))->add_name(
  	"Bridge_Beta", REGTAN);
  (colorList->color_category(colorCatIndex[MLCAT_SSTRUCT]))->add_name(
  	"Turn", REGCYAN);
  (colorList->color_category(colorCatIndex[MLCAT_SSTRUCT]))->add_name(
  	"Coil", REGWHITE);


  // atom names and types
  for(i=0; i < AtomColors_num; i++) {
    (colorList->color_category(colorCatIndex[MLCAT_NAMES]))->add_name(
    	AtomColors_names[i], AtomColors_colors[i]);
    (colorList->color_category(colorCatIndex[MLCAT_TYPES]))->add_name(
    	AtomColors_names[i], AtomColors_colors[i]);
  }
  
  // residue types
  for(i=0; i < ResTypeColors_num; i++)
    (colorList->color_category(colorCatIndex[MLCAT_RESTYPES]))->add_name(
    	ResTypeColors_names[i], ResTypeColors_colors[i]);

  // residue names
  for(i=0; i < ResNameColors_num; i++) {
    (colorList->color_category(colorCatIndex[MLCAT_RESNAMES]))->add_name(
    	ResNameColors_names[i], ResNameColors_colors[i]);

    // define which residues are in the different residue type categories
    int rtcode = (colorList->color_category(colorCatIndex[MLCAT_RESTYPES])) ->
    			typecode(ResTypeColors_names[ResTypeMapping[i]]);
    resTypes.add_name(ResNameColors_names[i], rtcode);
  }
}

///////////////////////  public routines  

// put new names from given molecule into color lists
void MoleculeList::add_color_names(int molIndx) {
  int i, indx, catIndx;
  char newname[2];
  NameList<int> *nlist, *molnames;
  Molecule *m;
  
  strcpy(newname, " ");

  // make sure this molecule is in the list already, otherwise just return
  if(!(m = molecule(molIndx))) {
    MSGDEBUG(1, "Attempt to add color names to a molecule not in the ");
    MSGDEBUG(1, "molecule list." << sendmsg);
    return;
  }
  
  MSGDEBUG(1, "Adding color name information to new molecule " << molIndx);
  MSGDEBUG(1, " (name=" << m->name << ")" << sendmsg);

  // tell the molecule to use our colorlist
  m->use_colors(colorList);

  // for the given molecule, go through the NameList objects, and add them
  // to the color categories.  For new names, the color to use is the
  // color for the new index, mod # of total colors
  
  // atom names
  molnames = &(m->atomNames);
  catIndx = MLCAT_NAMES;
  nlist = (colorList->color_categories()).data(colorCatIndex[catIndx]);
  for(i=0; i < molnames->num(); i++) {
    // get first char of name, and convert to upper case
    newname[0] = (char)toupper(*(molnames->name(i)));
    
    // add this single-char name to color list; if it exists, get color
    indx = nlist->add_name(newname, nlist->num() % VISCLRS);
    
    // for the molecule, set the color for this name
    molnames->set_data(i, indx);
  }
  
  // atom types
  molnames = &(m->atomTypes);
  catIndx = MLCAT_TYPES;
  nlist = (colorList->color_categories()).data(colorCatIndex[catIndx]);
  for(i=0; i < molnames->num(); i++) {
    // get first char of name, and convert to upper case
    newname[0] = (char)toupper(*(molnames->name(i)));
    
    // add this single-char name to color list; if it exists, get color
    indx = nlist->add_name(newname, nlist->num() % VISCLRS);
    
    // for the molecule, set the color for this name
    molnames->set_data(i, indx);
  }
  
  // residue names and types ... use full name
  molnames = &(m->resNames);
  catIndx = MLCAT_RESNAMES;
  nlist = (colorList->color_categories()).data(colorCatIndex[catIndx]);
  for(i=0; i < molnames->num(); i++) {
    indx = nlist->add_name(molnames->name(i), nlist->num() % VISCLRS);
    molnames->set_data(i, indx);
    
    // check for residue types
    if(resTypes.typecode(molnames->name(i)) < 0) {
      // the residue has not been added to the restype list yet
      int rtcode = (colorList->color_category(colorCatIndex[MLCAT_RESTYPES])) ->
    			typecode(ResTypeColors_names[ResTypeMapping[0]]);
      resTypes.add_name(molnames->name(i), rtcode);
    }
  }

  // chain names ... use full name
  molnames = &(m->chainNames);
  catIndx = MLCAT_CHAINS;
  nlist = (colorList->color_categories()).data(colorCatIndex[catIndx]);
  for(i=0; i < molnames->num(); i++) {
    indx = nlist->add_name(molnames->name(i), nlist->num() % VISCLRS);
    molnames->set_data(i, indx);
  }

  // segment names ... use full name
  molnames = &(m->segNames);
  catIndx = MLCAT_SEGNAMES;
  nlist = (colorList->color_categories()).data(colorCatIndex[catIndx]);
  for(i=0; i < molnames->num(); i++) {
    indx = nlist->add_name(molnames->name(i), nlist->num() % VISCLRS);
    molnames->set_data(i, indx);
  }

  // molecule name ... use full name
  nlist = (colorList->color_categories()).data(colorCatIndex[MLCAT_MOLECULES]);
  nlist->add_name(m->name, nlist->num() % VISCLRS);
  
  nlist = (colorList->color_categories()).data(colorCatIndex[catIndx]);
  for(i=0; i < molnames->num(); i++) {
    indx = nlist->add_name(molnames->name(i), nlist->num() % VISCLRS);
    molnames->set_data(i, indx);
  }

  // now, for the given molecule, add a new representation using the
  // default settings for the rep objects
  AtomColor *ac = new AtomColor(this, colorList);
  AtomRep *ar = new AtomRep();
  AtomSel *as = new AtomSel(this);
  add_rep(molIndx, ac, ar, as);
  delete as;
  delete ar;
  delete ac;
}


// add a new molecule; return it's position in molList, or (-1) if error
int MoleculeList::add_molecule(Molecule *newmol) {
  int retval;

  // make sure it has been added as a child as well
  if(child_index(newmol) < 0) {
    msgErr << "Molecule '" << newmol->name << "' has not been created ";
    msgErr << "properly." << sendmsg;
    return (-1);
  }

  // add the molecule to our list of molecules
  retval = molList.append(newmol);

  // set the color data properly, and add the first atom representation
  add_color_names(retval);
	
  // finally, make this the top molecule
  make_top(retval);
  // don't do this for Graphics molecules
  if (strcmp(newmol->source, "Graphics"))
    needTopMolReset = TRUE;		// only set here, so a reset is only
  					// done when a new molecule is added
  
  return retval;
}


// set the top molecule ...
// make sure the given molecule is in the list; if not, do not do anything
void MoleculeList::make_top(Molecule *m) {
   if (!m) {
      topMol = m;
   } else if(m && mol_index_from_id(m->id()) >= 0) {
      topMol = m;
   }
}


// remove the Nth molecule from the list, i.e. delete it.  Return success.
int MoleculeList::del_molecule(int n) {
  Molecule *m, *newtopmol = NULL;

  // for this particular case, must make sure index is correct
  if(!(m = molecule(n)))
    return FALSE;

  MSGDEBUG(2,"MoleculeList: Deleting molecule " << n << " ..." << sendmsg);

  // must change the top molecule, if necessary
  if(is_top(n)) {
    MSGDEBUG(2,"  Molecule " << n << " is the top molecule.  Changing to: ");
    if(n+1 < num()) {		// is there a molecule following this one?
      MSGDEBUG(2," mol " << n+1);
      newtopmol = molecule(n+1);	// get it
    } else if(n-1 >= 0)	 {	// is there a molecule before this one?
      MSGDEBUG(2," mol " << n-1);
      newtopmol = molecule(n-1);	// get it
    } else {
      MSGDEBUG(2," NO top mol.");
      newtopmol = topMol;		// signal there are NO molecules now
    }
    MSGDEBUG(2,sendmsg);
  }
  
  // delete the nth molecule
  delete m;
  molList.remove(n);
  
  // now, change the top molecule if necessary
  if(newtopmol != NULL) {
    if(newtopmol != topMol) {
      MSGDEBUG(2,"MoleculeList: Changing top mol after delete." << sendmsg);
      make_top(newtopmol);
    } else {
      MSGDEBUG(2,"MoleculeList: Now NO top mol after delete." << sendmsg);
      make_top((Molecule *)NULL);
    }
  }

  return TRUE;
}


// set current atom coloring method
int MoleculeList::set_color(char *s) {
  return (currAtomColor->change(s));
}


// set current atom representation method
int MoleculeList::set_representation(char *s) {
  return (currAtomRep->change(s));
}


// set current atom selection method
int MoleculeList::set_selection(char *s) {
  return (currAtomSel->change(s));
}


// add a new graphics representation to the specified molecule.
// uses the specified coloring, representation, and selection settings.
// if n < 0, do so for all active molecules.
int MoleculeList::add_rep(int n, AtomColor *ac, AtomRep *ar, AtomSel *as) {

  if(n >= 0 && n < num()) {
    return (molecule(n)->add_rep(ac, ar, as));

  } else if(n < 0) {
    int retval = TRUE;
    Molecule *mol;
    for(int i=0; i < num(); i++) {
      mol = molecule(i);
      if(mol->active)
        retval = retval && mol->add_rep(ac, ar, as);
    }
    return retval;

  } else
    return FALSE;
}


// change the graphics representation m, for the specified molecule n, to
// the new settings.  Return success.
int MoleculeList::change_rep(int m, int n, AtomColor *ac, AtomRep *ar,
	AtomSel *as) {
  Molecule *mol = molecule(n);
  return (mol != NULL && mol->change_rep(m, ac, ar, as));
}

// change just the coloring method for the mth rep in the nth molecule.
int MoleculeList::change_repcolor(int m, int n, char *s) {
  Molecule *mol = molecule(n);
  if(mol) {
    AtomColor ac(*currAtomColor);
    if(ac.change(s))
      return (mol->change_rep(m, &ac, NULL, NULL));
  }
  return FALSE;
}
  
// change just the representation for the mth rep in the nth molecule.
int MoleculeList::change_repmethod(int m, int n, char *s) {
  Molecule *mol = molecule(n);
  if(mol) {
    AtomRep ar(*currAtomRep);
    if(ar.change(s))
      return (mol->change_rep(m, NULL, &ar, NULL));
  }
  return FALSE;
}
  
// change just the selection for the mth rep in the nth molecule.
int MoleculeList::change_repsel(int m, int n, char *s) {
  Molecule *mol = molecule(n);
  if(mol) {
    AtomSel as(*currAtomSel);
    // this is a _very_ cumbersome and silly way to do things.
    //  By the time this gets to the end it has parsed the system TWICE
    // Since the first time 'mol' doesn't exist, the time isn't too bad.
    // However, it is a bit annoying.  Alas, this will have to wait
    // until the rewrite of BaseMolecule
    if(as.change(s)!=AtomSel::NO_PARSE) {
       return (mol->change_rep(m, NULL, NULL, &as));
    }
  }
  return FALSE;
}


// delete a graphics representation m, for the specified molecule n.
// return success.
int MoleculeList::del_rep(int m, int n) {
  Molecule *mol = molecule(n);
  return (mol != NULL && mol->del_rep(m));
}



////////////// functions that affect all molecules

// apply an action to the list of active molecules
void MoleculeList::act(MolAction& action, ActionTarget whichMol) {
  int i;
  Molecule *mol;
  
  if(whichMol == ALL) {
    for(i=0; i < num(); i++)
      action.act(molecule(i));
  } else if(whichMol == TOP) {
    if(top())
      action.act(top());
  } else if(whichMol == ACTIVE) {
    for(i=0; i < num(); i++) {
      mol = molecule(i);
      if(active(mol))
        action.act(mol);
    }
  } else if(whichMol == DISPLAYED) {
    for(i=0; i < num(); i++) {
      mol = molecule(i);
      if(displayed(mol))
        action.act(mol);
    }
  }
}


// apply an actino to the Nth molecule
void MoleculeList::act(MolAction& action, int n) {
  Molecule *mol = molecule(n);
  if(mol)
    action.act(mol);
}


//////////////////  public virtual routines  

// reset to identity the tranformation ... sets proper scaling and centering
// values for all the molecules, based on which molecule is the top molecule.
// If no molecule is the top molecule, just calls the original version
void MoleculeList::reset_transformation(void) {
  float x, y, z;
  // reset all to identity first
  Displayable3D::reset_transformation();
  
  // set centering and scaling to make the top molecule centered at it's
  // proper centering location, and scaled to fit in (-1 ... 1) box
  if(topMol) {
    mult_scale(topMol->scale_factor());
    topMol->cov(x, y, z);
    add_cent_trans(-x, -y, -z);
  }
}


// prepare for drawing
void MoleculeList::prepare(DisplayDevice *) {

  // checks to see if the top molecule has recently had it's first frame
  // added, so that the scaling, etc must be redone.
  if(needTopMolReset && topMol && (topMol->num() > 0 || topMol->nAtoms == 0)) {
    reset_transformation();
    needTopMolReset = FALSE;
  }  
}

// For the given Pickable, determine if it is a molecule or a representation
// of a molecule, and return the proper pointer if it is (NULL otherwise)
Molecule *MoleculeList::check_pickable(Pickable *pobj) {
  register int i,j, mnum, repnum;

  // search through each molecule in the MoleculeList ...
  mnum = num();
  for(i=0; i < mnum; i++) {

    // and check each molecule's representation to see if it matches
    Molecule *mol = molecule(i);
    if(pobj == mol)
      return mol;

    repnum = mol->components();
    for(j=0; j < repnum; j++)
      if(pobj == mol->component(j))
	return mol;
  }

  // if here, nothing found
  return NULL;
}


