/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeList.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.17 $	$Date: 1996/02/05 20:50:57 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The MoleculeList class, which is a list of the molecules being displayed.
 * This is a Displayable object, where each molecule is a child Displayable.
 *
 ***************************************************************************/
#ifndef MOLECULELIST_H
#define MOLECULELIST_H

#include "Displayable.h"
#include "Molecule.h"
#include "AtomColor.h"
#include "AtomRep.h"
#include "AtomSel.h"
#include "ResizeArray.h"
#include "utilities.h"
class MolAction;

// number of color categories, and where they're found in the table
enum { MLCAT_NAMES, MLCAT_TYPES, MLCAT_RESNAMES, MLCAT_RESTYPES,
       MLCAT_CHAINS, MLCAT_SEGNAMES, MLCAT_MOLECULES,
       MLCAT_SPECIAL, MLCAT_SSTRUCT, NUM_MLCAT};


class MoleculeList : public Displayable3D {

public:
  // enum for how actions are done: to ALL molecules, all ACTIVE, all
  // DISPLAYED, TOP, or NONE
  enum ActionTarget { ALL, TOP, ACTIVE, DISPLAYED, NONE };

private:
  // the 'top' molecule, which determines what the centering and scaling of
  // the molecules should be
  Molecule *topMol;
  
  // molecules in this list.
  ResizeArray<Molecule *> molList;

  // do we need to reset the transformation if the topMol has any frames?
  // Used only when only one molecule is loaded, and does not have any
  // frames yet.
  int needTopMolReset;

  // current atom selection, representation, and coloring methods
  AtomColor *currAtomColor;
  AtomRep *currAtomRep;
  AtomSel *currAtomSel;

  // set the given Molecule as the top molecule.
  void set_top_molecule(Molecule *);

protected:
  // do action when a new color list is provided
  virtual void do_use_colors(void);

public:
  // mapping of residue names <--> residue types (hydrophobic, neutral, etc.)
  // residue names are the 'name' key, and the index into the residue type
  // color list is the 'data' value.
  NameList<int> resTypes;

  // color category indices
  int colorCatIndex[NUM_MLCAT];

  // put new names from given molecule into color lists
  void add_color_names(int);

public:
  // constructor and destructor
  MoleculeList(Scene *);
  virtual ~MoleculeList(void);

  // return the number of molecules in this list
  int num(void) { return molList.num(); }

  // return the Nth molecule (index runs 0 ... (count-1))
  Molecule *molecule(int n) {
    Molecule *retval = NULL;
    if(n >= 0 && n < num())
      retval = molList[n];
    return retval;
  }
  
  // return the index of the molecule with given ID (-1 if error)
  int mol_index_from_id(int id) {
    for(int i=0; i < num(); i++) {
      if(id == (molList[i])->id())
        return i;
    }
    return (-1);
  }

  // add a new molecule; return it's position in molList, or (-1) if error
  int add_molecule(Molecule *);

  // remove the Nth molecule from the list, i.e. delete it.  Return success.
  int del_molecule(int);

  //
  // routines to get/set characteristics for new graphics representations
  //
  
  // get/set current atom coloring method
  int set_color(char *);
  char *color(void) { return currAtomColor->cmdStr; }
  
  // get/set current atom representation method
  int set_representation(char *);
  char *representation(void) { return currAtomRep->cmdStr; }
  
  // get/set current atom selection command
  int set_selection(char *);
  char *selection(void) { return currAtomSel->cmdStr; }
  
  // add a new graphics representation to the specified molecule.
  // uses the specified coloring, representation, and selection settings.
  // if n < 0, do so for all active molecules.
  int add_rep(int, AtomColor *, AtomRep *, AtomSel *);
  
  // add a new graphics representation to the specified molecule.
  // uses the 'current' coloring, representation, and selection settings.
  // if n < 0, do so for all active molecules.
  int add_rep(int n) {
    return add_rep(n, currAtomColor, currAtomRep, currAtomSel);
  }
  
  // change the graphics representation m, for the specified molecule n, to
  // the new settings.  Return success.
  int change_rep(int m, int n, AtomColor *, AtomRep *, AtomSel *);

  // change the graphics representation m, for the specified molecule n.
  // uses the 'current' coloring, representation, and selection settings.
  int change_rep(int m, int n) {
    return change_rep(m, n, currAtomColor, currAtomRep, currAtomSel);
  }

  // change just the coloring method for the mth rep in the nth molecule.
  int change_repcolor(int m, int n, char *);
  
  // change just the representation for the mth rep in the nth molecule.
  int change_repmethod(int m, int n, char *);
  
  // change just the selection for the mth rep in the nth molecule.
  int change_repsel(int m, int n, char *);

  // delete a graphics representation m, for the specified molecule n.
  // return success.
  int del_rep(int m, int n);

  //
  //
  // routines to get/set top, active, displayed, fixed molecules
  //
  
  // query or set the top molecule
  Molecule *top(void) { return topMol; }
  int is_top(int n) { return (molecule(n) == topMol); }
  int is_top(Molecule *m) { return topMol == m; }
  void make_top(int n) { make_top(molecule(n)); }
  void make_top(Molecule *m);

  // query/set active status of Nth molecule
  int active(int n) { return molecule(n)->active; }
  int active(Molecule *m) { return (m && m->active); }
  void activate(int n) { molecule(n)->active = TRUE; }
  void inactivate(int n) { molecule(n)->active = FALSE; }
  
  // query/set displayed status of Nth molecule
  int displayed(int n) { return molecule(n)->displayed(); }
  int displayed(Molecule *m) { return (m && m->displayed()); }
  void show(int n) { molecule(n)->on(); }
  void hide(int n) { molecule(n)->off(); }
  
  // query/set fixed status of Nth molecule
  int fixed(int n) { return molecule(n)->fixed(); }
  int fixed(Molecule *m) { return (m && m->fixed()); }
  void fix(int n) { molecule(n)->fix(); }
  void unfix(int n) { molecule(n)->unfix(); }

  //
  // functions that affect all molecules
  //

  // apply an action to the list of active molecules
  void act(MolAction&, ActionTarget = ACTIVE);

  // apply an actino to the Nth molecule
  void act(MolAction&, int);

  //
  // Displayable virtual functions
  //

  // reset to identity the tranformation ... virtual so resets can affect
  // other factors as well.
  // This version calls the parent version, but also sets all molecule's
  // tranformations to center and scale properly based on the top molecule.
  virtual void reset_transformation(void);

  // prepare for drawing
  virtual void prepare(DisplayDevice *);


  // given a pickable, find it and return the Molecule*, or NULL
  Molecule *check_pickable(Pickable *pobj);
};

#endif

