/***************************************************************************
 *cr
 *cr		(C) Copyright 1995 The Board of Trustees of the
 *cr			    University of Illinois
 *cr			     All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeSigma.C,v $
 *	$Author: leech $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 96/11/22 03:04:57 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A MoleculeSigma is inherited from MoleculeFile, and supplies additional
 * routines to link the Molecule with the SIgMA MD code.
 *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/param.h>

#include <mdio/mdsocket.h>
#include <mdio/global.h>
#include <mdio/Communication.h>

#include "CmdSigma.h"
#include "MoleculeSigma.h"
#include "DrawTugs.h"
#include "PickModeDrag.h"
#include "Atom.h"
#include "ReadPDB.h"
#include "ReadPSF.h"
#include "Scene.h"
#include "DisplayDevice.h"
#include "Mouse.h"
#include "Global.h"
#include "utilities.h"

// identifying code passed to add_pick_mode and returned to
//  create_pick_mode - probably no longer needed, since "Tugs" pick mode
//  is created in GeometryList constructor
static const int pickModeId = 42;

// constructor sets up communications with Sigma
// The parameter is the Molecule object corresponding to the Sigma simulation
MoleculeSigma::MoleculeSigma(Molecule *_mol) : mol(_mol) {
    // Indicate we're not tugging anything initially
    cur_tug_id = -1;

    // In mouse update mode at first, with nothing received
    update_mode = SmdMouse;
    received_update = 0;

    // Force constant has some reasonable intial value
    tug_force = 10.0;

    // We might want to query the current molecular coordinates here, so the
    //	display doesn't jump when the first update comes upon starting MD (this
    //	happens because the PDB file coordinates do not neccessarily match the
    //	initial coordinates in Sigma after reading snapshots or doing other
    //	setup work).

    // Look for an existing "Tugs" pick mode, created by the GeometryList.
    //	We know the PickList to use is the global Scene
    tugPickMode = ::scene->pick_mode_num("Tugs");
    if (tugPickMode == -1)
	msgErr << "MoleculeSigma: can't find pick mode for Tugs" << sendmsg;

    // Let the command interface know the simulation being talked to
    msgInfo << "MoleculeSigma: registering self with CmdSigma interface" << sendmsg;
    CmdSigmaEvent::register_simulation(this);

    drawTugs = new DrawTugs(mol, this, tugPickMode);

    // Socket is not connected yet
    sock = -1;

// Temporary hack - this should be passed to the constructor instead
if (getenv("SMD_DISCONNECT") == NULL) {

    // Initialize connection to SIgMA, and set up our special pick mode
    connect();

    // Send the startup message.
    buffer_send_event(MD_STARTUP);
} else {
    msgInfo << "MoleculeSigma: use \"sigma connect\" to start" << sendmsg;
}
}

// destructor shuts down communications
MoleculeSigma::~MoleculeSigma(void) {
    // Clean up
    delete drawTugs;

    // Send shutdown message to Sigma
    if (connected()) {
	buffer_send_event(MD_EXIT);
	send_buffer();
	sleep(2);
	disconnect();
    }
}

// Connect to a SIgMA process (user must start it up by hand at present)
void MoleculeSigma::connect() {
    char host[MAXHOSTNAMELEN];

    if (gethostname(host, sizeof(host)) < 0) {
       msgErr << "MoleculeSigma::connect: can't get host name" << sendmsg;
       return;
    }

    // Listen for an incoming connection
    int port = 0,
	listenfd = socket_connect(host, port, 1, &port);

    if (listenfd < 0) {
	msgErr << "MoleculeSigma::connect: can't establish listening socket" << sendmsg;
	return;
    }

    msgInfo << "Connect SIgMA to: " << host << " " << port << sendmsg;

    // Wait for a SIgMA process to connect to the listener
    sock = socket_accept(listenfd);
    close(listenfd);

    if (sock < 0) {
	msgErr << "MoleculeSigma::connect: accept on listening socket failed" << sendmsg;
	return;
    }

    buffer_set_socket(sock);
    if (buffer_handshake() < 0) {
	msgErr << "MoleculeSigma::connect: setup failed" << sendmsg;
	sock = -1;
	return;
    }

    // Don't block on reads from socket
    socket_setblocking(sock, 0);
}

// Disconnect from a SIgMA process
void MoleculeSigma::disconnect() {
    if (sock < 0)
	return;

    close(sock);
    sock = -1;
}

// Indicate if we're currently connected
int MoleculeSigma::connected() {
    return (sock >= 0);
}

// prepare to draw the molecule; handle messages on the communications
//  socket to Sigma. Request another frame if in synchronous mode.
void MoleculeSigma::read_input() {
    if (!connected())
	return;

    int event;
    char *buf;

    while (get_event(&event, &buf, TRUE) != -1) {
	switch (event) {
	    case MD_SET_COORDINATES:
		read_coords(buf);
		received_update = 1;
		break;
	    default:
		msgErr << "MoleculeSigma::prepare(): unknown event "
		       << md_message_name(event) << sendmsg;
		break;
	}
    }

    if (received_update && update_mode == SmdSynchronous) {
//GUIDEBUG(0, "MoleculeSigma::read_input: send request for synchronous update" << sendmsg);
	send_event(MD_COMPUTE_MODE, SmdSynchronous);
    }
}

// Check if event is in valid_events[0..num_valid_events-1]
static int valid_message(int event, int *valid_events, int num_valid_events) {
    for (int i = 0; i < num_valid_events; i++) {
	if (event == valid_events[i])
	    return TRUE;
    }
    msgErr << "MoleculeSigma::send_event: invalid event type " << event << sendmsg;
    return FALSE;
}
#define ASIZE(a) (sizeof(a) / sizeof(a[0]))

// Send an event with no arguments
void MoleculeSigma::send_event(int event) {
    static int valid_events[] = {
	SMD_CMD_CONNECT, SMD_CMD_DISCONNECT, SMD_CMD_LIST_TUGS, MD_STARTUP,
	MD_UPDATE_PROTEIN, MD_UPDATE_ALL, MD_WRITE_SNAPSHOT };

    if (!valid_message(event, valid_events, ASIZE(valid_events)))
	return;

    // Handle "messages" which are only done on the VMD side
    if (event == SMD_CMD_CONNECT) {
	if (connected())
	    msgErr << "MoleculeSigma: already connected to SigmaX!" << sendmsg;
	else
	    connect();
	return;
    }

    if (event == SMD_CMD_DISCONNECT) {
	if (!connected())
	    msgErr << "MoleculeSigma: not connected to SigmaX!" << sendmsg;
	else
	    disconnect();
	return;
    }

    // List all tugs
    if (event == SMD_CMD_LIST_TUGS) {
	drawTugs->list_tugs(-1);
	return;
    }

    if (connected())
	buffer_send_event(event);
}

// Send an event with an integer argument
void MoleculeSigma::send_event(int event, int param) {
    static int valid_events[] = {
	MD_UPDATE_PROTEIN_FREQUENCY, MD_UPDATE_SOLVENT_FREQUENCY,
	MD_COMPUTE_MODE };

    if (!valid_message(event, valid_events, ASIZE(valid_events)))
	return;

    // Handle "messages" which are only done on the VMD side
    // (none at present)

    // Keep track of the mode, so we can synchronize updates when needed
    if (event == MD_COMPUTE_MODE) {
	if (param < SmdIdle || param > SmdContinuous) {
	    msgErr << "MoleculeSigma::send_event: bad update mode " << param << sendmsg;
	    return;
	}

	update_mode = (SmdComputeMode)param;
	received_update = 0;
//GUIDEBUG(0, "MoleculeSigma::send_event: updatemode = " << update_mode << sendmsg);
    }

    if (connected())
	buffer_send_event(event, param);
}

// Send an event with a float argument
// For now, send float param as first coordinate of a Point3
void MoleculeSigma::send_event(int event, float param) {
    // Handle "messages" which are only done on the VMD side

    // Change the force constant, used for updating tugs
    if (event == SMD_CMD_FORCE_CONSTANT) {
	tug_force = param;
	return;
    }

    if (connected())
	send_event(event, param, 0.0, 0.0);
}

// Send an event with a Point3 (x,y,z) argument
void MoleculeSigma::send_event(int event, float x, float y, float z) {
    static int valid_events[] = { MD_NUM_EVENTS };  // none

    if (!valid_message(event, valid_events, 0))
	return;

    // Handle "messages" which are only done on the VMD side
    // (none at present)

    Point3 pt;
    pt.x = x;
    pt.y = y;
    pt.z = z;

    if (connected())
	buffer_send_event(event, &pt);
}

void MoleculeSigma::send_event(int event, TugInfo *tug) {
    static int valid_events[] = {
	MD_CREATE_TUG, MD_MOVE_TUG, MD_REMOVE_TUG };

    if (!valid_message(event, valid_events, ASIZE(valid_events)))
	return;

    // Handle "messages" which are only done on the VMD side
    // (none at present)

    // remove_tug() also sends the remove message to Sigma
    if (event == MD_REMOVE_TUG) {
	drawTugs->remove_tug(tug->tug_id);
	return;
    }
    // also MD_ADD_TUG, MD_MOVE_TUG


    if (connected())
	send_tug_event(event, tug);
}

// Read molecular coordinates from Sigma & update the model
void MoleculeSigma::read_coords(char *bufptr) {
    int ranges;

    bufptr = buffer_read_int(&ranges, bufptr);
    SIGMADEBUG(2, "MoleculeSigma::read_coords: ranges = " << ranges << sendmsg);
    int *from = new int[ranges],
	*to = new int[ranges],
	num_coords = 0;

    for (int i = 0; i < ranges; i++) {
	bufptr = buffer_read_int(&from[i], bufptr);
	bufptr = buffer_read_int(&to[i], bufptr);
	num_coords += to[i] - from[i] + 1;
	SIGMADEBUG(2, "\trange " << i << " from = " << from[i]
		<< " to = " << to[i] << sendmsg);
    }
    SIGMADEBUG(2, "# coordinates = " << num_coords << sendmsg);

    Timestep *newts;
    int new_frame = TRUE;   // Whether to add a new frame

    if (new_frame) {
	SIGMADEBUG(1, "MoleculeSigma::read_coords: creating new timestep with"
	    << mol->nAtoms << " atoms" << sendmsg);
	newts = new Timestep(mol->nAtoms, 0.0);
    } else {
	newts = mol->item(mol->num() - 1);
	if (newts == NULL) {
	    msgErr << "MoleculeSigma::read_coords: no Timestep to update!" << sendmsg;
	    return;
	}
    }

    // Pointer to data buffer in Timestep
    float *newts_xyz = newts->pos;

// NOTE: this doesn't yet handle the case when not all coordinates are
//  being sent (they should probably be inherited from the last timestep)
// This will generate major boo-boos since non-updated atoms will have
//  random coordinates.
    // Loop over each range reading all the coordinates in that range
    //	and updating the model. VMD indices are 0-based, Sculpt
    //	protocol is 1-based, so offset indices appropriately
    for (i = 0; i < ranges; i++) {
	Point3 pos;

	SIGMADEBUG(2, "range " << i << ": " << from[i] << " to " << to[i] << sendmsg);
	for (int j = from[i]; j <= to[i]; j++) {
	    int anum = j - 1;	// Offset from Sculpt to VMD indices

	    // Throw away atoms outside the defined range (generate an error?)
	    if (anum < 0 || anum >= mol->nAtoms)
		continue;
	    MolAtom *aptr = mol->atom(anum);

	    bufptr = buffer_read_point3(&pos, bufptr);
	    aptr->pos[0] = pos.x;
	    aptr->pos[1] = pos.y;
	    aptr->pos[2] = pos.z;

	    // Get address of anum'th point in Timestep buffer
	    float *dptr = newts_xyz + anum * 3;
	    dptr[0] = pos.x;
	    dptr[1] = pos.y;
	    dptr[2] = pos.z;
	    if (j == from[i]) {
		SIGMADEBUG(3, "\tatom " << j << " coord " << dptr[0] << ' '
		    << dptr[1] << ' ' << dptr[2] << sendmsg;)
	    }
	}
    }

    delete [] from;
    delete [] to;

    if (new_frame) {
	// new_frame_number =
	mol->append_frame(newts);   // returns new frame number
	SIGMADEBUG(1, "MoleculeSigma::read_coords: adding new timestep" << sendmsg);
    } else {
	newts->init();	    // Calculate bounds for these atoms
	mol->curr_frame_changed();   // Indicate frame data has changed
	//new_frame_number = mol->num() - 1;
	SIGMADEBUG(1, "MoleculeSigma::read_coords: updating current timestep" << sendmsg);
    }
}

// Determine if we're interested in this pick mode
int MoleculeSigma::want_pick_mode(int mode) {
    return (mode == tugPickMode);
}

PickMode *MoleculeSigma::create_pick_mode(int id) {
    if (id == pickModeId) {
	SIGMADEBUG(1, "\tMoleculeSigma::create_pick_mode: mode " << id
	    << ", creating new PickModeDrag" << sendmsg);
	return new PickModeDrag;
    } else {
	SIGMADEBUG(1, "\tMoleculeSigma::create_pick_mode: unknown mode " << id << sendmsg);
	return NULL;
    }
}

// See if the Pickable is the molecule, or one of its children components.
//  Note that the DrawForce object is *not* one of its children; picks
//  on existing tugs are handled by DrawForce.
// Returns -1 or the component number or 1 (if this->mol == p)
int MoleculeSigma::find_representation(Pickable *p) {
    if (p == this->mol) {
	return 1;
    } else {
	int repnum = mol->components();
	for (int j=0; j < repnum; j++) {
	    Pickable *c = mol->component(j);
	    if (p == c)
		return j;
	}
    }
    return -1;
}

// Copy current position of atom_id to pos[]
// Returns 0 on success, -1 on failure
int MoleculeSigma::get_atom_position(int atom_id, float *pos) {
    Timestep *newts = mol->item(mol->num() - 1);

    // Timestep available?
    if (newts == NULL)
	return -1;

    // Valid atom ID?
    if (atom_id < 0)
	return -1;

    // Offset from data buffer in Timestep
    float *p = newts->pos + atom_id * 3;
    for (int i = 0; i < 3; i++)
	pos[i] = p[i];

    return 0;
}

// called when a pick is begun
//    args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void MoleculeSigma::pick_start(DisplayDevice * /* d */, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    // See if the item selected is one of this molecule's representations.
    if (find_representation(p) < 0) {
	SIGMADEBUG(1, "MS::pick_start: pickable not a representation, returning "
	    << sendmsg);
	return;
    }
    if (!want_pick_mode(m)) {
	SIGMADEBUG(1, "MS::pick_start: unknown mode " << m << sendmsg);
	return;
    }

    // Initial tug position is the atom position, since that's where
    //	the mouse is.
    float tug_pos[3];
    if (get_atom_position(tag, tug_pos) < 0) {
	msgErr << "MS::pick_start: no atom position available for atom " << tag << sendmsg;
	return;
    }

    SIGMADEBUG(1, "MS: pick_start tag = " << tag << " dim = " << dim
	    << " button = " << b << " mode = " << m << sendmsg);
    SIGMADEBUG(1, "\tpos = " << pos[0] << "\t" << pos[1]
	    << "\t" << pos[2] << sendmsg);

    // This shouldn't happen, but if it does, clean up after the old
    //	tug
    if (cur_tug_id != -1) {
	msgErr << "MoleculeSigma: creating a new tug when one already exists!" << sendmsg;
	drawTugs->finalize_tug(cur_tug_id);
    }

    // Tell Sigma to create the restraint, and create a visual representation
    //	in VMD.
    cur_tug_id = drawTugs->add_tug(tag, tug_pos, tug_force);
}

// called when a pick moves
void MoleculeSigma::pick_move(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    if (find_representation(p) < 0)
	return;
    if (!want_pick_mode(m))
	return;
    if (cur_tug_id == -1) {
	msgErr << "MS::pick_move: no current tug to adjust!" << sendmsg;
	return;
    }

    float atom_pos[3];
    if (get_atom_position(tag, atom_pos) < 0)
	return;

    SIGMADEBUG(2, "MS::pick_move tag = " << tag << " dim = " << dim << " pos = "
	    << pos[0] << " " << pos[1] << " " << pos[2] << sendmsg);
    SIGMADEBUG(2, "\tbutton = " << b << " mode = " << m << sendmsg);
    SIGMADEBUG(1, "\tatom pos = " << atom_pos[0] << " "
	    << atom_pos[1] << " " << atom_pos[2] << sendmsg);

    float tug_pos[3];
    map_pick_move(d, mol->tm, dim, pos, atom_pos, tug_pos);

    SIGMADEBUG(2, "\ttug position set to " << tug_pos[0] << " "
	    << tug_pos[1] << " " << tug_pos[2] << sendmsg);

    // Update restraint position in SIgMA and visual representation
    // Update visual representation
    drawTugs->move_tug(cur_tug_id, tug_pos);
}

// called when a pick ends
void MoleculeSigma::pick_end(DisplayDevice * /* d */, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    if (find_representation(p) < 0)
	return;
    if (!want_pick_mode(m))
	return;
    if (cur_tug_id == -1)
	return;

    SIGMADEBUG(1, "MS: pick_end tag = " << tag << " dim = " << dim << " pos = "
	    << pos[0] << "\t" << pos[1] << "\t" << pos[2] << sendmsg);
    SIGMADEBUG(2, "MS: pick_end button = " << b << " mode = " << m << sendmsg);

    // Tell Sigma to stop specifying the restraint, update the visual
    //	representation, and indicate there's no current tug.
    drawTugs->finalize_tug(cur_tug_id);
    cur_tug_id = -1;
}

/////////////////////////////////////////////////////////////////////
// class constructor - specifies the coordinate file

MoleculeSigmaFilePDB::MoleculeSigmaFilePDB(char *fname, Scene *sc)
	: MoleculeFilePDB(fname, sc) {
    register_with_picklist(origScene);
    sigma = new MoleculeSigma(this);
    msgInfo << "MoleculeSigmaFilePDB: creating from PDB " << fname << sendmsg;
}

// this version uses a Displayable instead of Scene, so that it becomes a
// child Displayable to some parent.
MoleculeSigmaFilePDB::MoleculeSigmaFilePDB(char *fname, Displayable *dp)
	 : MoleculeFilePDB(fname, dp) {
    register_with_picklist(origScene);
    sigma = new MoleculeSigma(this);
    msgInfo << "MoleculeSigmaFilePDB: creating from PDB " << fname << sendmsg;
}

MoleculeSigmaFilePDB::~MoleculeSigmaFilePDB(void) {
    delete sigma;
    MSGDEBUG(1,"Deleting MoleculeSigmaFilePDB ..." << sendmsg);
}

// prepare to draw the molecule; read messages from Sigma
void MoleculeSigmaFilePDB::prepare(DisplayDevice *d) {
    sigma->read_input();
    Molecule::prepare(d);
}

// Determine if we're interested in this pick mode
int MoleculeSigmaFilePDB::want_pick_mode(int mode) {
    if (sigma->want_pick_mode(mode))
	return TRUE;
    else
	return MoleculeFilePDB::want_pick_mode(mode);
}

PickMode *MoleculeSigmaFilePDB::create_pick_mode(int id) {
    PickMode *p = sigma->create_pick_mode(id);

    if (p != NULL)
	return p;
    else
	return MoleculeFilePDB::create_pick_mode(id);
}

// called when a pick is begun
void MoleculeSigmaFilePDB::pick_start(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    MoleculeFilePDB::pick_start(d,p,b,m,tag,dim,pos);
    sigma->pick_start(d,p,b,m,tag,dim,pos);
}

// called when a pick moves
void MoleculeSigmaFilePDB::pick_move(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    MoleculeFilePDB::pick_move(d,p,b,m,tag,dim,pos);
    sigma->pick_move(d,p,b,m,tag,dim,pos);
}

// called when a pick ends
void MoleculeSigmaFilePDB::pick_end(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    MoleculeFilePDB::pick_end(d,p,b,m,tag,dim,pos);
    sigma->pick_end(d,p,b,m,tag,dim,pos);
}

//////////////////////////////////////////////////////////////////////////////
// class constructor - specifies the structure file, coordinates are
//  added later

MoleculeSigmaFilePSF::MoleculeSigmaFilePSF(char *fname, Scene *sc)
	: MoleculeFilePSF(fname, sc) {
    register_with_picklist(origScene);
    sigma = new MoleculeSigma(this);
    msgInfo << "MoleculeSigmaFilePSF: creating from PSF " << fname << sendmsg;
}

// this version uses a Displayable instead of Scene, so that it becomes a
// child Displayable to some parent.
MoleculeSigmaFilePSF::MoleculeSigmaFilePSF(char *fname, Displayable *dp)
	 : MoleculeFilePSF(fname, dp) {
    register_with_picklist(origScene);
    sigma = new MoleculeSigma(this);
    msgInfo << "MoleculeSigmaFilePSF: creating from PSF " << fname << sendmsg;
}

MoleculeSigmaFilePSF::~MoleculeSigmaFilePSF(void) {
    delete sigma;
    MSGDEBUG(1,"Deleting MoleculeSigmaFilePSF ..." << sendmsg);
}

// prepare to draw the molecule; read messages from Sigma
void MoleculeSigmaFilePSF::prepare(DisplayDevice *d) {
    sigma->read_input();
    Molecule::prepare(d);
}

// Determine if we're interested in this pick mode
int MoleculeSigmaFilePSF::want_pick_mode(int mode) {
    if (sigma->want_pick_mode(mode))
	return TRUE;
    else
	return MoleculeFilePSF::want_pick_mode(mode);
}

PickMode *MoleculeSigmaFilePSF::create_pick_mode(int id) {
    PickMode *p = sigma->create_pick_mode(id);

    if (p != NULL)
	return p;
    else
	return MoleculeFilePSF::create_pick_mode(id);
}

// called when a pick is begun
void MoleculeSigmaFilePSF::pick_start(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    MoleculeFilePSF::pick_start(d,p,b,m,tag,dim,pos);
    sigma->pick_start(d,p,b,m,tag,dim,pos);
}

// called when a pick moves
void MoleculeSigmaFilePSF::pick_move(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    MoleculeFilePSF::pick_move(d,p,b,m,tag,dim,pos);
    sigma->pick_move(d,p,b,m,tag,dim,pos);
}

// called when a pick ends
void MoleculeSigmaFilePSF::pick_end(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    MoleculeFilePSF::pick_end(d,p,b,m,tag,dim,pos);
    sigma->pick_end(d,p,b,m,tag,dim,pos);
}
