/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: OpenGLDisplayDevice.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.15 $	$Date: 1997/03/25 05:58:37 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that are OpenGL-specific.  Will render drawing
 * commands into a single X window.
 *
 ***************************************************************************/

#define VMDDEBUG
#ifdef VMDFORMS
#define VMDFORMS_NOTYET
#endif

#include "OpenGLDisplayDevice.h"
#include "Inform.h"
#include "utilities.h"
#include <stdlib.h>
#include <math.h>
#include <GL/glx.h>
#include <GL/gl.h>
#include <X11/cursorfont.h>

#ifdef VMDFORMS
// xforms has a button_down which conflicts with stuff we do
#ifdef button_down
#undef button_down
#define button_down button_down_forms
#endif
#include "forms.h"
#undef button_down
#endif

#ifndef YMAXSTEREO
#define YMAXSTEREO      491
#define YOFFSET         532
#endif


// static data for this object
static char *glStereoNameStr[OPENGL_STEREO_MODES] = { "Off",
	"CrystalEyes", "CrossEyes", "SideBySide", "Left", "Right" };


// X-window  visual configuration needed
int configuration[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_STEREO, None};
int configuration2[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, 16,
None};
int configuration3[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_ALPHA_SIZE, 0,
GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1, GLX_BLUE_SIZE, 1, None};

// size hints for opening a new window
XSizeHints sizeHints = {0};

// colors for cursors
XColor cursorFG = { 0, 0xffff, 0, 0, DoRed | DoGreen | DoBlue, 0 };
XColor cursorBG = { 0, 0xffff, 0xffff, 0xffff, DoRed | DoGreen | DoBlue, 0 };

// lighting characteristics default data
static GLfloat defSpecular[] = { 1.0, 1.0, 1.0, 1.0 };
static GLfloat defAmbient[] =  { 0.0, 0.0, 0.0, 1.0 };
static GLfloat defShininess = 40.0;

/////////////////////////  constructor and destructor  

// constructor ... open a window and set initial default values
OpenGLDisplayDevice::OpenGLDisplayDevice(int argc, char **argv,
					 int *size, int *loc)
    : OpenGLRenderer("vmd OpenGL Display") {

  MSGDEBUG(1, "Creating OpenGLDisplayDevice ... " << sendmsg);

  // set up data possible before opening window
  stereoNames = glStereoNameStr;
  stereoModes = OPENGL_STEREO_MODES;
  mainMenu = NULL;
  dpy = NULL;
  dpyScreen = 0;
  screenX = screenY = 0;

  // open the window
  windowID = open_window(name, size, loc, argc, argv);

  // set flags for the capabilities of this display
  has2D = TRUE;
  aaAvailable = TRUE;
  cueingAvailable = TRUE;

  // set default settings
  aa_on();
  cueing_off();
  set_sphere_mode(sphereMode);
  set_sphere_res(sphereRes);
  set_line_width(lineWidth);
  set_line_style(lineStyle);

  // reshape and clear the display, which initializes some other variables
  reshape();
  normal();
  clear();
  update();
}

// destructor ... close the window
OpenGLDisplayDevice::~OpenGLDisplayDevice(void) {

 // delete the current pop-up menu, if any
 menu_delete();

 // free the display lists used for the font
 if(fontStruct)
   glDeleteLists(fontListBase, 256);

 // close and delete windows, contexts, and display connections
 XUnmapWindow(dpy, windowID);
#ifdef VMDFORMS_NOTYET
  glXDestroyContext(dpy, cx);
#endif
 XDestroyWindow(dpy, windowID);
#ifndef VMDFORMS
 XCloseDisplay(dpy);
#endif
}


/////////////////////////  protected nonvirtual routines  

// return class of X visual ... this is because 'class' is a reserved C++
// word, and we have to access vi->class 
int get_visual_class(XVisualInfo *vi) {
  char *data = (char *)vi;
  data += sizeof(Visual *);
  data += sizeof(VisualID);
  data += sizeof(int);
  data += sizeof(unsigned int);
  return *((int *)data);
}


// create a new window and set it's characteristics
// STOLEN DIRECTLY FROM MARK KILGARD'S OpenGL + X ARTICLES ... THANKS MARK
Window OpenGLDisplayDevice::open_window(char *nm, int *size, int *loc,
					int /* argc */, char** /* argv */) {
  Window win;
  int i, SX = 596, SY = 190, W = 669, H = 834;
  if (loc) {
    SX = loc[0];
    SY = loc[1];
  }
  if (size) {
    W = size[0];
    H = size[1];
  }
  int mmm = 0;
  MSGDEBUG(2,"OpenGLDisplayDevice: opening window ..." << sendmsg);
  // (1) process X command-line arguments
  // not yet done here ...

  // (2) open a connection to the X server
#ifdef VMDFORMS
  dpy = fl_display;
#else
  char *dispname = getenv("DISPLAY");
  if(!(dpy = XOpenDisplay(dispname))) {
    msgErr << "Cannot open display.  Exiting ..." << sendmsg;
    exit(1);
  }
#endif
  // get info about root window
  dpyScreen = DefaultScreen(dpy);
  rootWindowID = RootWindow(dpy, dpyScreen);
  screenX = DisplayWidth(dpy, dpyScreen);
  screenY = DisplayHeight(dpy, dpyScreen);

  // (3) make sure the GLX extension is available
  if (!glXQueryExtension(dpy, NULL, NULL)) {
    msgErr << "The X server does not support the OpenGL GLX extension.";
    msgErr << "   Exiting ..." << sendmsg;
    XCloseDisplay(dpy);
    exit(1);
  }

#ifdef VMDFORMS_NOTYET
  int config[] = {GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_DOUBLEBUFFER, None};

  //GLXContext context;
  //cx = 0;

  //fl_initialize(&argc, argv, "FormDemo", 0, 0);

  if (!(win = fl_glwincreate(config, &cx, 250, 250))) {
    msgErr << "Couldn't create OpenGL window with forms library.";
    msgErr << "   Exiting ..." << sendmsg;
    exit(1);
  }
  XStoreName(dpy, win, nm);                   // set the window name
  //  This originally came from GL, which locates things with respect
  // to the lower-left corner of the window, and X uses the upper-left
  //  msgInfo << "SX " << SX << "  SY " << screenY - SY << " W " << W 
  //	  << " H " << H << sendmsg;
  if (W  < 20) W = 20;
  if (H  < 20) H = 20;
  XMoveResizeWindow(dpy, win, SX, screenY - SY, W, H);  // and size

  // create cursors to use in window
   cursor[0] = XCreateFontCursor(dpy, XC_left_ptr);
   cursor[1] = XCreateFontCursor(dpy, XC_fleur);
   cursor[2] = XCreateFontCursor(dpy, XC_sb_h_double_arrow);
   cursor[3] = XCreateFontCursor(dpy, XC_crosshair);
   cursor[4] = XCreateFontCursor(dpy, XC_watch);
   for(i=0; i < 5; i++)
     XRecolorCursor(dpy, cursor[i], &cursorFG, &cursorBG);
#else
  // (4) find an appropriate visual and colormap ...
  // we want double-buffered RGB with a Z buffer (possibly with stereo)
  stereoInWindow = TRUE;
  vi = glXChooseVisual(dpy, dpyScreen, configuration);
  if (!vi || (get_visual_class(vi) != TrueColor)) {
    // stereo not available; flag this, and try without stereo
    stereoInWindow = FALSE;
    vi = glXChooseVisual(dpy, dpyScreen, configuration2);
  } else {
    // we have a stereo, TrueColor visual available
    msgWarn << "Stereo in a window IS available." << sendmsg;
  }

  // check if we have a TrueColor visual.
  if( !vi || (get_visual_class(vi) != TrueColor) ) {
    // still no TrueColor.  Try again, with a very basic request ...
    vi = glXChooseVisual(dpy, dpyScreen, configuration3);
  }

  // make sure we have what we want, darnit ...
  if (!vi) {
    msgErr << "A TrueColor visual is required, but not available.\n";
    msgErr << "The X server is not capable of displaying double-buffered, \n";
    msgErr << "RGB images with a Z buffer.   Exiting ..." << sendmsg;
    exit(1);
  }

  Atom wmDeleteWindow = XInternAtom(dpy, "WM_DELETE_WINDOW", False);

  // (5) create an OpenGL rendering context
  if(!(cx = glXCreateContext(dpy, vi, None, GL_TRUE))) {
    msgErr << "Could not create OpenGL rendering context.  Exiting ...";
    msgErr << sendmsg;
    exit(1);
  }

  // (5.5) create cursors to use in window
  cursor[0] = XCreateFontCursor(dpy, XC_left_ptr);
  cursor[1] = XCreateFontCursor(dpy, XC_fleur);
  cursor[2] = XCreateFontCursor(dpy, XC_sb_h_double_arrow);
  cursor[3] = XCreateFontCursor(dpy, XC_crosshair);
  cursor[4] = XCreateFontCursor(dpy, XC_watch);
  for(i=0; i < 5; i++)
    XRecolorCursor(dpy, cursor[i], &cursorFG, &cursorBG);

  // (6) create an X window with selected visual and right properties
  XSetWindowAttributes swa;
//#ifdef VMDFORMS
//  swa.colormap = fl_create_colormap(vi, 1);
//#else
//  swa.colormap = XCreateColormap(dpy, rootWindowID, vi->visual, AllocNone);
  cout << "Output from color map is: " << 
    (swa.colormap = XCreateColormap(dpy, rootWindowID, vi->visual, AllocAll))
    << endl;
//#endif
  swa.background_pixmap = None;
  swa.border_pixel=0;
  swa.event_mask = ExposureMask;
  swa.cursor = cursor[0];
  if(size) {
    W = size[0];
    H = size[1];
  }
  if(loc) {
    SX = loc[0];
    // X screen uses Y increasing from upper-left corner down; this is
    // opposite to what GL does, which is the way VMD was set up originally
    SY = screenY - loc[1] - H;
  }

  win = XCreateWindow(dpy, rootWindowID, SX, SY, W, H, 0,
		      vi->depth, InputOutput, vi->visual,
		      CWBorderPixel | CWColormap | CWEventMask, &swa);
  XInstallColormap(dpy, swa.colormap);

  sizeHints.flags |= USSize;
  sizeHints.flags |= USPosition;
  sizeHints.width = W;
  sizeHints.height = H;
  sizeHints.x = SX;
  sizeHints.y = SY;
  msgInfo << " mmm is " << mmm++ << sendmsg;
  XSetStandardProperties(dpy, win, nm, "vmd", None, argv, argc, &sizeHints);
  msgInfo << " mmm is " << mmm++ << sendmsg;
  XWMHints *wmHints = XAllocWMHints();
  msgInfo << " mmm is " << mmm++ << sendmsg;
  wmHints->initial_state = NormalState;
  msgInfo << " mmm is " << mmm++ << sendmsg;
  wmHints->flags = StateHint;
  msgInfo << " mmm is " << mmm++ << sendmsg;
  XSetWMHints(dpy, win, wmHints);
  msgInfo << " mmm is " << mmm++ << sendmsg;
  //  Atom wmDeleteWindow = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
  msgInfo << " mmm is " << mmm++ << sendmsg;
  XSetWMProtocols(dpy, win, &wmDeleteWindow, 1);
  msgInfo << " mmm is " << mmm++ << sendmsg;

  // (7) bind the rendering context to the window
  glXMakeCurrent(dpy, win, cx);
  msgInfo << " mmm is " << mmm++ << sendmsg;
#endif

  // (8) actually request the window to be displayed
  XSelectInput(dpy, win, KeyPressMask | ButtonPressMask | ButtonReleaseMask |
	       StructureNotifyMask | ExposureMask);
  //  msgInfo << " mmm is " << mmm++ << sendmsg;
  XMapRaised(dpy, win);
  //  msgInfo << " mmm is " << mmm++ << sendmsg;

  // (9) configure the rendering properly
  glDepthFunc(GL_LEQUAL);
  glEnable(GL_DEPTH_TEST);           // use zbuffer for hidden-line removal
  glClearDepth(1.0);

  // automatically normalize normals
  glEnable(GL_NORMALIZE);

  // configure for dashed lines ... but initially show solid lines
  glLineStipple(1, 0x3333);
  glDisable(GL_LINE_STIPPLE);

  // configure the fogging characteristics ... color and position of fog
  // are set during the clear routine
  glFogi(GL_FOG_MODE, GL_LINEAR);

  // configure the light model
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, defShininess);
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, defAmbient);
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, defSpecular);
  glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
  glEnable(GL_COLOR_MATERIAL);       // have materials set by curr color
  glDisable(GL_POLYGON_SMOOTH);      // make sure not to antialias polygons
  do_activate_material(materialOn, materialsActive);

  MSGDEBUG(2,"OpenGLDisplayDevice: defining lights ..." << sendmsg);

  // turn on lights if necessary
  for(i=0; i < DISP_LIGHTS; i++) {
    if(lightDefined[i]) {
      do_define_light(i, lightColor[i], lightPos[i]);
      do_activate_light(i, lightOn[i]);
    } else
      do_activate_light(i, FALSE);
  }

  MSGDEBUG(2,"OpenGLDisplayDevice: defining materials ..." << sendmsg);

  // define materials if necessary
  for(i=0; i < MAXCOLORS; i++) {
    if(matDefined[i])
      do_define_material(i, matData[i]);
  }

  // load current transformation matrix on stack
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  // create font bitmaps for use in displaying text
  fontStruct = XLoadQueryFont(dpy, "fixed");
  if(!fontStruct) {
    msgErr << "Could not load font 'fixed' for text display." << sendmsg;
  } else {
    // get a range of display lists to use for the characters
    haveFontData = TRUE;
    fontListBase = glGenLists(256);
    glXUseXFont(fontStruct->fid, 0, 256, fontListBase);
  }

  // return window id
  return win;
}


// set the current perspective, based on the eye position and where we
// are looking.
void OpenGLDisplayDevice::set_persp(DisplayEye my_eye) {

  if(dim() == 3) {	// use window perspective for 3D view
    // set projection
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    if (projection() == PERSPECTIVE)
       {
       glFrustum((GLdouble)cpLeft, (GLdouble)cpRight, (GLdouble)cpDown,
  	         (GLdouble)cpUp, (GLdouble)nearClip, (GLdouble)farClip);
       }
    else
       {
       glOrtho( -0.25 * vSize * Aspect, 0.25 * vSize * Aspect,
             -0.25 * vSize, 0.25 * vSize, 
             nearClip, farClip);
       }

/*
    gluPerspective(90.0, (GLdouble)Aspect,
		   (GLdouble)nearClip, (GLdouble)farClip);
*/
    // define eye and look at some point.  Assumes up vector = (0,1,0)
    GLdouble ep[3];
    if(my_eye == NOSTEREO) {
      ep[0] = eyePos[0];
      ep[1] = eyePos[1];
      ep[2] = eyePos[2];
    } else if(my_eye == LEFTEYE) {
      ep[0] = eyePos[0] - eyeSepDir[0];
      ep[1] = eyePos[1] - eyeSepDir[1];
      ep[2] = eyePos[2] - eyeSepDir[2];
    } else if(my_eye == RIGHTEYE) {
      ep[0] = eyePos[0] + eyeSepDir[0];
      ep[1] = eyePos[1] + eyeSepDir[1];
      ep[2] = eyePos[2] + eyeSepDir[2];
    }
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(ep[0], ep[1], ep[2],
	      (GLdouble)(eyePos[0] + eyeDir[0]),
	      (GLdouble)(eyePos[1] + eyeDir[1]),
	      (GLdouble)(eyePos[2] + eyeDir[2]),
	      (GLdouble)(upDir[0]),
	      (GLdouble)(upDir[1]),
	      (GLdouble)(upDir[2]));

  } else {                     // use (0 .. 1, 0 ..1) window for 2D view
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, 1.0, 0.0, 1.0, -1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(0.5, 0.5, -1.0, 0.5, 0.5, 0.0, 0.0, 1.0, 0.0);
  }
}


/////////////////////////  public virtual routines  

//
// virtual routines to create and activate a popup (possibly pull-down) menu 
// These only work if compiled with xforms; otherwise, they do nothing.

// static flag telling us whether we've initialized the popup menus yet
#ifdef VMDFORMS
static int setPupMax = FALSE;
#endif

// given a PopupMenu definition, create it in the window (but do not
// activate it yet).  Return success.
int OpenGLDisplayDevice::menu_create(PopupMenu *pm) {
#ifdef VMDFORMS_NOTYET
  char mbuf[128];
  int needSep, itemAdded, totalAdded = 0;
  long addSubMenu;

  // set maximum number of popups if we have not done so yet
  if(!setPupMax) {
    fl_setpup_maxpup(1024);
    setPupMax = TRUE;
  }

  // start at the top level, and create items until a submenu is reached; at
  // that point create the submenu
  int n = pm->items();
  MSGDEBUG(2,"OpenGLDisplayDevice: Number of menu items ..."<< n << sendmsg);
  if(pm->menuExists)
    fl_freepup(pm->menuID);

  // only create menu if there are items in it
  if(n > 0) {

    // create the new menu
    pm->menuID = fl_newpup(windowID);
    pm->menuExists = TRUE;
    sprintf(mbuf,"%s%%t",pm->name());

    fl_addtopup(pm->menuID, mbuf, 0);

    // add all the items
    for(int i=0; i < n; i++) {
      PopupMenuItem *mItem = pm->menu_item(i);

      // make sure this is not a separator
      if(! mItem->is_separator()) {
        needSep = (i < (n-1) && (pm->menu_item(i+1))->is_separator());
        itemAdded = TRUE;
        addSubMenu = 0;

        // check if this is a submenu item; if so, need to construct it first
        if(mItem->is_submenu()) {
//          printf("Running menu_create recursively\n");
          if(menu_create(mItem->sub_menu())) {
            // submenu was successfully created; get ID from current mainMenu
            addSubMenu = mainMenu->menuID;
//           printf("Returned from recursive menu create\n");
          }
          else
            // submenu was not created for some reason; do not add this entry
            itemAdded = FALSE;
        }  

        // now construct the string describing the menu item
        if(itemAdded) {
          sprintf(mbuf,"%s%%x%2d", mItem->name(), mItem->return_code());

          if(addSubMenu != 0)
            strcat(mbuf,"%m");
          if(needSep)
            strcat(mbuf,"%l");
          if(! mItem->showing())
            strcat(mbuf,"%d");
          if(mItem->checked())
            strcat(mbuf,"%B");
          else if(mItem->show_check())
            strcat(mbuf,"%b");
          MSGDEBUG(2,"OpenGLDisplayDevice: Menu string: "<< mbuf << " Total Added: " << totalAdded << sendmsg);
          fl_addtopup(pm->menuID, mbuf, addSubMenu);

          // modify the item with special characteristics
/*          if(! mItem->showing())
            fl_setpup_mode(pm->menuID, totalAdded + 1, FL_PUP_GREY);
          if(mItem->checked())
           {
  MSGDEBUG(2,"OpenGLDisplayDevice: Checked item "<< i << sendmsg);
            fl_setpup_mode(pm->menuID, totalAdded + 1, FL_PUP_CHECK);
            }
          else if(mItem->show_check())
	    {
  MSGDEBUG(2,"OpenGLDisplayDevice: Show_check item "<< i << sendmsg);
            fl_setpup_mode(pm->menuID, totalAdded + 1, FL_PUP_BOX);
            }*/
	  
          // indicate we have one more item
          totalAdded++;
        }
      }
    }

    // set the current menu to the given one
    if(totalAdded > 0) {
      mainMenu = pm;
      return TRUE;
    } else {
      fl_freepup(pm->menuID);
      pm->menuExists = FALSE;
      return FALSE;
    }

  } else {
    pm->menuExists = FALSE;
    mainMenu = NULL;
    return FALSE;
  }
#else
  return DisplayDevice::menu_create(pm);
#endif
}


// activate a previously-created menu.  If the windowing system has no
// current menu, or cannot do this operation, returns (-1); otherwise,
// returns the return code for the select (-1 if none selected).
int OpenGLDisplayDevice::menu_activate(void) {
#ifdef VMDFORMS_NOTYET
  MSGDEBUG(2,"OpenGLDisplayDevice: Reactivating menu ..." << sendmsg);
  if(mainMenu && mainMenu->menuExists) {
    // do the menu, get the result
    return fl_dopup(mainMenu->menuID);
  }

  // if here, could not do the menu selection
  return (-1);
#else
  return DisplayDevice::menu_activate();
#endif
}


// delete the given menu from the display.
// If no argument is given, deletes the current menu (if any).
// Returns success.
int OpenGLDisplayDevice::menu_delete(PopupMenu *pm) {
#ifdef VMDFORMS_NOTYET
  // check which menu to delete
  if(!pm || pm == mainMenu) {
    if(!mainMenu)
      return FALSE;
    pm = mainMenu;
    mainMenu = NULL;
  }

  // delete this menu
  if(pm->menuExists) {
    fl_freepup(pm->menuID);
    pm->menuExists = FALSE;
  }

  // go through the menu and delete all submenus first
  int n = pm->items();
  for(int i=0; i < n; i++) {
    PopupMenuItem *mItem = pm->menu_item(i);
    if(mItem->is_submenu())
      menu_delete(mItem->sub_menu());
  }

  // done deleting menu
  return TRUE;
#else
  return DisplayDevice::menu_delete(pm);
#endif
}


//
// get the current state of the device's pointer (i.e. cursor if it has one)
//

// abs pos of cursor from lower-left corner of display
int OpenGLDisplayDevice::x(void) {
  Window rw, cw;
  int rx, ry, wx, wy;
  unsigned int keymask;

  // get pointer info
  XQueryPointer(dpy, windowID, &rw, &cw, &rx, &ry, &wx, &wy, &keymask);

  // return value
  return rx;
}


// same, for y direction
int OpenGLDisplayDevice::y(void) {
  Window rw, cw;
  int rx, ry, wx, wy;
  unsigned int keymask;

  // get pointer info
  XQueryPointer(dpy, windowID, &rw, &cw, &rx, &ry, &wx, &wy, &keymask);

  // return value
  // return value ... must subtract position from total size since
  // X is opposite to GL in sizing the screen
  return screenY - ry;
}


// whether a button is currently pressed
int OpenGLDisplayDevice::button_down(int b) {
  int retval = FALSE;

  if(b == B_LEFT || b == B2_LEFT || b == B_MIDDLE || b == B2_MIDDLE ||
     b == B_RIGHT || b == B2_RIGHT) {
    Window rw, cw;
    int rx, ry, wx, wy;
    unsigned int keymask;

    // get pointer info
    XQueryPointer(dpy, windowID, &rw, &cw, &rx, &ry, &wx, &wy, &keymask);

    if(b == B_LEFT || b == B2_LEFT)
      retval = ((keymask & Button1Mask) != 0);
    else if(b == B_MIDDLE || b == B2_MIDDLE)
      retval = ((keymask & Button2Mask) != 0);
    else
      retval = ((keymask & Button3Mask) != 0);
  }

  // right now, this does not do any checking for function keys or ESC
  return retval;
}


// return the current state of the shift, control, and alt keys
int OpenGLDisplayDevice::shift_state(void) {
  int retval = 0;

  // get pointer info
  Window rw, cw;
  int rx, ry, wx, wy;
  unsigned int keymask;
  XQueryPointer(dpy, windowID, &rw, &cw, &rx, &ry, &wx, &wy, &keymask);

  // determine state of keys, and OR results together
  if ((keymask & ShiftMask) != 0)
    retval |= SHIFT;

  if ((keymask & ControlMask) != 0)
    retval |= CONTROL;

  if ((keymask & Mod1Mask) != 0)
    retval |= ALT;

  // return the result
  return retval;
}


// allow the user to define a special shape for the cursor ... the data
// is assumed to consist of a single-dim array of 32 unsigned shorts; the
// first 16 define a 16x16-bit (row-major) pattern, the last 16 are a
// "second layer" which may be drawn in a different color
// args: which shape (>0, with 0 the "default" case which cannot be changed)
// the bitmap data, and the "hot spot" from the lower-left corner
void OpenGLDisplayDevice::change_cursor_shape(int, unsigned short *,
					      int, int) {
  // OK, anyone who can get this going has my appreciation ...
  // for now, we're just using default definitions from the cursor font
}


// set the Nth cursor shape as the current one.  If no arg given, the
// default shape (n=0) is used.
void OpenGLDisplayDevice::set_cursor(int n) {
  if(n >= 0 && n < 5)
    XDefineCursor(dpy, windowID, cursor[n]);
}


//
// event handling routines
//

// queue the standard events (need only be called once ... but this is
// not done automatically by the window because it may not be necessary or
// even wanted)
void OpenGLDisplayDevice::queue_events(void) {
  XSelectInput(dpy, windowID,
	       KeyPressMask | ButtonPressMask | ButtonReleaseMask |
	       StructureNotifyMask | ExposureMask);
}


// test if there is an event ready
int OpenGLDisplayDevice::test_events(void) {
  XEvent xev;

  // make sure there is an event to check ...
#ifdef VMDFORMS
  if(!fl_XEventsQueued(QueuedAlready))
    return WIN_NOEVENT;
#else
  if(!XPending(dpy))
    return WIN_NOEVENT;
#endif

  // ok, there is a pending event.  Check what it is and return info about it
#ifdef VMDFORMS
  fl_XNextEvent(&xev);
  fl_XPutBackEvent(&xev);  // put it right back again, we're just checking
#else
  XPeekEvent(dpy, &xev);
#endif

  // find what kind of event it was
  switch(xev.type) {
  case ReparentNotify:
  case MapNotify:
  case Expose:
  case ConfigureNotify: return WIN_REDRAW;
  case KeyPress:	return WIN_KEYBD;
  case ButtonPress:
  case ButtonRelease:
    {
      unsigned int button = xev.xbutton.button;
      if(button == Button1)
        return WIN_LEFT;
      if(button == Button2)
        return WIN_MIDDLE;
      if(button == Button3)
        return WIN_RIGHT;
      break;
    }
  }

  // if here, no proper event found ...
#ifdef VMDFORMS
  fl_print_xevent_name("Unknown event seen in OpenGLDisplayDevice:", &xev);
#endif
  return WIN_NOEVENT;
}


// read the next event ... returns an event type (one of the above ones),
// and a value.  Returns success, and sets arguments.
int OpenGLDisplayDevice::read_event(long &retdev, long &retval) {
  XEvent xev;
  char keybuf[10];
  int keybuflen = 9;
  KeySym keysym;
  XComposeStatus comp;

  // make sure there is an event to check, and if so, get it
#ifdef VMDFORMS
  if(!fl_XEventsQueued(QueuedAlready))
    return FALSE;
  fl_XNextEvent(&xev);
#else
  if(!XPending(dpy))
    return FALSE;
  XNextEvent(dpy, &xev);
#endif

  // find what kind of event it was
  switch(xev.type) {
  case Expose:
  case ConfigureNotify:
  case ReparentNotify:
  case MapNotify:
    {
      retdev = WIN_REDRAW;
      break;
    }
  case KeyPress:
    {
      int k = XLookupString(&(xev.xkey), keybuf, keybuflen,  &keysym, &comp);
      if(k > 0 && *keybuf != '\0') {
        retdev = WIN_KEYBD;
        retval = *keybuf;
      } else
        retdev = WIN_NOEVENT;
      break;
    }
  case ButtonPress:
  case ButtonRelease:
    {
      unsigned int button = xev.xbutton.button;
      retval = (xev.type == ButtonPress);
      if(button == Button1)
        retdev = WIN_LEFT;
      else if(button == Button2)
        retdev = WIN_MIDDLE;
      else if(button == Button3)
        retdev = WIN_RIGHT;
      else
        retdev = WIN_NOEVENT;
      break;
    }
  default:
    // if here, no proper event found ...
#ifdef VMDFORMS
    fl_print_xevent_name("Unknown event recvd in OpenGLDisplayDevice:", &xev);
#endif
    return FALSE;
  }

  return TRUE;
}


//
// virtual routines for preparing to draw, drawing, and finishing drawing
//

// reshape the display after a shape change
void OpenGLDisplayDevice::reshape(void) {

  MSGDEBUG(2,"OpenGLDisplayDevice: reshaping display." << sendmsg);

  // get and store size of window
  XWindowAttributes xwa;
  Window childwin;                  // not used, just needed for X call
  int rx, ry;

  XGetWindowAttributes(dpy, windowID, &xwa);
  XTranslateCoordinates(dpy, windowID, rootWindowID, -xwa.border_width,
			-xwa.border_width, &rx, &ry, &childwin);

  xSize = xwa.width;
  ySize = xwa.height;
  xOrig = rx;
  yOrig = screenY - ry - ySize;

  /*
  msgWarn << "Reshaping display:";
  msgWarn << "\n  size = " << xSize << ", " << ySize;
  msgWarn << "\n  orig = " << xOrig << ", " << yOrig << sendmsg;
  */

  if(inStereo == OPENGL_STEREO_SIDE || inStereo == OPENGL_STEREO_CROSSED) {
    set_screen_pos(0.5 * (float)xSize / (float)ySize);
  } else {
    set_screen_pos((float)xSize / (float)ySize);
  }
}

// clear the display
void OpenGLDisplayDevice::clear(void) {
  
  MSGDEBUG(3, "OpenGLDisplayDevice: clearing display." << sendmsg);

  // set viewport properly
  if(inStereo == OPENGL_STEREO_SIDE || inStereo == OPENGL_STEREO_CROSSED) {
    glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize);

  } else if(inStereo == OPENGL_STEREO_CRYSTAL) {
    // for now, just split the window horizontally until we see how to do this
    glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize);

//    viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
//    if(getgdesc(GD_STEREO_IN_WINDOW) != 1 || getgdesc(GD_XPMAX) > 1200)
  }

  // set fog color and range
  GLfloat fogcol[4];
  fogcol[0] = (GLfloat)(backColor[0]);
  fogcol[1] = (GLfloat)(backColor[1]);
  fogcol[2] = (GLfloat)(backColor[2]);
  fogcol[3] = 1.0;
  glFogfv(GL_FOG_COLOR, fogcol);
  glFogf(GL_FOG_START, (GLfloat)nearClip);
  glFogf(GL_FOG_END, (GLfloat)farClip);

  // set clear colors, and clear the buffers
  glClearColor((GLclampf)(backColor[0]),
	       (GLclampf)(backColor[1]),
	       (GLclampf)(backColor[2]), 1.0);
  glClear(GL_COLOR_BUFFER_BIT);
  glClear(GL_DEPTH_BUFFER_BIT);
}


// prepare to draw a 2D image
int OpenGLDisplayDevice::prepare2D(int do_clear) {

  MSGDEBUG(3, "OpenGLDisplayDevice: preparing to draw 2D." << sendmsg);

  Dim = 2;
  if(do_clear)
    clear();
  else
    glClear(GL_DEPTH_BUFFER_BIT);

  // this should return true for normal (non file-based) renderers
  return TRUE;
}

// prepare to draw a 3D image
int OpenGLDisplayDevice::prepare3D(int do_clear) {

  MSGDEBUG(3, "OpenGLDisplayDevice: preparing to draw 3D." << sendmsg);

  Dim = 3;
  if(do_clear)
    clear();
  else
    glClear(GL_DEPTH_BUFFER_BIT);

  // this should return true for normal (non file-based) renderers
  return TRUE;
}

// set up for normal (non-stereo) drawing.  Sets the viewport and perspective.
void OpenGLDisplayDevice::normal(void) {
  glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize);
  set_persp();
}

// set up for drawing the left eye image.  Assume always the left eye is
// drawn first (so no zclear is needed before it)
void OpenGLDisplayDevice::left(void) {

  if(inStereo == OPENGL_STEREO_SIDE || inStereo == OPENGL_STEREO_CROSSED) {
    glViewport(0, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
    if(inStereo == OPENGL_STEREO_SIDE)
      set_persp(RIGHTEYE);	// backwards since we're crossing our eyes
    else
      set_persp(LEFTEYE);	// these two should be switched eventually

  } else if(inStereo == OPENGL_STEREO_LEFT) {
    // set the eye perspective properly
    set_persp(LEFTEYE);

  } else if(inStereo == OPENGL_STEREO_RIGHT) {
    // set the eye perspective properly
    set_persp(RIGHTEYE);

  } else if(inStereo == OPENGL_STEREO_CRYSTAL) {
    // for now, just split the window horizontally until we see how to do this
    if(!stereoInWindow) {
      glViewport(0, (GLint)ySize / 2, (GLsizei)xSize, (GLsizei)ySize / 2);
      //viewport(0,getgdesc(GD_XPMAX),YOFFSET,YOFFSET + YMAXSTEREO);
    } else {
      glDrawBuffer(GL_BACK_LEFT);
    }
    set_persp(LEFTEYE);

  } else {			
    // left called even though we're non-stereo
    normal();
  }
}


// set up for drawing the right eye image.  Assume always the right eye is
// drawn last (so a zclear IS needed before it)
void OpenGLDisplayDevice::right(void) {

  if(inStereo == OPENGL_STEREO_SIDE || inStereo == OPENGL_STEREO_CROSSED) {
    glViewport((GLsizei)xSize/2, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
    if(inStereo == OPENGL_STEREO_SIDE)
      set_persp(LEFTEYE);	// backwards since we're crossing our eyes
    else
      set_persp(RIGHTEYE);	// these two should be switched eventually

  } else if(inStereo == OPENGL_STEREO_LEFT || inStereo == OPENGL_STEREO_RIGHT) {
    // no need to do anything, already done in call to left

  } else if(inStereo == OPENGL_STEREO_CRYSTAL) {
    // for now, just split the window horizontally until we see how to do this
    if(!stereoInWindow) {
      glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize / 2);
      //viewport(0,getgdesc(GD_XPMAX),YOFFSET,YOFFSET + YMAXSTEREO);
    } else {
      glClear(GL_DEPTH_BUFFER_BIT);
      glDrawBuffer(GL_BACK_RIGHT);
    }
    set_persp(RIGHTEYE);

  } else {
    // right called even though we're non-stereo
    normal();
  }
}

// update after drawing
void OpenGLDisplayDevice::update(int do_update) {

  MSGDEBUG(3, "OpenGLDisplayDevice: updating after drawing." << sendmsg);

  if(do_update)
    glXSwapBuffers(dpy, windowID);
  glDrawBuffer(GL_BACK);
}


//
// stereo
//

// change to a different stereo mode (0 means 'off')
void OpenGLDisplayDevice::set_stereo_mode(int newMode) {

  // do nothing if current mode is same as specified mode
  if(inStereo == newMode)
    return;

  if(newMode == OPENGL_STEREO_OFF) {
    
    // turn off crystal-eyes if necessary
    if(inStereo == OPENGL_STEREO_CRYSTAL) {

      // use offstereo program if large-screen (overlapping viewports)
      if(!stereoInWindow) {
        system("offstereo");
//        // restore the window to the size it was before being set to stereo
//        winposition(xOrig, xOrig + xSize, yOrig, yOrig + ySize);
//        winconstraints();
      }
    }

    inStereo = newMode;
    reshape();
    normal();
    clear();
    update();

    MSGDEBUG(2,"OpenGLDisplayDevice stereo turned off." << sendmsg);

  } else if(newMode != OPENGL_STEREO_CRYSTAL) {

    set_stereo_mode(OPENGL_STEREO_OFF);
    inStereo = newMode;
    reshape();
    normal();
    MSGDEBUG(2,"OpenGLDisplayDevice stereo mode set." << sendmsg);

  } else if(newMode == OPENGL_STEREO_CRYSTAL) {

    set_stereo_mode(OPENGL_STEREO_OFF);
    inStereo = newMode;

    // use onstereo program if large-screen (overlapping viewports)
    if(!stereoInWindow) {
      system("onstereo");
//      // resize the window to be the whole screen
//      prefposition(0,getgdesc(GD_XPMAX),0,getgdesc(GD_YPMAX));
//      winconstraints();
//      reshapeviewport();
//      viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
//      set_screen_pos(0.5*(float)getgdesc(GD_XPMAX)/(float)YMAXSTEREO);
      clear();
      update();
    }

    MSGDEBUG(2,"OpenGLDisplayDevice set to crystal-eyes stereo." << sendmsg);
  }
}


