/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: PSDisplayDevice.C,v $
 *	$Author: ulrich $	$Locker:  $		$State: Exp $
 *	$Revision: 1.10 $	$Date: 1997/03/21 18:45:55 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 *  We can render to Postscript!   Yippee!!
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "PopupMenu.h"
#include "PSDisplayDevice.h"
#include "Stack.h"
#include "utilities.h"
#include "DispCmds.h"
#include "Inform.h"


#define XOffset 306
#define YOffset 396
#define psHor 792
#define psVer 792
typedef char *char_ptr;




// constructor

PSDisplayDevice::PSDisplayDevice(void) : 
  FileRenderer ("PostScript","vmd.ps","ghostview %s &") {
    dataBlock = NULL;
}



void PSDisplayDevice::output_to_file (int numPts, float PSPoints[][3], 
				 char *objectType, int doTransform) {
/* This is the routine which does most of the actual writing to the
   PostScript file (save for the header, footer, and background coloring.
   Object routines set up data to be passed off to this procedure, which
   is a "generic" routine for handling the display of many types of
   polygons. */
int i;
float normal[3];	// Normal to the polygon surface to draw
float colorShade[3];	// Using normal and position of the light, one
			// can determine an appropriate colorShade.
float normalScale;	// Just a utility variable in calculating normal
float dotProduct;	// Again a utitlity variable in calculating
			// the colorShade.  

float **pointList;

if (my_detail_level == DETAIL_NONE) return;
pointList = (float **) (malloc (numPts * sizeof(float *)));
for (i = 0; i < numPts; i++)
  pointList[i] = (float *)(malloc(3*sizeof(float)));


			// Everything but spheres must be transformed
			// into screen coordinates.  The doTransform
			// flag indicates whether or not this object
			// being drawn needs to be transformed (i.e has
			// nothing to do with spheres) or can remain
			// unaltered (i.e. comes from sphere data)
if (doTransform) 
  for (i = 0; i < numPts; i++) 
    (transMat.top()).multpoint3d(PSPoints[i],pointList[i]);
else
  for (i = 0; i < numPts; i++) {
    pointList[i][0] = PSPoints[i][0];
    pointList[i][1] = PSPoints[i][1];
    pointList[i][2] = PSPoints[i][2];
  }

			// If we are dealing with a polygon and lighting
			// is on, then calcuate the appropriate color
			// using the normal to the polygon and the
			// normalized position of the light.
dotProduct = 1;
if ((numPts >2) && (my_detail_level == DETAIL_FULL)) {
  normal[0] = pointList[0][1] * (pointList[1][2]-pointList[2][2]) +
	      pointList[1][1] * (pointList[2][2]-pointList[0][2]) +
	      pointList[2][1] * (pointList[0][2]-pointList[1][2]);

  normal[1] = pointList[0][2] * (pointList[1][0]-pointList[2][0]) +
	      pointList[1][2] * (pointList[2][0]-pointList[0][0]) +
	      pointList[2][2] * (pointList[0][0]-pointList[1][0]);

  normal[2] = pointList[0][0] * (pointList[1][1]-pointList[2][1]) +
   	      pointList[1][0] * (pointList[2][1]-pointList[0][1]) +
	      pointList[2][0] * (pointList[0][1]-pointList[1][1]);
   
  normalScale = normal[0]*normal[0]+normal[1]*normal[1]+normal[2]*normal[2];
  normalScale = sqrt(normalScale);
  normal[0] /= normalScale; normal[1] /= normalScale; normal[2] /= normalScale;
  dotProduct = normal[0]*normLight[0] + normal[1]*normLight[1] 
		+ normal[2]*normLight[2];
  // Fix this hack below
  if (normalScale < .0000001) dotProduct = 1;
  if (dotProduct < 0) dotProduct = -dotProduct;
} 

colorShade[0] = col_ptr[0]*dotProduct;
colorShade[1] = col_ptr[1]*dotProduct;
colorShade[2] = col_ptr[2]*dotProduct;


if ((objectType[0] == 'p') && (objectType[1] == 'o')) 
  fprintf (outfile, "%d %d %3.2f %3.2f %3.2f point\n",
	   (int)(pointList[0][0]*XScaleFact+XOffset),
	   (int)(pointList[0][1]*YScaleFact+YOffset),
	   colorShade[0],colorShade[1],colorShade[2]);
else if ((objectType[0] == 'l') && (objectType[1] == 'i'))
  fprintf (outfile, "%d %d %d %d %3.2f %3.2f %3.2f line\n",
	   (int)(pointList[1][0]*XScaleFact+XOffset),
	   (int)(pointList[1][1]*YScaleFact+YOffset),
	   (int)(pointList[0][0]*XScaleFact+XOffset),
	   (int)(pointList[0][1]*YScaleFact+YOffset),
	   colorShade[0],colorShade[1],colorShade[2]);
else if ((objectType[0] == 't') && (objectType[1] == 'e')) {
  fprintf (outfile, "%3.2f %3.2f %3.2f setrgbcolor\n", colorShade[0],
	   colorShade[1], colorShade[2]);
  fprintf (outfile, "%d %d moveto\n", 
	   (int)(pointList[0][0]*XScaleFact+XOffset),
	   (int)(pointList[0][1]*YScaleFact+YOffset));
  fprintf (outfile, "/Helvetica findfont 12 scalefont setfont\n");
}
else if ((objectType[0] == 's') && (objectType[1] == 'q'))  {
  fprintf (outfile, "%d %d %d %d %d %d %d %d %3.2f %3.2f %3.2f",
	   (int)(pointList[3][0]*XScaleFact+XOffset),
	   (int)(pointList[3][1]*YScaleFact+YOffset),
	   (int)(pointList[2][0]*XScaleFact+XOffset),
	   (int)(pointList[2][1]*YScaleFact+YOffset),
	   (int)(pointList[1][0]*XScaleFact+XOffset),
	   (int)(pointList[1][1]*YScaleFact+YOffset),
	   (int)(pointList[0][0]*XScaleFact+XOffset),
	   (int)(pointList[0][1]*YScaleFact+YOffset),
	   colorShade[0],colorShade[1],colorShade[2]);
  if (my_detail_level > DETAIL_WIREFRAME) 
     fprintf (outfile, " square\n");
  else fprintf (outfile, " square2\n");
}
else if ((objectType[0] == 't') && (objectType[1] == 'r')) {
  fprintf (outfile, "%d %d %d %d %d %d %3.2f %3.2f %3.2f",
	   (int)(pointList[2][0]*XScaleFact+XOffset),
	   (int)(pointList[2][1]*YScaleFact+YOffset),
	   (int)(pointList[1][0]*XScaleFact+XOffset),
	   (int)(pointList[1][1]*YScaleFact+YOffset),
	   (int)(pointList[0][0]*XScaleFact+XOffset),
	   (int)(pointList[0][1]*YScaleFact+YOffset),
	   colorShade[0],colorShade[1],colorShade[2]);
  if (my_detail_level > DETAIL_WIREFRAME)
     fprintf (outfile, " triangle\n");
  else fprintf (outfile, " triangle2\n");
}

}

void PSDisplayDevice::write_header() {
   fprintf (outfile, "%%!PS-Adobe-1.0\n");
   fprintf (outfile, "%%%%DocumentFonts:Helvetica\n");
   fprintf (outfile, "%%%%Title:vmd.ps\n");
   fprintf (outfile, "%%%%Creator:VMD -- Visual Molecular Dynamics\n");
   fprintf (outfile, "%%%%CreationDate:\n");
   fprintf (outfile, "%%%%Pages:1\n");
   fprintf (outfile, "%%%%BoundingBox:0 0 612 792\n");
   fprintf (outfile, "%%%%EndComments\n");
}

void PSDisplayDevice::comment (char *s)
{
  fprintf (outfile, "%%%% %s\n", s);
}



void PSDisplayDevice::PSsquare (float *p1, float *p2, float *p3, float *p4) {
   int i;
   float PSPoints[4][3];

   for (i = 0; i < 3; i++) {
     PSPoints[0][i] = p1[i];
     PSPoints[1][i] = p2[i];
     PSPoints[2][i] = p3[i];
     PSPoints[3][i] = p4[i];
   }

   switch (my_detail_level) {
     case DETAIL_FULL:
     case DETAIL_FLAT:
     case DETAIL_WIREFRAME:
       output_to_file (4, PSPoints, "square", 1);
       break;
     case DETAIL_POINTS:
       for (i = 0; i < 4; i++)
       output_to_file (1, &(PSPoints[i]), "point", 1);
       break;
   }
}





void PSDisplayDevice::PStext(char *text, float x, float y, float z) {
   /* Outputs text to PostScript file  */
   float PSPoints[1][3];
   PSPoints[0][0] = x; PSPoints[0][1] = y; PSPoints[0][2] = z;
   output_to_file (1, PSPoints, "text", 1);
   if (my_detail_level > DETAIL_NONE)
      fprintf (outfile, "(%s) show\n",text);
} 




// **********************************************
// This code derived from Jon Leech 

void PSDisplayDevice::PSsphere(float *pos) {
#define XPLUS { 1, 0, 0 } /*  X */
#define XMIN  {-1, 0, 0 } /* -X */
#define YPLUS { 0, 1, 0 } /*  Y */
#define YMIN  { 0,-1, 0 } /* -Y */
#define ZPLUS { 0, 0, 1 } /*  Z */
#define ZMIN  { 0, 0,-1 } /* -Z */

    typedef struct {
	   float x,y,z;
    } Point;

    typedef struct {
	    Point pt[3];
	    float area;
    } Triangle;

    typedef struct {
	    int  npoly;
 	    Triangle *poly;
    } object;

    Triangle octahedron[] = {
	    { { XPLUS, ZPLUS, YPLUS }, 0.0 },
	    { { YPLUS, ZPLUS, XMIN  }, 0.0 },
	    { { XMIN,  ZPLUS, YMIN  }, 0.0 },
	    { { YMIN,  ZPLUS, XPLUS }, 0.0 },
	    { { XPLUS, YPLUS, ZMIN }, 0.0 },
	    { { YPLUS, XMIN,  ZMIN }, 0.0 },
	    { { XMIN,  YMIN,  ZMIN }, 0.0 },
	    { { YMIN,  XPLUS, ZMIN }, 0.0 },
    };



    /* A unit octahedron */
    object oct = {
    	    sizeof(octahedron) / sizeof(octahedron[0]),
	    &octahedron[0]
    };


    object *Old = &oct,  /* Default is octahedron */
	   *New;
    int     i,j,k,
	    level;              /* Current subdivision level */


    float cen[3];
    (transMat.top()).multpoint3d(pos, cen);
    float rad = scale_radius(pos[3]);


    /* Subdivide each starting triangle (sphereRes - 1) times */
    /* Actually, I subdivide it only half as many times as GL does,
       to avoid enormous file sizes. */
    for (level = 1; level < (int)(sphereRes/2); level++) {
        /* Allocate a new object */
        New = (object *)malloc(sizeof(object));
        if (New == NULL) {
            fprintf(stderr, "Out of memory on subdivision level %d\n",
               level);
            exit(1);
        }
        New->npoly = Old->npoly * 4;

        /* Allocate 4* the number of points in the current approximation */
        New->poly  = (Triangle *)malloc(New->npoly * sizeof(Triangle));
        if (New->poly == NULL) {
            fprintf(stderr, "Out of memory on subdivision level %d\n",
                level);
            exit(1);
        }

        /* Subdivide each triangle in the old approximation and normalize
         * the new points thus generated to lie on the surface of the unit
         *  sphere.
         * Each input triangle with vertices labelled [0,1,2] as shown
         *  below will be turned into four new triangles:
         *
         *                      Make new points
         *                          a = (0+2)/2
         *                          b = (0+1)/2
         *                          c = (1+2)/2
         *        1
         *       /\             Normalize a, b, c
         *      /  \
         *    b/____\ c         Construct new triangles
         *    /\    /\              [0,b,a]
         *   /  \  /  \             [b,1,c]
         *  /____\/____\            [a,b,c]
         * 0      a     2           [a,c,2]
         */
        for (i = 0; i < Old->npoly; i++) {
            Triangle
                 *Oldt = &Old->poly[i],
                 *Newt = &New->poly[i*4];
            Point a, b, c, m1,m2,m3;
	    float mag;
	    /* Calculate midpoints and normalize. */
            a = Oldt->pt[0];
	    b = Oldt->pt[1];
	    c = Oldt->pt[2];
	    m1.x = (a.x + c.x) * 0.5;
	    m2.x = (a.x + b.x) * 0.5;
	    m3.x = (b.x + c.x) * 0.5;

	    m1.y = (a.y + c.y) * 0.5;
	    m2.y = (a.y + b.y) * 0.5;
	    m3.y = (b.y + c.y) * 0.5;

	    m1.z = (a.z + c.z) * 0.5;
	    m2.z = (a.z + b.z) * 0.5;
	    m3.z = (b.z + c.z) * 0.5;

	    mag = m1.x * m1.x + m1.y * m1.y + m1.z * m1.z;
	    if (mag != 0.0) {
		mag = 1.0 / sqrt(mag);
		a.x = m1.x * mag;
		a.y = m1.y * mag;
		a.z = m1.z * mag;
            }

	    mag = m2.x * m2.x + m2.y * m2.y + m2.z * m2.z;
	    if (mag != 0.0) {
	       mag = 1.0 / sqrt(mag);
	       b.x = m2.x * mag;
	       b.y = m2.y * mag;
	       b.z = m2.z * mag;
            }

	    mag = m3.x * m3.x + m3.y * m3.y + m3.z * m3.z;
	    if (mag != 0.0) {
	       mag = 1.0 / sqrt(mag);
	       c.x = m3.x * mag;
	       c.y = m3.y * mag;
	       c.z = m3.z * mag;
            }
	    /* End calculation of midpoints and normalization */


            Newt->pt[0] = Oldt->pt[0];
            Newt->pt[1] = b;
            Newt->pt[2] = a;
            Newt++;

            Newt->pt[0] = b;
            Newt->pt[1] = Oldt->pt[1];
            Newt->pt[2] = c;
            Newt++;

            Newt->pt[0] = a;
            Newt->pt[1] = b;
            Newt->pt[2] = c;
            Newt++;

            Newt->pt[0] = a;
            Newt->pt[1] = c;
            Newt->pt[2] = Oldt->pt[2];
        }
        if (level > 1) {
            free(Old->poly);
            free(Old);
        }

        /* Continue subdividing new triangles */
        Old = New;
    }

    /* Print out resulting approximation */
    /* In order to properly handle interactions between
       spheres and other objects, I will have to sort the
       individual triangles of each sphere.  The code below
       adds all these triangles to the list of objects which
       must be sorted. Notice that at this time, the coordinates
       of the triangles are in screen space, not molecule space.
       In order to be consistent with how other distances from
       the eye are calculated, I will have to put these back
       into molecule space.  This is what the matrix stuff 
       below is doing. */
    Triangle *t;
    float p[3][3];
    char *dataPts;
    DepthSortObject2 depthSortObject;
    Matrix4 *utilMatrix;
    depthSortObject.matrix = transMat.top();
    utilMatrix = (Matrix4 *) (&(depthSortObject.matrix));
    utilMatrix = &((*utilMatrix).inverse());
    float molCoord[3];
    /* Assign some values that will not change for any triangles. */
    depthSortObject.color = color_index;
    depthSortObject.moredata = moredata;
    depthSortObject.textpos = PStext_position;
    depthSortObject.material = material_active;
    depthSortObject.num = 3;

    /* Then go through the triangles and add them to the
       list to be sorted. */
    for (i = 0; i < Old->npoly; i++) {
       t = &Old->poly[i];
       for (j = 0; j < 3; j++) {
	  p[j][0] = (t->pt[j].x) * rad + cen[0];
	  p[j][1] = (t->pt[j].y) * rad + cen[1];
	  p[j][2] = (t->pt[j].z) * rad + cen[2];
       }
       dataPts = new char[9*sizeof(float)+2*sizeof(int)];
       ((int *)dataPts)[0] = DSPHERE;
       ((int *)dataPts)[1] = 9*sizeof(float)+2*sizeof(int);
       float *triPts = (float *)(dataPts + 2*sizeof(int));
       for (k = 0; k < 3; k++) {
	 triPts[0+k] = p[0][k];
	 triPts[3+k] = p[1][k];
	 triPts[6+k] = p[2][k];
       }
       depthSortObject.data = new char_ptr[depthSortObject.num];
       depthSortObject.data[1] = sphere_res;
       depthSortObject.data[2] = sphere_mode;
       depthSortObject.data[0] = dataPts;
       (*utilMatrix).multpoint3d(&(triPts[0]),molCoord);
       depthSortObject.distToEye=DTOEYE(molCoord[0],molCoord[1],molCoord[2]);
       DepthSortList.append(depthSortObject);
    }  
}

 

void PSDisplayDevice::PStriangleScaled (float *p1, float *p2, float *p3) {
   /* This is used to display triangles making up spheres.  They are
      generated in screen space, so when I pass off to output_to_file
      they do not need to be rescaled.  This is accomplished by passing
      the 0 as the last parameter to output_to_file. */

   float PSPoints[3][3];
   int i;
   for (i = 0; i < 3; i++) {
     PSPoints[0][i] = p1[i];
     PSPoints[1][i] = p2[i];
     PSPoints[2][i] = p3[i];
   }

   switch (my_detail_level) {
     case DETAIL_FULL:
     case DETAIL_FLAT:
     case DETAIL_WIREFRAME:
       output_to_file (3, PSPoints, "triangle", 0);
       break;
     case DETAIL_POINTS:
       for (i = 0; i < 3; i++)
         output_to_file (1, &(PSPoints[i]), "point", 0);
       break;
   }
}


void PSDisplayDevice::PStriangle (float *p1, float *p2, float *p3) {
   /* This is used to display triangles other than those generated
      by the sphere drawing routine. */

    float PSPoints[3][3];
    int i;
    for (i = 0; i < 3; i++) {
      PSPoints[0][i] = p1[i];
      PSPoints[1][i] = p2[i];
      PSPoints[2][i] = p3[i];
    }

    switch (my_detail_level) {
    case DETAIL_FULL:
    case DETAIL_FLAT:
    case DETAIL_WIREFRAME:
      output_to_file (3, PSPoints, "triangle", 1);
      break;
    case DETAIL_POINTS:
      for (i = 0; i < 3; i++)
        output_to_file (1, &(PSPoints[i]), "point", 1);
      break;
    }
}



void PSDisplayDevice::PSline (float *start, float *end) {
    /* Simply draws a line between two given coordinates*/
    int i;
    float PSPoints[2][3];
 
    for (i = 0; i < 3; i++) {
      PSPoints[0][i] = start[i];
      PSPoints[1][i] = end[i];
    }

    switch (my_detail_level) {
    case DETAIL_FULL:
    case DETAIL_FLAT:
    case DETAIL_WIREFRAME:
      output_to_file (2, PSPoints, "line", 1);
      break;
    case DETAIL_POINTS:
      for (i = 0; i < 2; i++)
        output_to_file (1, &(PSPoints[i]), "point", 1);
      break;
    }
}

void PSDisplayDevice::PScylinder(int num, float *edges, int filled)
{
 if (num < 2)
   return;

 int n = num;
 float *start = edges;

 float t1[3];


 typedef struct edgeInfo {
   float distToEye;
   float *edgeStart;
   int lastEdgeFlag;
 } EdgeInfo;

 float *index;
 float *ptr1;
 float *ptr2;
 int i, j;

 EdgeInfo *edgeinfo;
 EdgeInfo temp;
 edgeinfo = (EdgeInfo *)(malloc (num * sizeof(EdgeInfo)));
 index = edges+3;

 for (i = 0; i < num; i++) {
   edgeinfo[i].distToEye = DTOEYE (index[0], index[1], index[2]);
   edgeinfo[i].edgeStart = index-3;
   edgeinfo[i].lastEdgeFlag = 0;
   index += 9;
 }
 edgeinfo[num-1].lastEdgeFlag = 1;

 // Do some simple insertion sort here
 for (i = num -2; i>=0; i--) {
     j = i+1;
     temp = edgeinfo[i];
     while ((temp.distToEye < edgeinfo[j].distToEye) && (j < num)) {
       edgeinfo[j-1] = edgeinfo[j];
       j = j+1;
     }
     edgeinfo[j-1] = temp;
 }


 /* Now process the list in sorted order.  The physical order of
    the edges in the cmdList is important, so there is pointer
    arithmetic below which correctly "links" together vertices of
    the edge, including the 'last edge' which has vertices found
    at the front as well. */
 for (i = 0; i < num; i++) {
     ptr1 = edgeinfo[i].edgeStart;
     if (edgeinfo[i].lastEdgeFlag != 1)
	ptr2 = edgeinfo[i].edgeStart + 9;
     else
	ptr2 = start;
     PSsquare (ptr1+6, ptr1+3, ptr2+3, ptr2+6);
 }

 float axisTrans[3];  // axis transformed to screen coordinates
 float colorShade[3]; // color shade for current object

  // Finally, do the ends if necessary
  /* Right now these are printed to the file here in the cylinder
     procedure instead of being passed off to output_to_file.  I
     don't like this inconsistency and eventually this should be
     changed. */
  if (0) {
        float axis[3];   
	axis[0] = start[6] - start[3];
	axis[1] = start[7] - start[4];
	axis[2] = start[8] - start[5];
	(transMat.top()).multpoint3d(axis,axisTrans);
	normalize(axisTrans);
	/* Determine current normal and choose the corresponding
	   correct color. */
	currNorm[0] = axisTrans[0];
	currNorm[1] = axisTrans[1];
	currNorm[2] = axisTrans[2];
	float dot_product = currNorm[0]*normLight[0]+currNorm[1]*normLight[1]+
		            currNorm[2]*currNorm[2];
	colorShade[0] = col_ptr[0]*dot_product;
	colorShade[1] = col_ptr[1]*dot_product;
	colorShade[2] = col_ptr[2]*dot_product;
	fprintf (outfile, "%3.2f %3.2f %3.2f setrgbcolor\n",
		 colorShade[0],colorShade[1], colorShade[2]);
	n = num-1;
	edges = start+3;
	fprintf (outfile, "newpath\n");
	(transMat.top()).multpoint3d(edges, t1);
	fprintf (outfile, " %d %d moveto\n",(int)(t1[0]*XScaleFact+XOffset), 
		 (int)(t1[1]*YScaleFact+YOffset));
	edges += 9;
	while (--n >= 0) {
	  (transMat.top()).multpoint3d(edges, t1);
	  fprintf (outfile, " %d %d lineto\n",(int)(t1[0]*XScaleFact+XOffset), 
		   (int)(t1[1]*YScaleFact+YOffset));
	  edges += 9;
        }
	fprintf (outfile, "closepath\nfill\nstroke\n");
        
	n = num-1;
	edges = start+6;
        fprintf (outfile, "newpath\n");
	(transMat.top()).multpoint3d(edges, t1);
	fprintf (outfile, " %d %d moveto\n",(int)(t1[0]*XScaleFact+XOffset), 
		 (int)(t1[1]*YScaleFact+YOffset));
	edges += 9;
	while (--n >= 0) {
	  (transMat.top()).multpoint3d(edges, t1);
	  fprintf (outfile, " %d %d lineto\n",
		   (int)(t1[0]*XScaleFact+XOffset), 
		   (int)(t1[1]*YScaleFact+YOffset));
	  edges += 9;
        }
	fprintf (outfile, "closepath\nfill\nstroke\n");
   }
}

void PSDisplayDevice::PScone (float *start, float *end, int rod_res,
			    float rod_radius) {
/* Attempts to draw a cone to the PostScript file. */
register int i, h;
register float hresinc, rodrad, length, theta, m,n;
float axis[3], lenaxis[3], perp[3], perp2[3];

float norms1[4], norms2[4], norms3[4], vert1[4], vert2[4];
// need to do some preprocessing
lenaxis[0] = start[0] - end[0];
lenaxis[1] = start[1] - end[1];
lenaxis[2] = start[2] - end[2];

m = lenaxis[0]*lenaxis[0] + lenaxis[1]*lenaxis[1] + lenaxis[2]*lenaxis[2];
if (m <= 0.0)
  return;
length = sqrtf(m);
// Normalize the axis
axis[0] = lenaxis[0] / length;
axis[1] = lenaxis[1] / length;
axis[2] = lenaxis[2] / length;

// dot product with unit axis vector
if ((ABS(axis[0]) < ABS(axis[1])) &&
    (ABS(axis[0]) < ABS(axis[2])))
  i = 0;
else if ((ABS(axis[1]) < ABS(axis[2])))
  i = 1;
else
  i = 2;

perp[i] = 0;
perp[(i+1)%3] = axis[(i+2)%3];
perp[(i+2)%3] = -axis[(i+1)%3];
m = sqrtf(perp[0]*perp[0] + perp[1]*perp[1] + perp[2]*perp[2]);
if (m > 0.0) {
    perp[0] /= m;
    perp[1] /= m;
    perp[2] /= m;
}


  // get cross product; perp2 will be normalized automatically
  perp2[0] = axis[1]*perp[2] - axis[2]*perp[1];
  perp2[1] = axis[2]*perp[0] - axis[0]*perp[2];
  perp2[2] = axis[0]*perp[1] - axis[1]*perp[0];

  // do the polygons
  rodrad = rod_radius;
  hresinc = TWOPI / ((float)rod_res);
  theta = 0.0;

for (h= 0; h < rod_res; h++) {
 if (h == 0) {
   m =cos(theta);
   n =sin(theta);
   norms1[0] = m*perp[0] + n*perp2[0];
   norms1[1] = m*perp[1] + n*perp2[1];
   norms1[2] = m*perp[2] + n*perp2[2];
 } else {
   norms1[0] = norms2[0];
   norms1[1] = norms2[1];
   norms1[2] = norms2[2];
 }


 theta += hresinc;
 m = cos(theta);
 n = sin(theta);
 norms2[0] = m*perp[0] + n*perp2[0];
 norms2[1] = m*perp[1] + n*perp2[1];
 norms2[2] = m*perp[2] + n*perp2[2];

 norms3[0] = 0.5 * (norms1[0] + norms2[0]);
 norms3[1] = 0.5 * (norms1[1] + norms2[1]);
 norms3[2] = 0.5 * (norms1[2] + norms2[2]);

 if(h==0) {
   vert1[0] = start[0] + rodrad * norms1[0];
   vert1[1] = start[1] + rodrad * norms1[1];
   vert1[2] = start[2] + rodrad * norms1[2];
  } else {
    vert1[0] = vert2[0];
    vert1[1] = vert2[1];
    vert1[2] = vert2[2];
  }

  vert2[0] = start[0] + rodrad * norms2[0];
  vert2[1] = start[1] + rodrad * norms2[1];
  vert2[2] = start[2] + rodrad * norms2[2];

  switch (my_detail_level) {
  case DETAIL_FULL:
  case DETAIL_FLAT:
  case DETAIL_WIREFRAME:
  case DETAIL_POINTS:
  case DETAIL_NONE:
     PStriangle(&end[0], &vert2[0], &vert1[0]);
     break;
  }
}
}


void PSDisplayDevice::PSpoint(float *p1) {
  /* Draws a point to the file */
  float PSPoints[1][3];
  int i;
  for (i = 0; i < 3; i++)
     PSPoints[0][i] = p1[i];
  output_to_file (1, PSPoints, "point", 1);
}


void PSDisplayDevice::set_line_width (int w) {
  if (w > 0) 
    fprintf (outfile,"%f setlinewidth\n", (float)(w/2));
}

void PSDisplayDevice::set_line_style(int s) {
  lineStyle = s;
  if (s == ::SOLIDLINE)
      fprintf (outfile, "[] 0 setdash\n");
  else if (s == ::DASHEDLINE)
      fprintf (outfile, "[3] 0 setdash\n");
  else {
      fprintf (outfile, "[] 0 setdash\n");
      lineStyle = ::SOLIDLINE;
  }
} 



//****************** process_depth_sorted_list() ********************
// process_depth_sorted_list() runs through a depth sorted list of 
// commands, updating PostScript output appropriately in the process
//

void PSDisplayDevice::process_depth_sorted_list()
{
 int cnt, cnt2;
 DepthSortObject2 temp;

 // Push current transformation matrix
 super_push();
 for (cnt = 0; cnt < DepthSortList.num(); cnt++) {
   temp = DepthSortList.item(cnt);
   // if there was a color change for this object, change it
   if (temp.color != NULL)
     render_token(temp.color,temp.moredata);
   // if there was a material on/off change for this object, change it
   if (temp.material != NULL)
     render_token(temp.material,temp.moredata);
   // just always load the appropriate matrix for this object
   if (temp.textpos != NULL) {
      textX = ((float *)(temp.textpos))[0];
      textY = ((float *)(temp.textpos))[1];
      textZ = ((float *)(temp.textpos))[2];
   }
   super_load ((float *)(&(temp.matrix)));

   for (cnt2 = temp.num-1;cnt2 >=0;cnt2--)
      if (temp.data[cnt2] != NULL)
        render_token(temp.data[cnt2],temp.moredata);
 }
 fprintf (outfile,"showpage\n");
 close_file();
   
 // Restore transformation matrix
 super_pop();
 // Delete entries on transparency list. We are done with it for now
 // First delete the internal data that was allocated.
 for (cnt = 0; cnt< DepthSortList.num(); cnt++)
   delete DepthSortList.item(cnt).data;
 // Then delete the elements themselves.
 DepthSortList.remove (-1, -1); 
}



//
//*******************  the depth sort rendering routine *****************
//
// This scans the given command list until the end, doing the commands
// in the order they appear. It makes particular exceptions for
// transparent objects when back to front drawing is being performed.
// Much of this code is repetion from the original render routine


void PSDisplayDevice::render_depth_sort(void *cmdList) {

  char *cmdptr = (char *)cmdList;
  int tok, cmdsize;
  int depthSortObjectFound = 0;
  float center[3];                      // Point of object to compare to eye
  Matrix4 *utilMatrix;
  float *utilPtr; 		        // utility pointer to make code shorter
  float *ptr1, *ptr2, *ptr3, *ptr4;     // more utility pointers, kept for
					// historical reasons

  DepthSortObject2 depthSortObject;	// My sorted array of Transparent
					// object info.  It contains info
					// about the color, material on/off
					// characteristics, dataBlock index,
					// and a pointer to the data itself
					// at the time the object was to
					// be drawn.

  int transformFlag = 0;		// This is a flag indicating that
					// something has happened to the
					// current view matrix. I would then
					// need to recalculate the eyePos.

	// Get information that will be needed in order to depth sort.
	// I have to get the current view matrix, invert it, and then
	// multiply the eyePosition by this inverted matrix in order to place
	// it into world coordinates that can be compared with the coordinates
	// of objects to be drawn.


	// Calculate the normalized position of the light source, for 
	// purposes of calculating the shading model
  float normFactor;
  normFactor = lightPos[0][0]*lightPos[0][0]+  lightPos[0][1]*lightPos[0][1] +
	       lightPos[0][2]*lightPos[0][2];
  normFactor = sqrt(normFactor);
  normLight[0]= lightPos[0][0]/normFactor;
  normLight[1]= lightPos[0][1]/normFactor;
  normLight[2]= lightPos[0][2]/normFactor;


  depthSortObject.matrix = transMat.top();
  utilMatrix = (Matrix4 *) (&(depthSortObject.matrix));
  utilMatrix = &((*utilMatrix).inverse());
  (*utilMatrix).multpoint3d(eyePos, scaledeyePos);

  MSGDEBUG(3, "PSDisplayDevice: rendering command list." << sendmsg);
   
  if(!cmdList)
    return;

  switch(my_detail_level) {
  case DETAIL_FULL: // full detail and material characteristics
     break;
  case DETAIL_FLAT: // full detail, no material characteristics
    deactivate_materials();
    break;
  case DETAIL_WIREFRAME: // set up the lines for wireframe
    set_line_style(SOLIDLINE);
    set_line_width(2);
    break;
  case DETAIL_POINTS: // points
    set_line_width(2);
    break;
  case DETAIL_NONE: // nothing
  default:
    // Then I'm outta here!!!!
     return;
  }


  // scan through the list, getting each command and executing it, until
  // the end of commands token is found
  dataBlock = NULL;
#if defined(USE_DISKS) || defined(USE_NURBS)
  int last_tok = -1;
#endif
  while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
    if (tok == DLINKLIST) {
       cmdptr += sizeof(int);       // step forward one (no length here)
       cmdptr = *((char **) cmdptr);// and go to the next link
       tok = ((int *)cmdptr)[0];    // this is guarenteed to be neither
    }                               // a DLINKLIST nor DLASTCOMMAND

    cmdsize = ((int *)cmdptr)[1];
    cmdptr += 2*sizeof(int);

    // In order to beautify code, do a few operations that would
    // always need to be done in the same way for transparent
    // objects.
      depthSortObject.matrix = transMat.top();
      utilMatrix = (Matrix4 *) (&(depthSortObject.matrix));
      depthSortObject.color = color_index;
      depthSortObject.moredata = moredata;
      depthSortObject.material = material_active;
      depthSortObject.textpos = PStext_position;

   MSGDEBUG(4, "PSDisplayDevice: doing command " << tok);
   MSGDEBUG(4, " of size " << cmdsize << sendmsg);

 switch (tok) {
   case DPOINT: 
	  utilPtr = (float *)cmdptr;
	  center[0] = utilPtr[0];
	  center[1] = utilPtr[1];
	  center[2] = utilPtr[2];
	  depthSortObject.num = 1;
	  depthSortObject.data = new char_ptr[depthSortObject.num];
	  depthSortObjectFound = 1;
    break;
  case DPOINT_I:
	utilPtr = (float *)(dataBlock + ((int *)cmdptr)[0]);
        center[0] = utilPtr[0];
        center[1] = utilPtr[1];
        center[2] = utilPtr[2];
	depthSortObject.num = 1;
	depthSortObject.data = new char_ptr[depthSortObject.num];
	depthSortObjectFound = 1;
     break;
   case DLINE:
	   utilPtr = (float *)cmdptr;
	   center[0] = (utilPtr[0] + utilPtr[3])/2;
	   center[1] = (utilPtr[1] + utilPtr[4])/2;
	   center[2] = (utilPtr[2] + utilPtr[5])/2;
	   depthSortObject.num = 3;
	   depthSortObject.data = new char_ptr[depthSortObject.num];
	   (depthSortObject.data)[1] = line_style;
	   (depthSortObject.data)[2] = line_width;
	   depthSortObjectFound = 1;
    break;
    case DLINE_I:
      // plot a line, using indices into dataBlock
         ptr1 = dataBlock + ((int *) cmdptr)[0];
         ptr2 = dataBlock + ((int *) cmdptr)[1];
	 center[0] = (ptr1[0] + ptr2[0])/2;
	 center[1] = (ptr1[1] + ptr2[1])/2;
	 center[2] = (ptr1[2] + ptr2[2])/2;
	 depthSortObject.num = 3;
	 depthSortObject.data = new char_ptr[depthSortObject.num];
	 (depthSortObject.data)[1] = line_style;
	 (depthSortObject.data)[2] = line_width;
	 depthSortObjectFound = 1;
   break;
   case DSPHERE:
#if !defined( __NPGL__) && !defined(ARCH_AIX3)
	PSsphere ((float *)cmdptr);
#else
      // plot a point at the given position
	 utilPtr = (float *)cmdptr;
         center[0] = utilPtr[0];
         center[1] = utilPtr[1];
         center[2] = utilPtr[2];
	 depthSortObject.num = 1;
	 depthSortObject.data = new char_ptr[depthSortObject.num];
	 depthSortObjectFound = 1;
#endif
    break;
    case DSPHERE_I:  
      {
   int n1 = (int)(((float *)cmdptr)[0]);
#if !defined( __NPGL__) && !defined(ARCH_AIX3)
	 float center_I[4];
	 center_I[0] = dataBlock[n1];
	 center_I[1] = dataBlock[n1+1];
	 center_I[2] = dataBlock[n1+2];
	 center_I[3] = ((float *)cmdptr)[1];
	 PSsphere (center_I);
#else
     utilPtr = dataBlock + n1;
     center[0] = utilPtr[0];
     center[1] = utilPtr[1];
     center[2] = utilPtr[2];
     depthSortObject.num = 1;
     depthSortObject.data = new char_ptr[depthSortObject.num];
     depthSortObjectFound = 1;
#endif
      }
   break;
   case DTRIANGLE:
	 utilPtr = (float *) cmdptr;
	 center[0] = (utilPtr[0]+utilPtr[3]+utilPtr[6])/3;
	 center[1] = (utilPtr[1]+utilPtr[4]+utilPtr[7])/3;
	 center[2] = (utilPtr[2]+utilPtr[5]+utilPtr[8])/3;
	 depthSortObject.num = 3;
	 depthSortObject.data = new char_ptr[depthSortObject.num];
	 (depthSortObject.data)[1] = line_style;
	 (depthSortObject.data)[2] = line_width;
	 depthSortObjectFound = 1;
  break;
  case DTRIANGLE_I:
 	 ptr1 = dataBlock + ((int *) cmdptr)[1];
	 ptr2 = dataBlock + ((int *) cmdptr)[2];
	 ptr3 = dataBlock + ((int *) cmdptr)[3];
         center[0] = (ptr1[0] + ptr2[0] + ptr3[0])/3;
	 center[1] = (ptr1[1] + ptr2[1] + ptr3[1])/3;
	 center[2] = (ptr1[2] + ptr2[2]+  ptr3[2])/3;
	 depthSortObject.num = 3;
	 depthSortObject.data = new char_ptr[depthSortObject.num];
	 (depthSortObject.data)[1] = line_style;
	 (depthSortObject.data)[2] = line_width;
	 depthSortObjectFound = 1;
  break;
  case DSQUARE:
	utilPtr = (float *)cmdptr;
	center[0] = (utilPtr[0]+utilPtr[3]+utilPtr[6]+utilPtr[9])/4;
        center[1] = (utilPtr[1]+utilPtr[4]+utilPtr[7]+utilPtr[10])/4;
        center[2] = (utilPtr[2]+utilPtr[5]+utilPtr[8]+utilPtr[11])/4;
	depthSortObject.num = 3;
	depthSortObject.data = new char_ptr[depthSortObject.num];
	(depthSortObject.data)[1] = line_style;
	(depthSortObject.data)[2] = line_width;
	depthSortObjectFound = 1;
   break;
   case DSQUARE_I:
	ptr1 = dataBlock + ((int *)cmdptr)[0];
	ptr2 = dataBlock + ((int *)cmdptr)[1];
	ptr3 = dataBlock + ((int *)cmdptr)[2];
	ptr4 = dataBlock + ((int *)cmdptr)[3];
        center[0] = (ptr1[0]+ptr2[0]+ptr3[0]+ptr4[0])/4;
	center[1] = (ptr1[1]+ptr2[1]+ptr3[1]+ptr4[1])/4;
	center[2] = (ptr1[2]+ptr2[2]+ptr3[2]+ptr4[2])/4;
	depthSortObject.num = 3;
	depthSortObject.data = new char_ptr[depthSortObject.num];
	depthSortObject.data[1] = line_style;
	depthSortObject.data[2] = line_width;
	depthSortObjectFound = 1;
    break;
    case DCYLINDER:
// The #define is a tradeoff between speed and memory usage
//#define USE_SLOW_CYLINDERS
#ifdef USE_SLOW_CYLINDERS
         utilPtr = (float *)cmdptr;
	 center[0] = (utilPtr[0] + utilPtr[3])/2;
	 center[1] = (utilPtr[1] + utilPtr[4])/2;
	 center[2] = (utilPtr[2] + utilPtr[5])/2;
	 depthSortObject.num = 3;
	 depthSortObject.data = new char_ptr[depthSortObject.num];
	 (depthSortObject.data)[1] = line_style;
	 (depthSortObject.data)[2] = line_width;
	 depthSortObjectFound = 1;
#else
         utilPtr = ((float *)cmdptr) + 9;
         center[0] = (utilPtr[3] + utilPtr[6])/2;
         center[1] = (utilPtr[4] + utilPtr[7])/2;
         center[2] = (utilPtr[5] + utilPtr[8])/2;
	 depthSortObject.num = 3;
	 depthSortObject.data = new char_ptr[depthSortObject.num];
	 (depthSortObject.data)[1] = line_style;
	 (depthSortObject.data)[2] = line_width;
	 depthSortObjectFound = 1;
#endif
    break;
    case DCYLINDER_I:
	ptr1 = dataBlock +  (int)(((float *)cmdptr)[0]);
	ptr2 = dataBlock +  (int)(((float *)cmdptr)[1]);
	center[0] = (ptr1[0] + ptr2[0])/2;
	center[1] = (ptr1[1] + ptr2[1])/2;
	center[2] = (ptr1[2] + ptr2[2])/2;
	depthSortObject.num = 3;
	depthSortObject.data = new char_ptr[depthSortObject.num];
	(depthSortObject.data)[1] = line_style;
	(depthSortObject.data)[2] = line_width;
	depthSortObjectFound = 1;
    break;
    case DCONE:
	utilPtr = (float *)cmdptr;
	center[0] = (utilPtr[0] + utilPtr[3])/2;
	center[1] = (utilPtr[1] + utilPtr[4])/2;
        center[2] = (utilPtr[2] + utilPtr[5])/2;
	depthSortObject.num = 3;
	depthSortObject.data = new char_ptr[depthSortObject.num];
	(depthSortObject.data)[1] = line_style;
	(depthSortObject.data)[2] = line_width;
	depthSortObjectFound = 1;
   break;
   case DCONE_I:
	ptr1 = dataBlock +  (int)(((float *)cmdptr)[0]);
	ptr2 = dataBlock +  (int)(((float *)cmdptr)[1]);
        center[0] = (ptr1[0] + ptr2[0])/2;
        center[1] = (ptr1[1] + ptr2[1])/2;
        center[2] = (ptr1[2] + ptr2[2])/2;
	depthSortObject.num = 3;
	depthSortObject.data = new char_ptr[depthSortObject.num];
	depthSortObject.data[1] = line_style;
	depthSortObject.data[2] = line_width;
	depthSortObjectFound = 1;
    break;
    case DTEXTPOS:
      // move the current character position to the point given
   	textX = ((float *)cmdptr)[0];
  	textY = ((float *)cmdptr)[1];
        textZ = ((float *)cmdptr)[2];
	PStext_position = cmdptr;
    break;
    case DTEXTPOS_I:
      {
      // move the current character position to the point given, using indices
      int n1 = ((int *)cmdptr)[0];
      textX = float(dataBlock[n1]);
      textY = float(dataBlock[n1+1]);
      textZ = float(dataBlock[n1+2]);
      PStext_position = (char *)(&(dataBlock[n1]));
      }
      break;
    case DTEXT: 
      // display text at current position
      {
      center[0] = textX;
      center[1] = textY;
      center[2] = textZ;
      depthSortObject.num = 1;
      depthSortObject.data = new char_ptr[depthSortObject.num];
      depthSortObjectFound = 1;
      }
      break;
    case DCOLORINDEX:
      {
      // set the current color to the given color index ... assumes the
      // color has already been defined
      int n = ((int *)cmdptr)[0];
      
      if(!matDefined[n]) {
        msgErr << "PSDisplayDevice: Error, color " << n << " not defined."
	         << sendmsg;
      } else {
        // set depth cue range if necessary
	color_index = cmdptr - 2*sizeof(int);

        // bind the material if materials enabled
	materialOn = n;
	if(materialsActive)
	  activate_material();
      }
      }
      break;
    case DCOLORRGB:
      // just set the non-material color to the given value.  Turns off
      // materials.
      deactivate_materials();
      // Actually, this should never be called!!!!
      break;
    case DPICKPOINT:
    case DPICKPOINT_I:
    case DPICKLINE:
    case DPICKLINE_I:
    case DPICKBOX:
    case DPICKBOX_I:
     break;  // do nothing, picking routines
    case DLIGHTONOFF:
      // turn on or off a light source
      activate_light(((int *)cmdptr)[0], ((int *)cmdptr)[1]);
      break;
    case DMATERIALS:  
      // enable/disable material characteristics
      if (my_detail_level > DETAIL_FLAT) {
	material_active = cmdptr - 2*sizeof(int);
	if(((int *)cmdptr)[0])
	  activate_material();
	else
	  deactivate_materials();
      }
      break;
     case DSPHERERES: 
      // set the current sphere resolution
      set_sphere_res(((int *)cmdptr)[0]);
      sphere_res = cmdptr - 2*sizeof(int);
      break;
     case DSPHERETYPE:
      // set the current sphere type
      sphere_mode = cmdptr - 2*sizeof(int);
      break;
     case DLINESTYLE: 
      // set the current line style
      line_style = cmdptr - 2*sizeof(int);
      if (my_detail_level > DETAIL_WIREFRAME) 
	set_line_style(((int *)cmdptr)[0]);
      break;
     case DLINEWIDTH: 
      // set the current line width
      line_width = cmdptr - 2*sizeof(int);
      if (my_detail_level > DETAIL_WIREFRAME) 
	set_line_width(((int *)cmdptr)[0]);
      break;
     case DPUSH:
      // push the current trans matrix
      super_push();
      transformFlag = 1;
      break;
     case DPOP:
      // pop the current trans matrix
      super_pop();
      transformFlag = 1;
      break;
     case DLOAD: 
      // load the given trans matrix onto the stack
      super_load( (float *) cmdptr);
      transformFlag = 1;
      break;
     case DMULT:
      // multiply the given trans matrix
      super_multmatrix( (float *) cmdptr);
      transformFlag = 1;
      break;
     case DTRANS:
      super_translate( (float *) cmdptr);
      transformFlag = 1;
      break;
     case DSCALE:
      super_scale( (float *) cmdptr);
      transformFlag = 1;
      break;
     case DROT:
      // do a rotation about a given axis
      super_rot( (float *) cmdptr);
      transformFlag = 1;
      break;
     case DCOLORDEF:
       {
      // define a new color
      int n = (int)(((float *)cmdptr)[0]);
      define_material(n, ((float *)cmdptr) + 1);
       }
      break;
     case DLIGHTDEF:
       {
      // define a new light source
      float *light_ptr = ((float *)cmdptr)+4;
      light_position[0] = light_ptr[0];
      light_position[1] = light_ptr[1];
      light_position[2] = light_ptr[2];
       }
      break;
     case DCLEAR:
      // clear the display
      clear();
      break;
     case DCOMMENT:
      comment( (char *)cmdptr);
      break;
     case DMOREDATA:
     case DNEWDATA:
      // set the current drawing data block
#ifdef VMDCAVE
      dataBlock = (float *)cmdptr;	// point to current position in list
      moredata = dataBlock;
#else
      dataBlock = ((float **)cmdptr)[0];	// point to given memory loc
      moredata = dataBlock;
#endif
      break;
     default: 
      // command not found, help!
      msgErr << "PSDisplayDevice: Unknown drawing token " << tok
             << " encountered ... Skipping this command." << sendmsg;
      break;
 }  // End switch statement for token type

    // If the command was a DLOAD, etc. which altered the view matix,
    // then the new scaled eye position needs to be recalculated
    if (transformFlag) {
      depthSortObject.matrix = transMat.top();
      utilMatrix = (Matrix4 *) (&(depthSortObject.matrix));
      utilMatrix = &((*utilMatrix).inverse());
      (*utilMatrix).multpoint3d(eyePos, scaledeyePos);
    }
    transformFlag = 0; 

    
    // If the command indicated an object to be drawn, finish up
    // saving important state information and then add this object
    // to the list which is to be depth sorted.
    if (depthSortObjectFound) {
      depthSortObject.data[0] = cmdptr - (2*sizeof(int));
      depthSortObject.distToEye=DTOEYE(center[0],center[1],center[2]);
      DepthSortList.append(depthSortObject);
    }
    depthSortObjectFound = 0;
    
    
    // update position in array
    cmdptr += cmdsize;
  }  
#if defined(USE_DISKS) || defined(USE_NURBS)
  // if the last with a sphere, need to undo the special matrix
  if (last_tok == DSPHERE || last_tok == DSPHERE_I) {
    ::popmatrix();
  }
#endif
  // restore transformation matrix
  //pop();
}

//***************************** render_done ******************************
/* This is called after all molecules to be displayed have been sent
   through the list which depth sorts if necessary.  It sorts the 
   list and then calls code which goes through and does the actual 
   drawing of the object. */

void PSDisplayDevice::render_done() {
    screenHor = Aspect*vSize;
    screenVer = Aspect*vSize;
    XScaleFact = 1.33 * (psHor/screenHor);
    YScaleFact = 1.33 * (psVer/screenVer);
    // Set background first so it can be changed by user if necessary
    fprintf (outfile, "%3.2f %3.2f %3.2f setrgbcolor ", backColor[0],
	     backColor[1],backColor[2]);
    fprintf (outfile, "%%Background color\n");
    fprintf (outfile, "newpath\n");
    fprintf (outfile, "0 0 moveto\n");
    fprintf (outfile, "0 %d lineto\n", psHor);
    fprintf (outfile, "%d %d lineto\n", psHor, psVer);
    fprintf (outfile, "%d 0 lineto\n", psVer);
    fprintf (outfile, "closepath\nfill\nstroke\n");
    define_PS_procs();

    if (DepthSortList.num() != 0) {
       DepthSortList.qsort(0, DepthSortList.num()-1);
       process_depth_sorted_list();
    }
}

void PSDisplayDevice::define_PS_procs() {
    // square
    fprintf (outfile, "/square %%stack: R G B x1 y1 x2 y2 x3 y3 x4 y4\n");
    fprintf (outfile, "{\n  setrgbcolor\n  newpath\n  moveto\n");
    fprintf (outfile, "  lineto\n  lineto\n  lineto\n  closepath\n");
    fprintf (outfile, "  fill\n  stroke\n} def\n\n");
    // end square

    // square2  -- for wireframe output
    fprintf (outfile, "/square2 %%same as square,");
    fprintf (outfile, " except no fill, used for wireframe\n");
    fprintf (outfile, "{\n  setrgbcolor\n  newpath\n  moveto\n");
    fprintf (outfile, "  lineto\n  lineto\n  lineto\n  closepath\n");
    fprintf (outfile, "  stroke\n} def\n\n");
    // end square2

    // triangle
    fprintf (outfile, "/triangle %%stack: R G B x1 y1 x2 y2 x3 y3\n");
    fprintf (outfile, "{\n  setrgbcolor\n  newpath\n  moveto\n");
    fprintf (outfile, "  lineto\n  lineto\n  closepath\n  fill\n");
    fprintf (outfile, "  stroke\n} def\n\n");
    // end triangle

    // triangle2 -- for wireframe output
    fprintf (outfile, "/triangle2 %%same as triangle,");
    fprintf (outfile, " except no fill, used for wireframe\n");
    fprintf (outfile, "{\n  setrgbcolor\n  newpath\n  moveto\n");
    fprintf (outfile, "  lineto\n  lineto\n  closepath\n");
    fprintf (outfile, "  stroke\n} def\n\n");
    // end triangle2

    // point
    fprintf (outfile, "/point %%stack R G B x1 y1\n");
    fprintf (outfile, "{\n  5 1 roll\n  5 1 roll\n  5 1 roll\n");
    fprintf (outfile, "  dup\n  3 1 roll\n  2 1 roll\n  dup\n");
    fprintf (outfile, "  3 1 roll\n  4 1 roll\n");

    fprintf (outfile, "  dup\n  3 1 roll\n  2 1 roll\n  dup\n");
    fprintf (outfile, "  3 1 roll\n  4 1 roll\n");

    fprintf (outfile, "  dup\n  3 1 roll\n  2 1 roll\n  dup\n");
    fprintf (outfile, "  3 1 roll\n  4 1 roll\n");

    fprintf (outfile, "  11 -1 roll\n  11 -1 roll\n  11 -1 roll\n");
    fprintf (outfile, "  setrgbcolor\n  newpath\n  -1 add\n");
    fprintf (outfile, "  moveto\n  1 add\n  lineto\n");
    fprintf (outfile, "  2 1 roll\n  -1 add\n  2 1 roll\n");
    fprintf (outfile, "  moveto\n  2 1 roll\n  1 add\n  2 1 roll\n  lineto\n");
    fprintf (outfile, "  closepath\n  stroke\n} def\n\n");
    // end point

    // line
    fprintf (outfile, "/line %%stack R G B x1 y1 x2 y2\n");
    fprintf (outfile, "{\n  setrgbcolor\n  newpath\n");
    fprintf (outfile, "  moveto\n  lineto\n  closepath\n");
    fprintf (outfile, "  stroke\n} def\n\n");
    // end line


}


//************************** render_token ******************************
// This code processes a single command indicated by what the parameter
// cmdptr is pointing to.  Much of the code is repetion of what is
// found in render().
void PSDisplayDevice::render_token(char *cmdptr, float *dataBlock) {
  int tok, cmdsize;
  MSGDEBUG(3, "PSDisplayDevice: rendering command list." << sendmsg);

    tok = ((int *)cmdptr)[0]; 
    cmdsize = ((int *)cmdptr)[1];
    cmdptr += 2*sizeof(int);


    switch(tok) {
    case DPOINT:
      PSpoint ((float *)cmdptr);
      break;
    case DPOINT_I:
      break;
    case DLINE:
      PSline((float *)cmdptr, ((float *)cmdptr) + 3);
      break;
    case DLINE_I:
      {
      float *ptr1 = dataBlock + ((int *) cmdptr)[0];
      float *ptr2 = dataBlock + ((int *) cmdptr)[1];
      PSline(ptr1, ptr2); 
      }
      break;
    case DSPHERE:
      {
      float *ver1 = (float *)cmdptr;
      float *ver2 = ver1 + 3;
      float *ver3 = ver2 + 3;
      PStriangleScaled (ver1, ver2, ver3);
      }
      break;
    case DSPHERE_I:
      {
     int n1 = (int)(((float *)cmdptr)[0]);
     float sphData[4];
     sphData[0] = dataBlock[n1++];
     sphData[1] = dataBlock[n1++];
     sphData[2] = dataBlock[n1];
     sphData[3] = ((float *)cmdptr)[1];
     PSsphere (sphData);
      }
     break;
    case DTRIANGLE:
     PStriangle(  (float *) cmdptr      , ((float *) cmdptr) + 3,
		 ((float *) cmdptr) +  6);
     break;
    case DTRIANGLE_I:
      // draw a triangle between the three points, using indices into dataBlock
     PStriangle( dataBlock + ((int *) cmdptr)[1],
  	       dataBlock + ((int *) cmdptr)[2],
	       dataBlock + ((int *) cmdptr)[3]);
     break;
    case DSQUARE:
      // draw a square, given the four points
     PSsquare(  ((float *) cmdptr) + 3, 
             ((float *) cmdptr) + 6, ((float *) cmdptr) + 9,
             ((float *) cmdptr) +12 );
     break;
    case DSQUARE_I:
      // draw a square, given the four points, using indices into dataBlock
     PSsquare(dataBlock + ((int *)cmdptr)[1],
            dataBlock + ((int *)cmdptr)[2], dataBlock + ((int *)cmdptr)[3],
            dataBlock + ((int *)cmdptr)[4]);
     break;
    case DCYLINDER:
#ifdef USE_SLOW_CYLINDERS
      PScylinder ((int)(((float *)cmdptr)[7]), ((float *)cmdptr) + 9,
		     int(((float *) cmdptr)[8]) != 0);
#else
      PScylinder((int)(((float *)cmdptr)[7]), ((float *)cmdptr) + 9,
	       int(((float *) cmdptr)[8]) != 0);
#endif
         break;
        case DCYLINDER_I:
          break;
        case DCONE:
         // draw a cone of given radius and resolution
	  PScone((float *)cmdptr, ((float *)cmdptr) + 3,
	      (int)(((float *)cmdptr)[7]), ((float *)cmdptr)[6]);
          break;
        case DCONE_I:
         // draw a cone of given radius and resolution, using indices
         break; 
        case DCOLORINDEX:
	  {
          // set the current color to the given color index ... assumes the
      // color has already been defined
         int n = ((int *)cmdptr)[0];
         col_ptr[0] = (&(matData[n][COLOR_INDEX]))[0];
         col_ptr[1] = (&(matData[n][COLOR_INDEX]))[1];
         col_ptr[2] = (&(matData[n][COLOR_INDEX]))[2];
	  }
         break;
        case DSPHERERES: 
          // set the current sphere resolution
          set_sphere_res(((int *)cmdptr)[0]);
	  break;
        case DSPHERETYPE:
          // set the current sphere type
         set_sphere_mode(((int *)cmdptr)[0]);
	 break;
 	case DTEXT:
	 PStext(cmdptr,textX,textY,textZ);
	 break;
        case DLINESTYLE:
         // set the current line style
         if (my_detail_level > DETAIL_WIREFRAME) 
	   set_line_style(((int *)cmdptr)[0]);
         break;
        case DLINEWIDTH:
          // set the current line width
          if (my_detail_level > DETAIL_WIREFRAME) 
	    set_line_width(((int *)cmdptr)[0]);
        break;
	case DMATERIALS:
	  if (my_detail_level > DETAIL_FLAT) {
	    if (((int *)cmdptr)[0])
	      activate_material();
            else
	      deactivate_materials();
          }
        break;
}
}



/////////////////////////////////// render the display lists
/* Eventually, this should be eliminated, render_depth_sort should
   be renamed render.  There isn't much reason for keeping this
   around. */

void PSDisplayDevice::render(void *display_list) {
  if (!display_list) return;
  sphere_res = NULL;
  sphere_mode = NULL;
  line_style = NULL;
  line_width = NULL;
  color_index = NULL;
  material_active = NULL;
  moredata = NULL;
  PStext_position = NULL;


  // scan through the list and do the action based on the token type
  // if the command relates to the viewing state, keep track of it
  // for those renderers that don't store state


  while (transMat.num()) {    // clear the stack
    transMat.pop();
  }
  Matrix4 m;
  transMat.push(m);           // push on the identity matrix

  colorIndex = 0;
  lineWidth = 1;
  lineStyle = 1;
  sphereResolution = 4;
  sphereStyle = 1;
  render_depth_sort (display_list);
}


