/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: PickModeCenter.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 1996/03/20 06:57:14 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Pick on an atom to change the molecule's centering/global matricies
 *
 ***************************************************************************/

#include <math.h>
#include <string.h>
#include "PickModeCenter.h"
#include "Pickable.h"
#include "DisplayDevice.h"
#include "Inform.h"

#include "Global.h"
#include "CommandQueue.h"
#include "MoleculeList.h"

#include "CmdLabel.h"

//////////////////////////// constructor  ////////////////////////////
PickModeCenter::PickModeCenter(void) {
  pCenter = NULL;
  needName = FALSE;
}


///////////////////// public virtual routines  ////////////////////////

// called when a pick is begun:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// Here, who cares what the button was
void PickModeCenter::pick_start(DisplayDevice *,
			Pickable *p, int, int, int dim, float *pos) {
  pCenter = p;
  memcpy((void *)pPos, (void *)pos, dim*sizeof(float));
  needName = (p != NULL);
}

  
// called when a pick moves:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// Here, who cares what the button was
void PickModeCenter::pick_move(DisplayDevice *,
			Pickable *p, int, int, int dim, float *pos) {
  if(needName) {
    float mvdist = 0.0;
    for(int i=0; i < dim; i++)
      mvdist += fabs(pPos[i] - pos[i]);
    if(mvdist > 0.02 || p != pCenter)
      needName = FALSE;
  }
}


// called when a pick ends:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// Here, who cares what the button was
void PickModeCenter::pick_end(DisplayDevice *,
			Pickable *p, int , int atm, int, float *) {
  if(p == pCenter && needName) {
    // Find which molecule was picked
    int mnum = moleculeList -> num();
    Molecule *tmpmol, *mol = NULL;
    for (int i=0; i<mnum; i++) {
      tmpmol = moleculeList -> molecule(i);
      if (p == tmpmol) {
	mol = tmpmol;
	break;
      }

      int repnum = tmpmol->components();
      for (int j=0; j<repnum; j++) {
        if (p == tmpmol->component(j)) {
          mol = tmpmol;
          break;
        }
      }
      if (mol) break;
    }
    if (mol) {
      Timestep *ts = mol -> current();
      if (!(atm < 0 || ts == NULL)) {
        // get the coordinate
        float *coord = ts->pos + atm * 3;

	// and apply the result to all the active molecules
	Molecule *m;
	for (int m_id = 0; m_id < mnum; m_id++) {
	  m = moleculeList -> molecule(m_id);
	  if (m -> active) {
	    m->change_center(coord[0], coord[1], coord[2]);
	  }
	  // since this isn't a pure command, I'll log it myself
	  char tmps[123];
	  sprintf(tmps, "molinfo %d set center {{%f %f %f}}",
		  m->id(), coord[0], coord[1], coord[2]);
	  commandQueue -> print_log(tmps);
	}
	
	// Add the label to say I centered
	{
	  char temps[128];
	  mol -> atom_full_name(atm, temps);
	  char *s = temps;
	  commandQueue -> append(new CmdLabelAdd("Atoms", 1, &s));
	}
      }
    }

  }
  pCenter = NULL;
  needName = FALSE;
}

