/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Pickable.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1995/11/15 04:29:29 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Pickable object is one which contains data which may be selected by
 * using a pointer device in an open DisplayDevice.  Each Pickable registers
 * itself with a PickList object which indicates it has items in its
 * drawing list which may be picked, and which indicates it should be told
 * when something has been successfully selected.  There are one or many
 * picking modes, which are represented by derivations of the PickMode class.
 * When picking is done, it is done while in a current pick mode, which
 * controls what to do with the information.
 *
 * A Pickable must provide versions of virtual routines which tell what
 * pick modes the object is interested in.
 ***************************************************************************/

#include "Pickable.h"
#include "PickList.h"
#include "PopupMenu.h"


//////////////////////////// constructor  ////////////////////////////
Pickable::Pickable(PickList *p) {

  // init picklist pointer
  pickList = NULL;
  
  // possibly register with this pickList
  register_with_picklist(p);
}


// destructor
Pickable::~Pickable(void) {

  // unregister with the picklist
  if(pickList)
    pickList->remove_pickable(this);
}


//////////////////////////// protected routines  ////////////////////////////

// creates a new pick mode of a type specified by the code integer argument.
// this is typically only used by add_pick_mode; the code is both defined
// by and used by the derived class
PickMode *Pickable::create_pick_mode(int) {
  // by default, this will not create a new mode
  return NULL;
}



//////////////////////////// public routines  ////////////////////////////

// register with the given PickList.  If already registered, deregisters
// with the old one first
void Pickable::register_with_picklist(PickList * p) {

  // only do anything if p is not NULL
  if(p) {
    // deregister from picklist if necessary first
    if(pickList)
      pickList->remove_pickable(this);
      
    // register with new picklist
    pickList = p;
    pickList->add_pickable(this);
  }
}


// check to see if a PickMode with the given name has been created yet
// for the PickList.  If so, return the index of the mode.  If not,
// this routine calls the virtual create_pick_mode with the 2nd argument,
// which is a code for the derived class to know what kind of PickMode
// object to create.  Returns (-1) if some sort of error occurs.
int Pickable::add_pick_mode(char *name, int mcode) {

  // can only work if we have a current pick list
  if(!pickList)
    return (-1);

  // for our current PickList object, check if mode already exists
  int indx = pickList->pick_mode_num(name);

  // if it does not exist, create a new one first
  if(indx < 0) {
    PickMode *pm = create_pick_mode(mcode);
    if(!pm)
      return (-1);
    indx = pickList->add_pick_mode(name, pm);
  }
  
  // return the index of the mode
  return indx;
}



//
// public virtual informative routines
//
  
// return if we are interested in the given pick mode or not
int Pickable::want_pick_mode(int) { return FALSE; }
  
// return our list of draw commands with picking draw commands in them
void *Pickable::pick_cmd_list(void) { return NULL; }
  
// return name of pickable
char *Pickable::pick_name(void) { return NULL; }

// return whether the pickable object is being displayed
int Pickable::pickable_on(void) { return TRUE; }


//
// public virtual specialized popup menu routines
//

// Create a popup menu, with the given name; return NULL if none available
// subclasses should have the parent first create a menu, and then add items
// to the end of it.
PopupMenu *Pickable::create_popup_menu(int) {
  // By default, this does nothing and returns NULL.
  return NULL;
}

  
//
// public virtual pick action routines ... by default, these do nothing
//

// called when a pick is begun:
//	args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void Pickable::pick_start(DisplayDevice *, Pickable *,
  				int, int, int, int, float *) { }
  
// called when a pick moves:
//	args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void Pickable::pick_move(DisplayDevice *, Pickable *,
  				int, int, int, int, float *) { }
  
// called when a pick ends:
//	args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void Pickable::pick_end(DisplayDevice *, Pickable *,
  				int, int, int, int, float *) { }


