/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: RayShadeDisplayDevice.C,v $
 *      $Author: ulrich $        $Locker:  $                $State: Exp $
 *      $Revision: 1.10 $      $Date: 1997/03/21 17:55:01 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * FileRenderer type for the Rayshade raytracer 
 *
 ***************************************************************************/
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "RayShadeDisplayDevice.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"
#define DEFAULT_RADIUS 0.002
#define DASH_LENGTH 0.02
#define SCALE_FACTOR 1.3

///////////////////////// constructor and destructor

// constructor ... initialize some variables
RayShadeDisplayDevice::RayShadeDisplayDevice() : FileRenderer("RayShade",
	                                                      "plot.ray",
						    "rayshade < %s > %s.rle"){
	
    MSGDEBUG(1,"Creating RayShadeDisplayDevice ..." << sendmsg);
}
        
// destructor
RayShadeDisplayDevice::~RayShadeDisplayDevice(void) { }

///////////////////////// protected nonvirtual routines

// fix the scaling
float RayShadeDisplayDevice::scale_fix(float x) {
    return ( x / SCALE_FACTOR );  
}

// draw a point
void RayShadeDisplayDevice::point(float spdata[]) {
    float vec[3];

    // transform the world coordinates
    (transMat.top()).multpoint3d(spdata, vec);
   
    // draw the sphere

    fprintf(outfile, "applysurf diffuse %3.2f %3.2f %3.2f specular 1.0 1.0 1.0 transp %3.2f", 
	matData[colorIndex][DIFFUSE_INDEX+0],
	matData[colorIndex][DIFFUSE_INDEX+1],
	matData[colorIndex][DIFFUSE_INDEX+2],
	matData[colorIndex][ALPHA_INDEX]);

    fprintf(outfile, "specpow 40.0\n");
    fprintf(outfile, "sphere %5f %5f %5f %5f\n", 
    scale_fix(float(lineWidth)*DEFAULT_RADIUS),
        scale_fix(vec[0]), scale_fix(vec[1]), scale_fix(vec[2]));
 }

// draw a sphere
void RayShadeDisplayDevice::sphere(float spdata[]) {
    float vec[3];
    float radius;
    
    // transform the world coordinates
    (transMat.top()).multpoint3d(spdata, vec);
    radius = scale_radius(spdata[3]);
   
    // draw the sphere

    fprintf(outfile, "applysurf diffuse %3.2f %3.2f %3.2f specular 1.0 1.0 1.0 transp %3.2f", 
	matData[colorIndex][DIFFUSE_INDEX+0],
	matData[colorIndex][DIFFUSE_INDEX+1],
	matData[colorIndex][DIFFUSE_INDEX+2],
	matData[colorIndex][ALPHA_INDEX]);
    fprintf(outfile, "specpow 40.0\n");
    fprintf(outfile, "sphere %5f %5f %5f %5f\n", scale_fix(radius), 
        scale_fix(vec[0]), scale_fix(vec[1]), scale_fix(vec[2]));
}

// draw a line (cylinder) from a to b
void RayShadeDisplayDevice::line(float *a, float*b) {
    int i, j, test;
    float dirvec[3], unitdirvec[3];
    float from[3], to[3], tmp1[3], tmp2[3];
    float len;
    
    if(lineStyle == ::SOLIDLINE ) {
  
        // transform the world coordinates
        (transMat.top()).multpoint3d(a, from);
        (transMat.top()).multpoint3d(b, to);
    
        // draw the cylinder
        
    fprintf(outfile, "applysurf diffuse %3.2f %3.2f %3.2f specular 1.0 1.0 1.0 transp %3.2f", 
	matData[colorIndex][DIFFUSE_INDEX+0],
	matData[colorIndex][DIFFUSE_INDEX+1],
	matData[colorIndex][DIFFUSE_INDEX+2],
	matData[colorIndex][ALPHA_INDEX]);

        fprintf(outfile, "specpow 40.0\n");
        fprintf(outfile, "cylinder %5f %5f %5f %5f %5f %5f %5f\n",
            scale_fix(float(lineWidth)*DEFAULT_RADIUS),
            scale_fix(from[0]), scale_fix(from[1]), scale_fix(from[2]),
            scale_fix(to[0]), scale_fix(to[1]), scale_fix(to[2])); 
        
    } else if (lineStyle == ::DASHEDLINE ) {
        
         // transform the world coordinates
        (transMat.top()).multpoint3d(a, tmp1);
        (transMat.top()).multpoint3d(b, tmp2);

        // how to create a dashed line
        for(i=0;i<3;i++) {
            dirvec[i] = tmp2[i] - tmp1[i];  // vector from a to b
        }
        len = sqrtf( dirvec[0]*dirvec[0] + dirvec[1]*dirvec[1] + dirvec[2]*dirvec[2] );
        for(i=0;i<3;i++) {
            unitdirvec[i] = dirvec[i] / sqrtf(len);  // unit vector pointing from a to b
        }
           
        test = 1;
        i = 0;

        while( test == 1 ) {
            for(j=0;j<3;j++) {
                from[j] = tmp1[j] + (2*i)*DASH_LENGTH*unitdirvec[j];
                to[j] = tmp1[j] + (2*i + 1)*DASH_LENGTH*unitdirvec[j];
            }
            if( fabsf(tmp1[0] - to[0]) >= fabsf(dirvec[0]) ) {
                for(j=0;j<3;j++) {
                    to[j] = tmp2[j];
                }
                test = 0;
            }
    
            // draw the cylinder

    fprintf(outfile, "applysurf diffuse %3.2f %3.2f %3.2f specular 1.0 1.0 1.0 transp %3.2f", 
	matData[colorIndex][DIFFUSE_INDEX+0],
	matData[colorIndex][DIFFUSE_INDEX+1],
	matData[colorIndex][DIFFUSE_INDEX+2],
	matData[colorIndex][ALPHA_INDEX]);

            fprintf(outfile, "specpow 40.0\n");
            fprintf(outfile, "cylinder %5f %5f %5f %5f %5f %5f %5f\n",
                scale_fix(float(lineWidth)*DEFAULT_RADIUS),
                scale_fix(from[0]), scale_fix(from[1]), scale_fix(from[2]),
                scale_fix(to[0]), scale_fix(to[1]), scale_fix(to[2])); 
            
            i++;
        }
    } else {
        msgErr << "RayShadeDisplayDevice: Unknown line style " << lineStyle << sendmsg;
    }

}

// draw a cylinder
void RayShadeDisplayDevice::cylinder(float *a, float *b, float r,int filled) {

  float from[3], to[3];
  float radius;
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, from);
  (transMat.top()).multpoint3d(b, to);
  radius = scale_radius(r);
    
  // draw the cylinder
  
        // draw the cylinder

    fprintf(outfile, "applysurf diffuse %3.2f %3.2f %3.2f specular 1.0 1.0 1.0 transp %3.2f", 
	matData[colorIndex][DIFFUSE_INDEX+0],
	matData[colorIndex][DIFFUSE_INDEX+1],
	matData[colorIndex][DIFFUSE_INDEX+2],
	matData[colorIndex][ALPHA_INDEX]);

        fprintf(outfile, "specpow 40.0\n");
        fprintf(outfile, "cylinder %5f %5f %5f %5f %5f %5f %5f\n",
            scale_fix(radius),
            scale_fix(from[0]), scale_fix(from[1]), scale_fix(from[2]),
            scale_fix(to[0]), scale_fix(to[1]), scale_fix(to[2])); 

	// put disks on the ends
  fprintf(outfile, "disc %5f %5f %5f %5f %5f %5f %5f\n",
	  scale_fix(radius),
	  scale_fix(from[0]), scale_fix(from[1]), scale_fix(from[2]),
	  scale_fix(from[0]-to[0]), scale_fix(from[1]-to[1]), 
	  scale_fix(from[2]-to[2]));
  fprintf(outfile, "disc %5f %5f %5f %5f %5f %5f %5f\n",
	  scale_fix(radius),
	  scale_fix(to[0]), scale_fix(to[1]), scale_fix(to[2]),
	  scale_fix(to[0]-from[0]), scale_fix(to[1]-from[1]), 
	  scale_fix(to[2]-from[2]));
}

// draw a cone
void RayShadeDisplayDevice::cone(float *a, float *b, float r) {
  float from[3], to[3];
  float radius;
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, from);
  (transMat.top()).multpoint3d(b, to);
  radius = scale_radius(r);
    
    
  // draw the cone

    fprintf(outfile, "applysurf diffuse %3.2f %3.2f %3.2f specular 1.0 1.0 1.0 transp %3.2f", 
	matData[colorIndex][DIFFUSE_INDEX+0],
	matData[colorIndex][DIFFUSE_INDEX+1],
	matData[colorIndex][DIFFUSE_INDEX+2],
	matData[colorIndex][ALPHA_INDEX]);

    fprintf(outfile, "specpow 40.0\n");
    fprintf(outfile, "cone %5f %5f %5f %5f %5f %5f %5f %5f\n",
        scale_fix(radius), 
        scale_fix(from[0]), scale_fix(from[1]), scale_fix(from[2]), 
        scale_fix(float(lineWidth)*DEFAULT_RADIUS), 
        scale_fix(to[0]), scale_fix(to[1]), scale_fix(to[2]));
}

// draw a triangle
void RayShadeDisplayDevice::triangle(float *a, float *b, float *c, float *n1, float *n2, float *n3) {
    float vec1[3], vec2[3], vec3[3];
    float norm1[3], norm2[3], norm3[3];
  
  
    // transform the world coordinates
    (transMat.top()).multpoint3d(a, vec1);
    (transMat.top()).multpoint3d(b, vec2);
    (transMat.top()).multpoint3d(c, vec3);

    // and the normals
    (transMat.top()).multnorm3d(n1, norm1);
    (transMat.top()).multnorm3d(n2, norm2);
    (transMat.top()).multnorm3d(n3, norm3);

    // draw the triangle

    fprintf(outfile, "applysurf diffuse %3.2f %3.2f %3.2f specular 1.0 1.0 1.0 transp %3.2f", 
	matData[colorIndex][DIFFUSE_INDEX+0],
	matData[colorIndex][DIFFUSE_INDEX+1],
	matData[colorIndex][DIFFUSE_INDEX+2],
	matData[colorIndex][ALPHA_INDEX]);

    fprintf(outfile, "specpow 40.0\n");
    fprintf(outfile, "triangle %5f %5f %5f %5f %5f %5f ", 
        scale_fix(vec1[0]), scale_fix(vec1[1]), scale_fix(vec1[2]),
        scale_fix(norm1[0]), scale_fix(norm1[1]), scale_fix(norm1[2]));
    fprintf(outfile, "%5f %5f %5f %5f %5f %5f ",
        scale_fix(vec2[0]), scale_fix(vec2[1]), scale_fix(vec2[2]),
        scale_fix(norm2[0]), scale_fix(norm2[1]), scale_fix(norm2[2]));
    fprintf(outfile, "%5f %5f %5f %5f %5f %5f\n",
        scale_fix(vec3[0]), scale_fix(vec3[1]), scale_fix(vec3[2]),
        scale_fix(norm3[0]), scale_fix(norm3[1]), scale_fix(norm3[2]));
  
}

// draw a square
void RayShadeDisplayDevice::square(float *norm, float *a, float *b, float *c, float *d) {
  
     // draw as two triangles
     triangle(a, b, c, norm, norm, norm);
     triangle(a, c, d, norm, norm, norm);
  
}

///////////////////// public virtual routines

// initialize the file for output
void RayShadeDisplayDevice::write_header() {
    time_t t;
    
    
        // file for RayShade raytracer.
        
        t = time(NULL);
        fprintf(outfile,"/* Rayshade input script: %s\n", my_filename);
        fprintf(outfile,"   Generated for user: %s\n",cuserid(NULL));
        fprintf(outfile,"   Creation date: %s",asctime(localtime(&t)));
        fprintf(outfile,"  ---------------------------- */\n\n");
        
                
        // The current view 
        
        fprintf(outfile, "\n/* Define current view */\n");        
        fprintf(outfile, "eyep %5f %5f %5f\n", eyePos[0], eyePos[1], eyePos[2]);
        fprintf(outfile, "lookp %5f %5f %5f\n", eyeDir[0], eyeDir[1], eyeDir[2]);
        fprintf(outfile, "up %5f %5f %5f\n", 0.0, 1.0, 0,0);
        fprintf(outfile, "fov %5f\n", 45.0);
        if(stereo_mode()) {
            fprintf(outfile, "eyesep %5f\n", eyesep() );
        }
        fprintf(outfile, "maxdepth 10\n");
        
        // Size of image in pixels
        fprintf(outfile, "screen %d %d\n", xSize, ySize);
      
        // Lights
        
        fprintf(outfile, "\n/* Light Definitions */\n");
        for(int i=0;i<DISP_LIGHTS;i++) {
            if(lightDefined[i] && lightOn[i]) {
                fprintf(outfile, "light %3.2f %3.2f %3.2f ", lightColor[i][0],
                    lightColor[i][1], lightColor[i][2]);
                fprintf(outfile, "directional %5f %5f %5f\n", lightPos[i][0],
                    lightPos[i][1], lightPos[i][2]);
           }
       }
       
       // background color
       fprintf(outfile, "\n/* Set background color */\n");
       fprintf(outfile, "background %3.2f %3.2f %3.2f\n", 
           backColor[0], backColor[1], backColor[2]);
                  
       // start the objects
       fprintf(outfile, "\n/* Start object descriptions */\n");       
    // that's all for the header.
}

void RayShadeDisplayDevice::comment (char *s)
{
  fprintf (outfile, "\n/* %s */\n", s);
}

    
// clean up after yourself
void RayShadeDisplayDevice::write_trailer() {

    fprintf(outfile,"\n/* End of File */\n");
    fclose(outfile);
    msgInfo << "Rayshade file generation finished" << sendmsg;
}


