/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ReadPDB.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 1997/03/20 20:32:04 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * General routines to read .pdb files.
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ReadPDB.h"

/* read the next record from the specified pdb file, and put the string found
   in the given string pointer (the caller must provide adequate (81 chars)
   buffer space); return the type of record found
*/
int read_pdb_record(FILE *f, char *retStr) {

  char inbuf[PDB_RECORD_LENGTH+2];
  int recType = PDB_UNKNOWN;
  
  /*	read the next line	*/
  if(inbuf != fgets(inbuf, PDB_RECORD_LENGTH+1, f)) {
    strcpy(retStr,"");
    recType = PDB_EOF;
  } else {
    /*	remove the newline character, if there is one */
    if(inbuf[strlen(inbuf)-1] == '\n')
      inbuf[strlen(inbuf)-1] = '\0';

    // what was it?
    if (!strncmp(inbuf, "REMARK", 6)) {
      recType = PDB_REMARK;

    } else if (!strncmp(inbuf, "CRYST1", 6)) {
      recType = PDB_CRYST1;
      
    } else if (!strncmp(inbuf, "ATOM  ", 6) ||
	       !strncmp(inbuf, "HETATM", 6)) {
      recType = PDB_ATOM;

      // the only two END records are "END   " and "ENDMDL"
    } else if (!strcmp(inbuf, "END") ||       // If not space " " filled
	       !strncmp(inbuf, "END ", 4) ||  // Allows other stuff
	       !strncmp(inbuf, "ENDMDL", 6)) { // NMR records
      recType = PDB_END;

    } else {
      recType = PDB_UNKNOWN;
    }

    if(recType == PDB_REMARK || recType == PDB_ATOM || 
       recType == PDB_CRYST1) {
      strcpy(retStr,inbuf);
    } else {
      strcpy(retStr,"");
    }
  }

  // read the '\r', if there was one
  {
    int ch = fgetc(f);
    if (ch != '\r') {
      ungetc(ch, f);
    }
  }
  
  return recType;
}

void get_pdb_cryst1(char *record, float *alpha, float *beta, float *gamma,
		    float *a, float *b, float *c)
{
  char tmp[81];
  for (int i=0; i<81; i++) tmp[i] = 0;
  strncpy(tmp, record, 80);
  tmp[80] = 0;
  char ch, *s = tmp;

  s = tmp+6 ;          ch = tmp[15]; tmp[15] = 0;
  *a = atof(s);
  s = tmp+15; *s = ch; ch = tmp[24]; tmp[24] = 0;
  *b = atof(s);
  s = tmp+24; *s = ch; ch = tmp[33]; tmp[33] = 0;
  *c = atof(s);
  s = tmp+33; *s = ch; ch = tmp[40]; tmp[40] = 0;
  *alpha = atof(s);
  s = tmp+33; *s = ch; ch = tmp[47]; tmp[47] = 0;
  *beta = atof(s);
  s = tmp+47; *s = ch; ch = tmp[54]; tmp[54] = 0;
  *gamma = atof(s);
}

/* Extract the x,y, and z coordinates from the given ATOM record.	*/
void get_pdb_coordinates(char *record, float *x, float *y, float *z,
	float *occup, float *beta) {
  char numstr[9];

  /* get X, Y, and Z */
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 30, 8);
  *x = atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 38, 8);
  *y = atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 46, 8);
  *z = atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 54, 6);
  *occup = atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 60, 6);
  *beta = atof(numstr);
}

  
/* Break a pdb ATOM record into its fields.  The user must provide the
   necessary space to store the atom name, residue name, and segment name.
   Character strings will be null-terminated.  Returns the atom serial number.
*/
int get_pdb_fields(char *record, char *name, char *resname, char *chain, char *segname, 
   char *resid, float *x, float *y, float *z, float *occup, float *beta) {
  int i,len, num;

  /* get serial number */
  if (record[6] >= 'A' && record[6] <= 'Z') {
    // If there are too many atoms, XPLOR uses 99998, 99999, A0000, A0001, ...
    int base = (int(record[6] - 'A') + 10) * 100000;
    sscanf(record + 6, "%d", &num);
    num += base;
  } else {
    sscanf(record + 6,"%d",&num);
  }

  /* get atom name */
  strncpy(name,record + 12, 4);
  name[4] = '\0';
  while((len = strlen(name)) > 0 && name[len-1] == ' ')
    name[len-1] = '\0';
  while(len > 0 && name[0] == ' ') {
    for(i=0; i < len; i++)  name[i] = name[i+1];
    len--;
  }

  /* get residue name */
  strncpy(resname,record + 17, 4);
  resname[4] = '\0';
  while((len = strlen(resname)) > 0 && resname[len-1] == ' ')
    resname[len-1] = '\0';
  while(len > 0 && resname[0] == ' ') {
    for(i=0; i < len; i++)  resname[i] = resname[i+1];
    len--;
  }

  chain[0] = record[21];
  chain[1] = 0;

  /* get residue id number */
  strncpy(resid,record + 22, 4);
  resid[4] = '\0';
  while((len = strlen(resid)) > 0 && resid[len-1] == ' ')
    resid[len-1] = '\0';
  while(len > 0 && resid[0] == ' ') {
    for(i=0; i < len; i++)  resid[i] = resid[i+1];
    len--;
  }

  /* get x, y, and z coordinates */
  get_pdb_coordinates(record, x, y, z, occup, beta);

  /* get segment name	*/
  if(strlen(record) >= 73) {
    strncpy(segname, record + 72, 4);
    segname[4] = '\0';
    while((len = strlen(segname)) > 0 && segname[len-1] == ' ')
      segname[len-1] = '\0';
    while(len > 0 && segname[0] == ' ') {
      for(i=0; i < len; i++)  segname[i] = segname[i+1];
      len--;
    }
  } else {
    strcpy(segname,"");
  }
   
  return num;
}  

