/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Remote.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.18 $	$Date: 1995/11/04 02:56:45 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Remote object, which maintains a connection to a remote computer that
 * provides data and structures.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include "Remote.h"
#include "Inform.h"
#include "utilities.h"

#include "MDCommUserCommands.h"

// static string storage
static char *no_err_msg = "No error.";
static char *default_rapp_pass = "rappVMDp";

// static status storage
static char *statusStrings[Remote::TOTAL_STATUS] = {
	"No Connection", "Selecting Application",
	"Editing Parameters", "Running Simulation" };

// string descriptions, and types, for available simulation settings
char *remoteSettingName[Remote::TOTAL_SETTINGS] = {
	"Trans Rate: ", " Keep Rate:" };
char *remoteSettingKeyword[Remote::TOTAL_SETTINGS] = {
	"rate", "keep" };
char remoteSettingType[Remote::TOTAL_SETTINGS] = { 'i', 'i' };


// static flags for this class
int Remote::run_mdc_init = FALSE;

/****************************************************************************
 * NOTES:
 *
 * A remote connection proceeds in three parts:
 *	1) Make initial connection to remote manager.  This requires setting
 * the remote machine and username, and is done in the constructor.  This
 * results in this object getting a list of possible remote applications (if
 * new app is to be started), or a list of running apps (if a connection to
 * a running program is to be made).
 *
 *	2) If a new app is to be started, a parameter list is fetched and
 * edited.  If a connection to running app is required, this step is not
 * necessary.
 *
 *	3) Connect to the program, possible starting it first.
 ****************************************************************************/

///////////////////////////  constructor  
Remote::Remote(char *newuser, char *machine) {
  char *envtxt;

  // make copies of machine and username
  computer = stringdup(machine);
  username = stringdup(newuser);

  // determine the consumer program to use
  if((envtxt = getenv("MDC_CONSUMER")) != NULL) {
    consumerProg = stringdup(envtxt);
    msgWarn << "Remote: Using consumer program " << consumerProg << sendmsg;
  } else {
    consumerProg = NULL;
    msgWarn << "Remote: NO CONSUMER PROGRAM SPECIFIED." << sendmsg;
    msgWarn << "        Cannot use remote connection capabilities."<< sendmsg;
    msgWarn << "        Set MDC_CONSUMER to full path of consumer program.";
    msgWarn << sendmsg;
  }

  // initialize our internal variables
  startingNewApp = TRUE;		// default is to start a new app
  editingParams = FALSE;		// no parameters retrieved yet
  madeConnection = FALSE;		// no connection yet either
  application = (-1);			// if < 0, no app defined yet
  availApps = 0;
  availJobs = 0;
  availOptions = 0;
  returnValue = RAPP_EOK;
  remoteStatus = NO_CONNECT;
  
  // modifiable parameters
  transferRate = 1;
  saveFrameRate = 0;
  
  // initialization for mdcomm library
  proglist = NULL;
  joblist = NULL;
  optlist = NULL;
  consumer = NULL;

  if(!run_mdc_init) {
    if(rapp_init() != 0) {
      msgErr << "Cannot initialize connection library." << sendmsg;
      remoteStatus = CONNECT_ERROR;
      returnValue = RAPP_EINIT;
    } else {
      run_mdc_init = TRUE;
    }
  }

  // get data about available jobs and applications
  if(returnValue == RAPP_EOK) {
    remoteStatus = SELECTING_APP;

    // get list of apps
    if((proglist = rapp_available_apps(computer)) == NULL) {
      msgErr << "Cannot retrieve available application from " << computer;
      msgErr << "." << sendmsg;
      remoteStatus = CONNECT_ERROR;
      returnValue = rapp_errno;
    } else {
      // find number of applications in list
      struct rapp_proglist *proglistscan = proglist;
      while(proglistscan) {
        availApps++;
	proglistscan = proglistscan->next;
      }
    }
    
    // get list of running jobs
    if(returnValue == RAPP_EOK) {
      if((joblist = rapp_ps(computer, username)) == NULL) {
        // there are no running jobs for the user that can be found
	msgWarn << "There are no simulations running on " << computer;
	msgWarn << " for user " << username << "." << sendmsg;
      } else {
        // find number of running applications
	struct rapp_joblist *joblistscan = joblist;
        while(joblistscan) {
          availJobs++;
	  joblistscan = joblistscan->next;
        }
      }
    }
  }
}


///////////////////////////  destructor  
Remote::~Remote(void) {
  // free up any other mdcomm-allocated space
   proglist = NULL;
  joblist = NULL;

  // kill the processes, if they are running
  remote_close(FALSE);
  
  // free other allocated storage
  if(username) delete [] username;
  if(computer) delete [] computer;
  if(consumerProg) delete [] consumerProg;
}


//////////////////////////  public routines  

// return the currently selected application name.  NULL if error.
char *Remote::app(void) {

  // are we running?
  if(application < 0)
    return NULL;
    
  // running from previous job, or new job?
  if(startingNewApp)
    return avail_app_name(application);
  else
    return running_job_name(application);
}


// current status of connection (not the same as the error status string)
char *Remote::status(void) {
  return statusStrings[remoteStatus];
}


// name of molecule to simulate (taken from STR keyword, if available)
char *Remote::mol_name(void) {
  // check for STR keyword
  for(int i=0; i < options(); i++) {
    if(!strcmp(option_keyword(i), "STR"))
      return option_default(i);
  }
  
  // if here, not found ... return NULL
  return NULL;
}


// get parameters, for the Nth available app.  Return success.
// This can only work properly if we have just created this object, and
// have not yet selected an app to work with.  This is used to start a new
// job, not to connect to a running one.
int Remote::get_parameters(int n) {

  // are we running?
  if(application >= 0 || n < 0 || n >= availApps)
    return FALSE;
    
  // get parameters, if possible
  if((optlist = rapp_get_usage(computer, avail_app_name(n))) != NULL) {
    // got parameters, switch to editing parameters state
    availOptions = optlist->count;
    application = n;
    startingNewApp = TRUE;
    editingParams = TRUE;
    madeConnection = FALSE;
    remoteStatus = EDITING_PARAMS;
    returnValue = RAPP_EOK;
    return TRUE;
  }
  
  // if here, could not get options
  returnValue = rapp_errno;
  return FALSE;
}


// attach to the Nth running job.  Return success.
// This can only work properly if we have just created this object, and
// have not yet selected an app to work with.  This is used to connect to a
// previously-running job, instead of starting a new one.
int Remote::attach_to_job(int n) {

  // are we running?
  if(application >= 0 || n < 0 || n >= availJobs)
    return FALSE;
    
  // make connection, if possible
  struct rapp_joblist *joblistscan = joblist;
  int jobcount = 0;
  while(jobcount++ < n)
    joblistscan = joblistscan->next;
msgWarn << "Attaching to job " << n << " on " << computer << " using port ";
msgWarn << (joblistscan->job)->port << sendmsg;
  consumer = rapp_connect(computer,
		(joblistscan->job)->port, 
		default_rapp_pass,
		mdcomm_recv_static,
		mdcomm_send_static,
		&staticData,
		&dynamicData,
		transferRate,
		consumerProg,
		mdcomm_mem_calc,
		mdcomm_map_shmem);

  // see if connection was successful
  if(consumer) {
msgWarn << "Attaching successful." << sendmsg;
    // connection was successful
    application = n;
    startingNewApp = FALSE;
    editingParams = FALSE;
    madeConnection = TRUE;
    remoteStatus = CONNECTED;
    returnValue = RAPP_EOK;
    return TRUE;
  }

  // if here, connection failed
msgWarn << "Attaching failed.  Error: " << rapp_errno << sendmsg;
  returnValue = rapp_errno;
  return FALSE;
}


// if a new simulation is being run, and get_parameters has been called with
// the parameters all set up, this makes a connection and runs the sim.
// return success.
int Remote::start_new_simulation(void) {

  // are we currently editing parameters?
  if(!editingParams)
    return FALSE;
    
  // make a connection using the current parameters
msgWarn << "Starting new job on " << computer << sendmsg;
  consumer = rapp_exec_and_connect(computer,
		username,
		NULL,
		default_rapp_pass,
		optlist,
		mdcomm_recv_static,
		mdcomm_send_static,
		&staticData,
		&dynamicData,
		transferRate,
		consumerProg,
		mdcomm_mem_calc,
		mdcomm_map_shmem);

  if(consumer) {
msgWarn << "Connect successful." << sendmsg;
    // connection was successful
    editingParams = FALSE;
    madeConnection = TRUE;
    remoteStatus = CONNECTED;
    returnValue = RAPP_EOK;
    return TRUE;
  }
  
  // if here, connection failed
msgWarn << "Connect failed.  Error: " << rapp_errno << sendmsg;
  returnValue = rapp_errno;
  return FALSE;
}


// close: finishes the connection, return to original state
// if arg = TRUE, and an application is running, this will disconnect from
// the job but let it keep running.  Otherwise, this will kill the remote
// application.
void Remote::remote_close(int keepRunning) {

  // are we currently connected?
  if(madeConnection)  {
    // do we let the remote application keep running?
    if(keepRunning)
      rapp_disconnect(consumer);
    else
      rapp_kill(consumer);

    consumer = NULL;
  }

  // deallocate memory for other lists
  if(optlist)
    rapp_optlist_dealloc(optlist);
  optlist = NULL;
  availOptions = 0;

  // return variables to initial state
  startingNewApp = TRUE;		// default is to start a new app
  editingParams = FALSE;		// no parameters retrieved yet
  madeConnection = FALSE;		// no connection yet either
  application = (-1);			// if < 0, no app defined yet
  returnValue = RAPP_EOK;
  remoteStatus = NO_CONNECT;
  transferRate = 1;
  saveFrameRate = 10;

  return;
}


// return index of for given setting, -1 if error
int Remote::setting_index(char *p) {
  int n = 0;
  while(n < avail_settings()) {
    if(!strcmp(p, remoteSettingKeyword[n]))
      return n;
    n++;
  }
  return (-1);
}


// change the value of the specified setting.  return success.
int Remote::change_setting(char *p, char *newval) {
  int n;

  // can only do so if sim is running and n is valid
  if(madeConnection && (n = setting_index(p)) >= 0) { 
    if(n == TRANSFER_RATE) {
      int nv = atoi(newval);
      rapp_set_xfer_interval(consumer, nv);
      transferRate = nv;
    } else if(n == SAVEFRAME_RATE) {
      saveFrameRate = atoi(newval);
    } else
      return FALSE;		// unknown setting
  } else
    return FALSE;		// no connection, or bad parameter
    
  return TRUE;
}


// write the value of the specified setting into the given string storage.
// return success.
int Remote::get_setting(char *p, char *newval) {
  int n;

  // can only do so if n is valid
  if((n = setting_index(p)) >= 0) { 
    if(n == TRANSFER_RATE)
      sprintf(newval, "%d", transferRate);
    else if(n == SAVEFRAME_RATE)
      sprintf(newval, "%d", saveFrameRate);
    else
      return FALSE;		// unknown setting
  } else
    return FALSE;		// bad parameter
    
  return TRUE;
}


// write the value of the specified setting into the given string storage.
// return success.
int Remote::get_setting(int n, char *newval) {
  // can only do so if n is valid
  if(n == TRANSFER_RATE)
    sprintf(newval, "%d", transferRate);
  else if(n == SAVEFRAME_RATE)
    sprintf(newval, "%d", saveFrameRate);
  else
    return FALSE;		// unknown setting
    
  return TRUE;
}


// process list of options of the form "<keyword> = <value>" in a file
// use config routines to process file data
// return success.
int Remote::read_param_file(FILE *f) {
  int argc, slen, retval;
  char *tokstr, *argv[64];
  char strbuf[256];

  if(!optlist)
    return FALSE;

  // loop over the file, reading until EOF
  while(fgets(strbuf,255,f)) {
    if((slen=strlen(strbuf)) > 0) {
      if(strbuf[slen-1] == '\n')
        strbuf[slen-1] = '\0';
    }
    if((tokstr = command_tokenize(strbuf, &argc, argv)) != NULL) {
      retval = set_option(argv[0], argv[2]);
      delete [] tokstr;
      if(!retval)
        return FALSE;
    }
  }
  
  return TRUE;
}


// write current list of options to a file, in form "<keyword> = <value>"
// return success.
int Remote::write_param_file(FILE *f) {
  int i;
  
  if(!optlist)
    return FALSE;
  
  // only write parameters that have assigned values to the file
  for(i=0; i < optlist->count; i++) {
    if((optlist->opt[i])->def)
      fprintf(f,"%s = %s\n",(optlist->opt[i])->keyword,(optlist->opt[i])->def);
  }
  
  return TRUE;
}


// check for a new timestep, and return TRUE if available.
int Remote::next_ts_available(void) {
  int retval;
  
  if(madeConnection && returnValue == RAPP_EOK) {
    if((retval = rapp_probe(consumer)) == RAPP_EOK) {
      MSGDEBUG(1,"A timestep is available from '" << host() << "' ... ");

      return TRUE;
    } else if(retval == (-1)) {
      returnValue = rapp_errno;
      remoteStatus = READ_ERROR;
      msgErr << "Problem encounted while probing for new timestep from ";
      msgErr << host() << ":\n   " << error_status_desc() << sendmsg;
    } else {
      MSGDEBUG(2,"No timestep is available from '" << host() << "' ... ");
    }
  }
      
  // if here, problem
  return FALSE;
}


// gets timestep data, if available.  returns pointer to dynamic
// data structure if successful, NULL otherwise.
VmdDynData *Remote::get_next_ts(void) {

  if(next_ts_available()) {
    rapp_update(consumer);
    remoteStatus = CONNECTED;
    
    MSGDEBUG(1,"read new timestep." << sendmsg);

    return dynamic_data();
  }
  
  // if here, no data available
  return NULL;
}


// signal a new timestep can be sent.  Should be called after get_next_ts
// has returned true, and the data has been fetched properly.
void Remote::done_with_ts(void) {
  // this just assumes a timestep has been fetched with get_next_ts previously
  rapp_freebuf(consumer);
}


// query return status description
char *Remote::error_status_desc(int st) {
  if(st == RAPP_EOK)
    return no_err_msg;
  else
    return rapp_errlist[st];
}


// query return status description of current error
char *Remote::error_status_desc(void) {
  if(returnValue == RAPP_EOK)
    return no_err_msg;
  else
    return rapp_errlist[returnValue];
}


// return pid and uid of the Nth job
void Remote::running_job_ids(int n, int &pid, int &uid) {

  if(n < 0 || n >= availJobs)
    return;
    
  struct rapp_joblist *joblistscan = joblist;
  int jobcount = 0;
  while(jobcount++ < n)
    joblistscan = joblistscan->next;

  pid = (joblistscan->job)->jid;
  uid = (joblistscan->job)->uid;
}


// return name of Nth available job's program
char *Remote::running_job_name(int n) {

  if(n < 0 || n >= availJobs)
    return NULL;
    
  struct rapp_joblist *joblistscan = joblist;
  int jobcount = 0;
  while(jobcount++ < n)
    joblistscan = joblistscan->next;

  return (joblistscan->job)->prog;
}


// return name of Nth available job's program
char *Remote::avail_app_name(int n) {

  if(n < 0 || n >= availApps)
    return NULL;
    
  struct rapp_proglist *proglistscan = proglist;
  int appcount = 0;
  while(appcount++ < n)
    proglistscan = proglistscan->next;

  return proglistscan->prog;
}


// strings used by option_string routine
static char *optiontypeString[3] = { "f", "i", "s" };
static char formattedOptionString[256];


// complete string description of Nth option, including keyword, value,
// required, and type
char *Remote::option_string(int n) {

  if(n < 0 || n >= availOptions) {
    strcpy(formattedOptionString, "");
  } else {
    sprintf(formattedOptionString, "%12.12s: %30.30s %s %s ==> %s",
    	option_keyword(n), option_desc(n), (option_req(n) ? "REQ" : "opt"),
	optiontypeString[option_type(n)],
	(option_default(n) ? option_default(n) : ""));
  }
  
  return formattedOptionString;
}

void Remote::send_new_coords(int num, int *indicies, float *fx,
				  float *fy, float *fz)
{
  msgInfo << "Sending " << num << " new coords" << sendmsg;
  for (int i=0; i<num; i++) {
    msgInfo << i << " " << indicies[i] << " " << fx[i] << " " << fy[i] 
	    << " " << fz[i] << sendmsg;
  }
}
void Remote::send_new_forces(int num, int *indicies, float *fx,
			     float *fy, float *fz)
{
  //  msgInfo << "Sending " << num << " new forces" << sendmsg;
  //  for (int i=0; i<num; i++) {
  //    msgInfo << i << " " << indicies[i] << " " << fx[i] << " " << fy[i] 
  //	    << " " << fz[i] << sendmsg;
  //  }

  // and send the data to MDComm
  mdcomm_send_force_cmd(consumer, num, indicies, fx, fy, fz);
}


