/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: SBDisplayDevice.C,v $
 *	$Author: leech $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $		$Date: 95/07/26 03:16:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that are Starbase-specific.  Will render drawing
 * commands into a single X11 window being used by Starbase, currently in
 * window-dumb mode.
 *
 ***************************************************************************/
#include <stdlib.h>
#include <math.h>
#include <tclExtend.h>
#include "SBDisplayDevice.h"
#include "UITk.h"
#include "Inform.h"
#include "Global.h"
#include "utilities.h"

// The #define works a problem with CC +a1 vs. <starbase.c.h> (header
//  assumes float promotion in parameter passing otherwise).
#define __STDC__
#include <starbase.c.h>

void zclear() { /* does nothing */ }
void viewport(short,short,short,short) { /* does nothing */ }

#include <stdio.h>
extern "C" int matherr(struct exception *x)
{
    fprintf(stderr, "matherr: %d exception (%s)\n", x->type, x->name);
    return 0;
}

/////////////////////////  constructor and destructor

// ID # of the next Starbase window to be created
int SBDisplayDevice::next_tkid = 0;

// constructor ... open a window and set initial default values
SBDisplayDevice::SBDisplayDevice(int *size, int *loc)
	: SBRenderer("SB Display") {
    MSGDEBUG(1, "Creating SBDisplayDevice ... " << sendmsg);

    // Cache the specified initial window size and position
    if (size) {
	wsize[0] = size[0];
	wsize[1] = size[1];
    } else
	wsize[0] = wsize[1] = -1;

    if (loc) {
	wloc[0] = loc[0];
	wloc[1] = loc[1];
    } else
	wloc[0] = wloc[1] = -1;

    // Don't do anything else until the bind_Tk() method is called,
    //	giving a Tk interpreter in which to set up the window.
    uitk = NULL;
    tkid = -1;
}

// destructor ... close the window
SBDisplayDevice::~SBDisplayDevice(void) {
    if (interp && tkid >= 0) {
	char cmd[100];
	sprintf(cmd, "vmd::close_sbwin %d", tkid);
	Tcl_Eval(interp, cmd);
	tkid = -1;
	fd = -1;
    }
}

/////////////////////////  protected nonvirtual routines

// create a new window and set its characteristics
void SBDisplayDevice::open_window(char *name, int *size, int *loc) {
    long retval = 0;
    int i;

    MSGDEBUG(2,"SBDisplayDevice: opening window ..." << sendmsg);

    if (!interp)
	VMDexit("No Tk interpreter available to create Starbase window!", -1);

    if (tkid != -1)
	VMDexit("Starbase window already created!", -1);

    // Allocate a unique (within the class) identifier for the
    //	Starbase window, used in constructing its Tk pathname.
    tkid = next_tkid++;

    // Execute a Tcl command to create the window:
    //	vmd::open_sbwin id label width height xpos ypos
    char cmdbuf[200];
    sprintf(cmdbuf, "vmd::open_sbwin %d {%s} %d %d %d %d",
	tkid, name, size[0], size[1], loc[0], loc[1]);
    if (Tcl_Eval(interp, cmdbuf) != TCL_OK)
	VMDexit("Tk failed to create Starbase window!", -1);

    // If the command succeeded, the interpreter's command string
    //	contains a file descriptor followed by the current window
    //	width and height.
    if (sscanf(interp->result, "%d %d %d", &fd, &width, &height) != 3) {
	fd = -1;
	tkid = -1;
	VMDexit("Cannot determine fd and width/height for Starbase window!", -1);
    }

    // Determine device limits
    float limits[2][3], res[3], p1[3], p2[3];
    int cmap_size;
    inquire_sizes(fd, limits, res, p1, p2, &cmap_size);

    maxwidth = int(1 + limits[1][0] - limits[0][0]);
    maxheight = int(1 + fabs(limits[1][1] - limits[0][1]));
printf("open_window: maxwidth = %d maxheight = %d width = %d height = %d\n",
    maxwidth, maxheight, width, height);

    // Set up p1-p2 mapping
    configure(width, height);

    // Turn off dbuffering and zbuffering prior to fiddling with projection
    //	and HSR setup, so Starbase doesn't complain.
    hidden_surface(fd, FALSE, FALSE);
    double_buffer(fd, FALSE, 0);

    shade_mode(fd, CMAP_FULL, TRUE);
    double_buffer(fd, TRUE, 12);	// Turn double-buffering on
    dbuffer_switch(fd, display_buffer+1);   // Write into this buffer initially

    vdc_extent(fd, -1,-1,-1, 1,1,1);	// canonical NDC space
    clear_control(fd, CLEAR_DISPLAY_SURFACE|CLEAR_ZBUFFER);

    hidden_surface(fd, TRUE, FALSE);	// zbuffer yes, cull no
    background_color(fd, .4, .2, 0);
    clear_view_surface(fd);
    line_color(fd, 1, 1, 1);
    fill_color(fd, 1, 1, 1);

    light_ambient(fd, 0.5, 0.5, 0.5);	// make sure we see *something*
    light_switch(fd, 1);		// enable ambient light
    interior_style(fd, INT_SOLID, 0);	// filled, no edges drawn
    bf_control(fd, 1, 0);               // flip normals if needed
    // coord=0	-> no extra coords
    // use=0	-> no extras for color
    // rgb=0	-> no color
    // normals=1-> per-polygon normals
    // CLOCKWISE-> vertex order
    vertex_format(fd, 0, 0, 0, 1, CLOCKWISE);
    flush_matrices(fd);

    // turn on lights if necessary
    for (i = 0; i < DISP_LIGHTS; i++) {
	if (lightDefined[i]) {
	    do_define_light(i, lightColor[i], lightPos[i]);
	    do_activate_light(i, lightOn[i]);
	} else
	    do_activate_light(i, FALSE);
    }

    // define materials if necessary
    for (i = 0; i < MAXCOLORS; i++) {
	if (matDefined[i])
	    do_define_material(i, matData[i]);
    }
    do_activate_material(materialOn, materialsActive);

    // load current transformation matrix on stack
    this->loadmatrix(transMat.top());

    set_sphere_mode(sphereMode);
    set_sphere_res(sphereRes);
    set_line_width(lineWidth);
    set_line_style(lineStyle);

    // Do something about depth cueing for Starbase...
    return;
}

// set the current perspective, based on the eye position and where we
// are looking.  This form is for a non-stereo perspective
void SBDisplayDevice::set_persp(DisplayEye my_eye) {
    if (dim() == 3) {	   // use window perspective for 3D view
	camera_arg cam;

	//window((Coord)cpLeft, (Coord)cpRight, (Coord)cpDown, (Coord)cpUp,
	//	 (Coord)nearClip, (Coord)farClip);

	// ignore stereo, for now (see GLDD.C)
	switch (my_eye) {
	};

	cam.camx = eyePos[0];
	cam.camy = eyePos[1];
	cam.camz = eyePos[2];
	cam.refx = eyePos[0] + eyeDir[0];
	cam.refy = eyePos[1] + eyeDir[1];
	cam.refz = eyePos[2] + eyeDir[2];
	cam.field_of_view = 90;
	cam.front = -.9;    // assumes eyeDir is normalized
	cam.back = 20;	    // should make use of near/farClip
	cam.upx = 0;
	cam.upy = 1;
	cam.upz = 0;
	cam.projection = CAM_PERSPECTIVE;

	view_camera(fd, &cam);
    } else {			 // use (0 .. 1, 0 ..1) window for 2D view
	view_window(fd, 0, 0, 1, 1);
    }
}

/////////////////////////  public virtual routines

//
// virtual routines for preparing to draw, drawing, and finishing drawing
//

// reshape the display after a shape change
void SBDisplayDevice::reshape(void) {
    msgErr << "SBDisplayDevice::reshape: not implementable (use configure)!"
	    << sendmsg;
}

// reshape the display (size is obtained elsewhere, since Starbase
//  can't inquire about this).
void SBDisplayDevice::configure(int w, int h) {
    width = (w > maxwidth) ? maxwidth : w;
    height = (h > maxheight) ? maxheight : h;

    xSize = width;
    ySize = height;

    float
	width_fraction = float(width) / maxwidth,
	height_fraction = float(height) / maxheight;

printf("SBDD:configure: w = %d h = %d wf = %f hf = %f\n",
    width, height, width_fraction, height_fraction);

    hidden_surface(fd, FALSE, FALSE);
    set_p1_p2(fd, FRACTIONAL,
	    0.0, 1.0 - height_fraction, 0.0,
	    width_fraction, 1.0, 1.0);
    hidden_surface(fd, TRUE, FALSE);
}

// clear the display
void SBDisplayDevice::clear(void) {
    MSGDEBUG(3, "SBDisplayDevice: clearing display." << sendmsg);

    background_color(fd, backColor[0], backColor[1], backColor[2]);
    clear_view_surface(fd);
}

// prepare to draw a 2D image
void SBDisplayDevice::prepare2D(int do_clear) {
    MSGDEBUG(3, "SBDisplayDevice: preparing to draw 2D." << sendmsg);

    Dim = 2;
    if (do_clear)
	clear();
    else
	zclear();
}

// prepare to draw a 3D image
void SBDisplayDevice::prepare3D(int do_clear) {
    MSGDEBUG(3, "SBDisplayDevice: preparing to draw 3D." << sendmsg);

    Dim = 3;
    if (do_clear)
	clear();
    else
	zclear();
}

// set up for normal (non-stereo) drawing.  Sets the viewport and perspective.
void SBDisplayDevice::normal(void) {
    viewport(0, (short)xSize, 0, (short)ySize);
    set_persp();
}

// update after drawing
void SBDisplayDevice::update(int do_update) {
    MSGDEBUG(3, "SBDisplayDevice: updating after drawing." << sendmsg);

    if (do_update) {
	// Write into display buffer, display former write buffer
	dbuffer_switch(fd, display_buffer);
	// Keep track of current display buffer
	display_buffer = !display_buffer;
    }
}

// Bind a UIObject connected to a Tk interpreter to this DisplayDevice,
//  using the interpreter to set up the Starbase window.
void SBDisplayDevice::bind_Tk(UITk *ui)
{
    uitk = ui;
    interp = uitk->tcl_interp();

    // open the window
    open_window(name, wsize, wloc);

    // set flags for the capabilities of this display
    has2D = TRUE;
    aaAvailable = FALSE;
    cueingAvailable = TRUE;

    // turn on antialiasing, turn off depth-cueing
    aa_on();
    cueing_off();

    // reshape and clear the display, which initializes some other variables
    reshape();
    normal();
    clear();
    update();
}

