/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: SBRenderer.C,v $
 *	$Author: leech $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $		$Date: 95/07/26 03:16:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that are Starbase-specific.  Will render drawing
 * commands into a single window.  This is not the complete definition,
 * however, of a DisplayDevice; something must provide routines to open
 * windows, reshape, clear, set perspective, etc.  This object contains the
 * code to render a display command list.
 *
 ***************************************************************************/
#include <stdlib.h>
#include <math.h>
#include "SBRenderer.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"
// The #define works a problem with CC +a1 vs. <starbase.c.h> (header
//  assumes float promotion in parameter passing otherwise).
#define __STDC__
#include <starbase.c.h>

static const int ptSize = 3 * sizeof(float);

/////////////////////////  constructor and destructor

// constructor ... initialize some variables
SBRenderer::SBRenderer(char *nm) : DisplayDevice(nm) {
    need_aaon = need_cueon = FALSE;
    dataBlock = NULL;
    fd = -1;			// Haven't opened the device yet
    maxwidth = maxheight = 1;	// Don't know device limits or current size
    width = height = 1;
    lightmask = 0;		// Haven't turned on any lights yet
    text_pos[0] =
    text_pos[1] =
    text_pos[2] = 0.0;
    display_buffer = 0;		// The buffer being displayed

    MSGDEBUG(1,"Creating SBRenderer ..." << sendmsg);
}

// destructor ... nothing to do, really
SBRenderer::~SBRenderer(void) { }

/////////////////////////  protected nonvirtual routines

// change current line width
void SBRenderer::set_line_width(int w) {
    if(w > 0) {
	// linewidthf((float)w);
	lineWidth = w;
    }
}

// change current line style ... styles defined in constructor
void SBRenderer::set_line_style(int s) {
    switch (s) {
	case ::DASHEDLINE:
	    line_type(fd, DASH);
	    break;
	case ::SOLIDLINE:
	default:
	    line_type(fd, SOLID);
	    s = ::SOLIDLINE;
	    break;
    }
    lineStyle = s;
}

// change current sphere resolution
void SBRenderer::set_sphere_res(int r) {
    if(r >= 1 && r <= 10) {
	// Starbase call here?
	sphereRes = r;
    }
}

// change current sphere type
void SBRenderer::set_sphere_mode(int m) {
    sphereMode = m;
    // Starbase call here?
    // m = SPH_MESH, SPH_POINT
}

// this routine draws a triangle between three points
// assumes vertex_format(coord=0,use=0,rgb=0,normals=1,order=CLOCKWISE)
//  e.g. vertex data with per-polygon normal, no colors
// this routine draws a triangle between three points
void SBRenderer::triangle(float *p1, float *p2, float *p3,
			  float *n1, float * /*n2*/, float * /*n3*/) {
    float buf[3+3*3];	// normal, vertices (just one normal, for now)

    memcpy(buf, n1, ptSize);
    memcpy(&buf[3], p1, ptSize);
    memcpy(&buf[6], p2, ptSize);
    memcpy(&buf[9], p3, ptSize);

    polygon3d(fd, buf, 3, 0);
}

// this routine draws a rectangle between 4 points
// assumes vertex_format(coord=0,use=0,rgb=0,normals=1,order=CLOCKWISE)
//  e.g. vertex data with per-polygon normal, no colors
void SBRenderer::square(float *perp, float *p1, float *p2, float *p3,
			float *p4) {
    float buf[3+4*3];	// normal, vertices

    memcpy(buf, perp, ptSize);
    memcpy(&buf[3], p1, ptSize);
    memcpy(&buf[6], p2, ptSize);
    memcpy(&buf[9], p3, ptSize);
    memcpy(&buf[12], p4, ptSize);

    polygon3d(fd, buf, 4, 0);
}


// this routine draws a cylinder from end to start, using rod_res panels,
//	and of radius rod_radius
void SBRenderer::cylinder(float *end, float *start, int rod_res,
			  float rod_radius) {
  register int i, h;
  register float hresinc, rodrad, length, theta, m, n;
  float axis[3], lenaxis[3], perp[3], perp2[3];
  float norms1[4], norms2[4], vert1[4], vert2[4], vert3[4], vert4[4];

  // need to do some preprocessing
  lenaxis[0] = start[0] - end[0];
  lenaxis[1] = start[1] - end[1];
  lenaxis[2] = start[2] - end[2];

  m = lenaxis[0]*lenaxis[0] + lenaxis[1]*lenaxis[1] + lenaxis[2]*lenaxis[2];
  if (m <= 0.0)
    return;
  length = sqrtf(m);
  axis[0] = lenaxis[0] / length;
  axis[1] = lenaxis[1] / length;
  axis[2] = lenaxis[2] / length;

  // dot product with unit axis vector
  if ((ABS(axis[0]) < ABS(axis[1])) &&
      (ABS(axis[0]) < ABS(axis[2])))
    i = 0;
  else if ((ABS(axis[1]) < ABS(axis[2])))
    i = 1;
  else
    i = 2;
  perp[i] = 0;
  perp[(i+1)%3] = axis[(i+2)%3];
  perp[(i+2)%3] = -axis[(i+1)%3];
  m = sqrtf(perp[0]*perp[0] + perp[1]*perp[1] + perp[2]*perp[2]);
  if(m > 0.0) {
    perp[0] /= m;
    perp[1] /= m;
    perp[2] /= m;
  }

  // get cross product; perp2 will be normalized automatically
  perp2[0] = axis[1]*perp[2] - axis[2]*perp[1];
  perp2[1] = axis[2]*perp[0] - axis[0]*perp[2];
  perp2[2] = axis[0]*perp[1] - axis[1]*perp[0];

  // do the polygons
  rodrad = rod_radius;
  hresinc = TWOPI / ((float)rod_res);
  theta = 0.0;
  for (h=0; h < rod_res; h++) {

      if(h==0) {
	m = cos(theta);
	n = sin(theta);
	norms1[0] = m*perp[0] + n*perp2[0];
	norms1[1] = m*perp[1] + n*perp2[1];
	norms1[2] = m*perp[2] + n*perp2[2];
      } else {
	norms1[0] = norms2[0];
	norms1[1] = norms2[1];
	norms1[2] = norms2[2];
      }

      theta += hresinc;
      m = cos(theta);
      n = sin(theta);
      norms2[0] = m*perp[0] + n*perp2[0];
      norms2[1] = m*perp[1] + n*perp2[1];
      norms2[2] = m*perp[2] + n*perp2[2];

      if(h==0) {
	vert1[0] = end[0] + rodrad * norms1[0];
	vert1[1] = end[1] + rodrad * norms1[1];
	vert1[2] = end[2] + rodrad * norms1[2];
	vert2[0] = vert1[0] + lenaxis[0];
	vert2[1] = vert1[1] + lenaxis[1];
	vert2[2] = vert1[2] + lenaxis[2];
      } else {
	vert1[0] = vert4[0];
	vert1[1] = vert4[1];
	vert1[2] = vert4[2];
	vert2[0] = vert3[0];
	vert2[1] = vert3[1];
	vert2[2] = vert3[2];
      }

      vert3[0] = start[0] + rodrad * norms2[0];
      vert3[1] = start[1] + rodrad * norms2[1];
      vert3[2] = start[2] + rodrad * norms2[2];
      vert4[0] = vert3[0] - lenaxis[0];
      vert4[1] = vert3[1] - lenaxis[1];
      vert4[2] = vert3[2] - lenaxis[2];

      // Render the polygon
      // Should use norms1 as normal for vertices 12, norms2 for vertices 34
      float buf[3+4*3];   // normal, vertices

      memcpy(buf, norms1, ptSize);
      memcpy(&buf[3], vert1, ptSize);
      memcpy(&buf[6], vert2, ptSize);
      memcpy(&buf[9], vert3, ptSize);
      memcpy(&buf[12], vert4, ptSize);

      polygon3d(fd, buf, 4, 0);
  }
}

// this routine also draws a cylinder.	However, it assumes that
// the cylinder drawing command has precomputed the data.  This
// uses more memory, but is faster
// the data are: num == number of edges
//  edges = a normal, start, and end edge
#ifndef USE_SLOW_CYLINDERS
void SBRenderer::cylinder(int num, float *edges)
{
    if (num < 2) {return;}
    int n = num;
    float *start = edges;

    // Layout of data structure:
    //	n12 v1 v2 n34 v3  v4  n56 v5 v6 ...
    //	+0  +3 +6 +9  +12 +15 +18
    //
    //	   v2-v4-v6
    //	    |  |  |
    //	    |  |  |
    //	   v1-v3-v5
    //
    // So successive polygons are (v1,v2,v4,v3) = offsets(+3,+6,+15,+12)
    // Should use n12 as normal for vertices 12, n34 for vertices 34

    float buf[3+4*3];	// normal, vertices

    while (--n > 0) {
	memcpy(buf, edges, ptSize);
	memcpy(&buf[3], edges + 3, ptSize);
	memcpy(&buf[6], edges + 6, ptSize);
	memcpy(&buf[9], edges + 15, ptSize);
	memcpy(&buf[12], edges + 12, ptSize);

	polygon3d(fd, buf, 4, 0);

	edges += 9;
    }

    // close off the cylinder
    memcpy(buf, edges, ptSize);
    memcpy(&buf[3], edges + 3, ptSize);
    memcpy(&buf[6], edges + 6, ptSize);
    memcpy(&buf[9], start + 6, ptSize);
    memcpy(&buf[12], start + 3, ptSize);

    polygon3d(fd, buf, 4, 0);
}
#endif

// this routine draws a cone with axis from end to start, using rod_res panels,
//	and with base of radius rod_radius
void SBRenderer::cone(float start[], float end[], int rod_res,
   float rod_radius) {
    // This is not used anywhere in VMD at present, so no
    //	implementation is supplied.
    (void)start; (void)end; (void)rod_res; (void)rod_radius;
}

// draw a sphere of specified radius at pos[]
void SBRenderer::sphere(float *pos, float radius) {
    static float scale = .7071067811865475; // 1 / sqrt(2)
    //move3d(fd, p[0], p[1], p[2]);
    //draw3d(fd, p[0], p[1], p[2]);

    // for now, draw an open cylinder that inscribes the sphere
    float start[3], end[3];
    memcpy(start, pos, ptSize);
    memcpy(end, pos, ptSize);

    radius *= scale;
    start[2] += radius;
    end[2] -= radius;
    cylinder(start, end, 6, radius);
}

/////////////////////////  protected virtual routines

// define a new light source ... return success of operation
int SBRenderer::do_define_light(int n, float *color, float *position) {
    light_source(fd, n+1, DIRECTIONAL, color[0], color[1], color[2],
	position[0], position[1], position[2]);

    return TRUE;
}

// activate a given light source ... return success of operation
int SBRenderer::do_activate_light(int n, int turnon) {
    if (turnon)
	lightmask |= (1 << (n+1));
    else
	lightmask &= ~(1 << (n+1));

    light_switch(fd, lightmask);

    return TRUE;
}

// define a new material ... return success of operation
int SBRenderer::do_define_material(int n, float *data) {
    // Do nothing at all, because we know data is always this->matData[n],
    //	and Starbase doesn't have named materials
    (void)n; (void)data; return TRUE;
}

// activate the use of material characteristics ... return success
int SBRenderer::do_activate_material(int n, int turnon) {
    if (turnon) {
	if (aa_enabled()) {
	    need_aaon = TRUE;
	    aa_off();
	}
	if (cueing_enabled()) {
	    need_cueon = TRUE;
	    cueing_off();
	}
	// lmbind(MATERIAL, n+1);
    } else {
	if (need_aaon && aa_available())
	    aa_on();
	if (need_cueon && cueing_available())
	    cueing_on();
	need_aaon = need_cueon = FALSE;
	// lmbind(MATERIAL, 0);
    }

    // Set up surface model using
    //	RGB matData[n][AMBIENT_INDEX,DIFFUSE_INDEX,SPECULAR_INDEX]
    //	float matData[n][SHININESS_INDEX,ALPHA_INDEX];
    (void)n; return TRUE;
}

void SBRenderer::translate(float dx, float dy, float dz) {
    static float m[4][4] = {
	{ 1, 0, 0, 0 },
	{ 0, 1, 0, 0 },
	{ 0, 0, 1, 0 },
	{ 0, 0, 0, 1 }
    };

    m[3][0] = dx;
    m[3][1] = dy;
    m[3][2] = dz;
    concat_transformation3d(fd, m, PRE, REPLACE);
}

void SBRenderer::scale(float sx, float sy, float sz) {
    static float m[4][4] = {
	{ 1, 0, 0, 0 },
	{ 0, 1, 0, 0 },
	{ 0, 0, 1, 0 },
	{ 0, 0, 0, 1 }
    };

    m[0][0] = sx;
    m[1][1] = sy;
    m[2][2] = sz;
    concat_transformation3d(fd, m, PRE, REPLACE);
}

void SBRenderer::rot(float angle, char axis) {
    float m[4][4];
    enum { X, Y, Z, W };

    float ca, sa;
    ca = cos(angle);
    sa = sin(angle);

    for (int r = X; r <= W; r++)
	for (int c = X; c <= W; c++)
	    m[r][c] = 0;
    m[W][W] = 1.0;

    switch (axis) {
	case 'x':
	    m[X][X] = 1;
			    m[Y][Y] = ca; m[Y][Z] =-sa;
			    m[Z][Y] = sa; m[Z][Z] = ca;
	    break;
	case 'y':
	    m[X][X] = ca;		  m[X][Z] = sa;
			    m[Y][Y] = 1;
	    m[Z][X] =-sa;		  m[Z][Z] = ca;
	    break;
	case 'z':
	    m[X][X] = ca;   m[X][Y] =-sa;
	    m[Y][X] = sa;   m[Y][Y] = ca;
					  m[Z][Z] = 1;
	    break;
	default:
	    msgErr << "bad axis " << axis << sendmsg;
	    return;
    }

    concat_transformation3d(fd, m, PRE, REPLACE);
}

/////////////////////////  public virtual routines

// size of physical display object
void SBRenderer::screen_size_mm(long& x, long& y) {
    x = y = 0;
    // Apply some scaling or other
}

void SBRenderer::screen_size_pixel(long& x, long& y) {
    float limits[2][3], resolution[3], p1[3], p2[3];
    int cmap_size;

    inquire_sizes(fd, limits, resolution, p1, p2, &cmap_size);

    x = limits[1][0] - limits[0][0];
    y = limits[1][1] - limits[0][1];
}

//
// virtual routines to affect the device's transformation matrix
// These do not use the local transMat matrix, but just use the SB viewing
// matrix directly.
//

// An identity matrix
static float Identity[4][4] = {
    { 1, 0, 0, 0 },
    { 0, 1, 0, 0 },
    { 0, 0, 1, 0 },
    { 0, 0, 0, 1 }
};

void SBRenderer::push(void) {
    concat_transformation3d(fd, Identity, PRE, PUSH);
}

void SBRenderer::pop(void) {
    pop_matrix(fd);
}

void SBRenderer::loadmatrix(Matrix4 &m) {
    // This relies on gopen in MODEL_XFORM mode, so that the CTM is pushed
    //	and *replaced* by m.mat rather than being pushed and *concatenated*
    //	by m.mat
    push_matrix3d(fd, m.mat);
}

void SBRenderer::multmatrix(Matrix4 &m) {
    concat_transformation3d(fd, m.mat, PRE, REPLACE);
}

//
// virtual routines to return 2D screen coordinates, given 2D or 3D world
// coordinates.  These assume the proper SB window has focus, etc.
// The coordinates returned are absolute screen coords, relative to the lower
// left corner of the display monitor
//


void SBRenderer::abs_screen_loc_3D(float *loc, long *spos) {
    float vx, vy, vz;
    int dx, dy, dz;

    wc_to_vdc(fd, loc[0], loc[1], loc[2], &vx, &vy, &vz);
    vdc_to_dc(fd, vx, vy, vz, &dx, &dy, &dz);

    spos[0] = dx;
    spos[1] = dy;
}

void SBRenderer::abs_screen_loc_2D(float *loc, long *spos) {
    float loc3[3];

    loc3[0] = loc[0];
    loc3[1] = loc[1];
    loc3[2] = 0.0;

    abs_screen_loc_3D(loc, spos);
}


// Given a 3D point (pos A),
// and a 2D rel screen pos point (for pos B), computes the 3D point
// which goes with the second 2D point at pos B.  Result returned in B3D.
// NOTE: currently, this algorithm only supports the simple case where the
// eye look direction is along the Z-axis.  A more sophisticated version
// requires finding the plane to which the look direction is normal, which is
// assumed here to be the Z-axis (for simplicity in coding).
void SBRenderer::find_3D_from_2D(float *A3D, float *B2D, float *B3D) {
    // Aggh, don't even think about this yet
#if 0
    Screencoord lvp, rvp, bvp, tvp;	  // for current viewport
    Screencoord sx, sy;			  // screen position
    Matrix mview, mproj;		  // for current modelview and projection
    Object transobj = 1;		  // object to use to reverse-map position
    Coord w1[3], w2[3];			  // points on line returned by rev map
    float lsx, lsy;			  // used to convert rel screen -> abs

    // get current status of viewport and modelview/proj matricies
    getviewport(&lvp, &rvp, &bvp, &tvp);
    mmode(MPROJECTION);
    getmatrix(mproj);
    mmode(MVIEWING);
    getmatrix(mview);

    // create a display list object which sets these values
    makeobj(transobj);
      viewport(lvp, rvp, bvp, tvp);
      mmode(MPROJECTION);
      ::loadmatrix(mproj);
      mmode(MVIEWING);
      ::loadmatrix(mview);
    closeobj();

    // now use this object to map back from the 2D point to 2 3D points (a line)
    lsx = B2D[0];	  lsy = B2D[1];
    abs_screen_pos(lsx, lsy);
    sx = (Screencoord)lsx - (Screencoord)xOrig;
    sy = (Screencoord)lsy - (Screencoord)yOrig;
    mapw(transobj, sx, sy, w1, w1+1, w1+2, w2, w2+1, w2+2);

    // finally, find the point where line returned as w1..w2 intersects the
    // given 3D point's plane (this plane is assumed to be parallel to the X-Y
    // plane, i.e., with a normal along the Z-axis.  A more general algorithm
    // would need to find the plane which is normal to the eye look direction,
    // and which contains the given 3D point.)

    // special case: w1z = w2z ==> just return given 3D point, since there
    //		  is either no solution, or the line is in the given plane
    if(w1[2] == w2[2]) {
      memcpy((void *)B3D, (void *)A3D, 3*sizeof(float));
    } else {
      B3D[0] = (w2[0] - w1[0]) * (A3D[2] - w1[2]) / (w2[2] - w1[2]) + w1[0];
      B3D[1] = (w2[1] - w1[1]) * (B3D[2] - w1[2]) / (w2[2] - w1[2]) + w1[1];
      B3D[2] = A3D[2];
    }
#endif
}


//
// antialiasing and depth-cueing
//

// turn on antialiasing effect
void SBRenderer::aa_on(void) {
    if (!aaAvailable) {
	aaEnabled = FALSE;
    } else {
	// Starbase calls go here
	//blendfunction(BF_SA, BF_MSA);
	//pntsmooth(SMP_ON);
	//linesmooth(SML_ON);
	//polysmooth(PYSM_ON);
	aaEnabled = TRUE;
    }
}

// turn off antialiasing effect
void SBRenderer::aa_off(void) {
    if (aaAvailable) {
	// Starbase calls go here
	//pntsmooth(SMP_OFF);
	//linesmooth(SML_OFF);
	//polysmooth(PYSM_OFF);
    }
    aaEnabled = FALSE;
}

// turn on hardware depth-cueing
void SBRenderer::cueing_on(void) {
    if (!cueingAvailable) {
	cueingEnabled = FALSE;
    } else {
	// Later, use DC_COLOR and depth_cue_{color,range}
	depth_cue(fd, TRUE | DC_MIN, 0.0);
	cueingEnabled = TRUE;
    }
}

// turn off hardware depth-cueing
void SBRenderer::cueing_off(void) {
    if (cueingAvailable) {
	depth_cue(fd, FALSE, 0.0);
    }
    cueingEnabled = FALSE;
}


//
//*******************  the rendering routine  *********************
//
// This scans the given command list until the end, doing the commands
// in the order they appear
//
void SBRenderer::render(void *cmdList) {
    char *cmdptr = (char *)cmdList;
    int tok, cmdsize;

    MSGDEBUG(3, "SBRenderer: rendering command list." << sendmsg);

    if (!cmdList)
	return;

    // save transformation matrix
    push();

    // scan through the list, getting each command and executing it, until
    // the end of commands token is found
    dataBlock = NULL;
    while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
	float *fp;
	int *ip, n;

	if (tok == DLINKLIST) {
	    cmdptr += sizeof(int);	 // step forward one (no length here)
	    cmdptr = *((char **) cmdptr);// and go to the next link
	    tok = ((int *)cmdptr)[0];	 // this is guaranteed to be neither
	}				// a DLINKLIST nor DLASTCOMMAND

	cmdsize = ((int *)cmdptr)[1];
	cmdptr += 2*sizeof(int);

	MSGDEBUG(4, "SBRenderer: doing command " << tok);
	MSGDEBUG(4, " of size " << cmdsize << sendmsg);

	fp = (float *)cmdptr;
	ip = (int *)cmdptr;

	switch (tok) {
	    case DPOINT:
		move3d(fd, fp[0], fp[1], fp[2]);
		draw3d(fd, fp[0], fp[1], fp[2]);
		break;
	    case DPOINT_I:
		fp = dataBlock + ip[0];
		move3d(fd, fp[0], fp[1], fp[2]);
		draw3d(fd, fp[0], fp[1], fp[2]);
		break;
	    case DLINE:
		move3d(fd, fp[0], fp[1], fp[2]);
		draw3d(fd, fp[3], fp[4], fp[5]);
		break;
	    case DLINE_I:
		fp = dataBlock + ip[0];
		move3d(fd, fp[0], fp[1], fp[2]);
		fp = dataBlock + ip[1];
		draw3d(fd, fp[0], fp[1], fp[2]);
		break;
	    case DSPHERE:
		sphere(fp, fp[3]);    // fp = (xyz,r)
		break;
	    case DSPHERE_I:
		sphere(dataBlock + int(fp[0]), fp[1]);
		break;
	    case DTRIANGLE:
		// draw a triangle between the three points
		triangle(fp, fp+3, fp+6, fp+9, fp+12, fp+15);
		break;
	    case DTRIANGLE_I:
		// draw a triangle between the three points, using indices into dataBlock
		triangle(dataBlock + ip[1],
			 dataBlock + ip[2],
			 dataBlock + ip[3],
			 dataBlock + ip[0],
			 dataBlock + ip[0],
			 dataBlock + ip[0]);
		break;
	    case DSQUARE:
		// draw a square, given the four points
		square(fp, fp+3, fp+6, fp+9, fp+12);
		break;
	    case DSQUARE_I:
		// draw a square, given the four points, using indices into dataBlock
		square(dataBlock + ip[0],
		       dataBlock + ip[1],
		       dataBlock + ip[2],
		       dataBlock + ip[3],
		       dataBlock + ip[4]);
		break;
	    case DCYLINDER:
		// draw a cylinder of given radius and resolution
		// The #define is a tradeoff between speed and memory usage
//#define USE_SLOW_CYLINDERS
#ifdef USE_SLOW_CYLINDERS
		cylinder(fp, fp+3, int(fp[7]), fp[6]);
#else
		cylinder(int(fp[7]), fp+8);
#endif
		break;
	    case DCYLINDER_I:
		// draw a cylinder of given radius and resolution, using indices
		cylinder(dataBlock + int(fp[0]),
			 dataBlock + int(fp[1]),
			 int(fp[3]),
			 fp[2] );
		break;
	    case DCONE:
		// draw a cone of given radius and resolution
		cone(fp, fp+3, int(fp[7]), fp[6]);
		break;
	    case DCONE_I:
		// draw a cone of given radius and resolution, using indices
		cone(dataBlock + int(fp[0]),
		     dataBlock + int(fp[1]),
		     int(fp[3]),
		     fp[2] );
		break;
	    case DTEXTPOS:
		// move the current character position to the point given
		memcpy(text_pos, fp, ptSize);
		break;
	    case DTEXTPOS_I:
		// move the current character position to the point given, using indices
		memcpy(text_pos, (float *)(dataBlock + ip[0]), ptSize);
		break;
	    case DTEXT:
		// display text at current position
		text3d(fd, text_pos[0], text_pos[1], text_pos[2],
		       cmdptr, ANNOTATION_TEXT, 0);
		break;
	    case DCOLORINDEX:
		// set the current color to the given color index ... assumes the
		// color has already been defined
		n = ip[0];

		if(!matDefined[n]) {
		    msgErr << "SBRenderer: Error, color " << n << " not defined."
					<< sendmsg;
		} else {
		    // set depth cue range if necessary
		    if (cueing_enabled()) {
			//lRGBrange((short)(255 * backColor[0]),
			//(short)(255 * backColor[1]),
			//(short)(255 * backColor[2]),
			//(short)(255 * matData[n][COLOR_INDEX + 0]),
			//(short)(255 * matData[n][COLOR_INDEX + 1]),
			//(short)(255 * matData[n][COLOR_INDEX + 2]),
			//getgdesc(GD_ZMIN), getgdesc(GD_ZMAX));
		    }
		    fp = &matData[n][COLOR_INDEX];
		    fill_color(fd, fp[0], fp[1], fp[2]);
		    line_color(fd, fp[0], fp[1], fp[2]);

		    // bind the material if materials enabled
		    materialOn = n;
		    if (materialsActive)
			activate_material();
		}
		break;
	    case DCOLORRGB:
		// just set the non-material color to the given value.	Turns off
		// materials.
		deactivate_materials();

		// set depth cue range if necessary
		if(cueing_enabled()) {
		    //lRGBrange((short)(255 * backColor[0]),
		    //		(short)(255 * backColor[1]),
		    //		(short)(255 * backColor[2]),
		    //		(short)(255 * ((float *)cmdptr)[0]),
		    //		(short)(255 * ((float *)cmdptr)[1]),
		    //		(short)(255 * ((float *)cmdptr)[2]),
		    //		getgdesc(GD_ZMIN), getgdesc(GD_ZMAX));
		}
		fill_color(fd, fp[0], fp[1], fp[2]);
		line_color(fd, fp[0], fp[1], fp[2]);
		break;
	    case DPICKPOINT:
	    case DPICKPOINT_I:
	    case DPICKLINE:
	    case DPICKLINE_I:
	    case DPICKBOX:
	    case DPICKBOX_I:
		;		  // do nothing, these are picking tokens
		break;
	    case DLIGHTONOFF:
		// turn on or off a light source
		activate_light(ip[0], ip[1]);
		break;
	    case DMATERIALS:
		// enable/disable material characteristics
		if (ip[0])
		    activate_material();
		else
		    deactivate_materials();
		break;
	    case DSPHERERES:
		// set the current sphere resolution
		set_sphere_res(ip[0]);
		break;
	    case DSPHERETYPE:
		// set the current sphere type
		set_sphere_mode(ip[0]);
		break;
	    case DLINESTYLE:
		// set the current line style
		set_line_style(ip[0]);
		break;
	    case DLINEWIDTH:
		// set the current line width
		set_line_width(ip[0]);
		break;
	    case DPUSH:
		// push the current trans matrix
		push();
		break;
	    case DPOP:
		// pop the current trans matrix
		pop();
		break;
	    case DLOAD:
		// load the given trans matrix onto the stack
		replace_matrix3d(fd, (float (*)[4])cmdptr);
		break;
	    case DMULT:
		// multiply the given trans matrix
		concat_transformation3d(fd, (float (*)[4])cmdptr, PRE, REPLACE);
		break;
	    case DTRANS:
		// do a translation
		translate(fp[0], fp[1], fp[2]);
		break;
	    case DSCALE:
		// do a scale
		scale(fp[0], fp[1], fp[2]);
		break;
	    case DROT:
		// do a rotation about a given axis
		rot(fp[0], 'x' + int(fp[1]));
		break;
	    case DCOLORDEF:
		// define a new color
		define_material(int(fp[0]), fp+1);
		break;
	    case DLIGHTDEF:
		// define a new light source
		define_light(int(fp[0]), fp+1, fp+4);
		break;
	    case DCLEAR:
		// clear the display
		clear();
		break;
	    case DMOREDATA:
	    case DNEWDATA:
		// set the current drawing data block
#ifdef VMDCAVE
		dataBlock = (float *)cmdptr;	// point to current position in list
#else
		dataBlock = ((float **)cmdptr)[0];  // point to given memory loc
#endif
		break;
	    default:
		// command not found, help!
		msgErr << "SBRenderer: Unknown drawing token " << tok
		       << " encountered ... Skipping this command." << sendmsg;
	}

	// update position in array
	cmdptr += cmdsize;
    }

    // restore transformation matrix
    pop();
}

