/***************************************************************************
 *cr
 *cr		(C) Copyright 1995 The Board of Trustees of the
 *cr			    University of Illinois
 *cr			     All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: STLDisplayDevice.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1997/03/20 03:59:20 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 *   Render to a STL (Stereo-Lithography file).  
 *   See http://www.sdsc.edu/tmf/ for more information on the file and
 *  how to make a physical 3-D model from VMD.
 *
 ***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "STLDisplayDevice.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"


#define XSCALE 		1.0    	// Change these to scale data
#define YSCALE 		1.0
#define ZSCALE 		1.0
#define MAKE_POSITIVE 	1      	// 1=translate data to make minimum in
			 	// each direction zero. 0 = don't translate
#define CALC_NORMALS 	0   	/* Some STL programs require
			 	   that normals be given.  Others don't.
			 	   Set this to zero and the normals will
			 	   be assigned zero values. A value greater
			 	   than zero and VMD will calculate them. */

// constructor ... call the parent with the right values
STLDisplayDevice::STLDisplayDevice(void) : 
  FileRenderer("STL", "vmd.stl", "true") {
}


// draw a triangle
void STLDisplayDevice::STLtriangle(float *a, float *b, float *c) {

    if (!CALC_NORMALS)
          fprintf(outfile, "  facet normal 0.0 0.0 0.0\n");
    else {
	  float nx, ny, nz, n;
	  nx = a[1]*(b[2]-c[2])+b[1]*(c[2]-a[2])+c[1]*(a[2]-b[2]);
	  ny = a[2]*(b[0]-c[0])+b[2]*(c[0]-a[0])+c[2]*(a[0]-b[0]);
	  nz = a[0]*(b[1]-c[1])+b[0]*(c[1]-a[1])+c[0]*(a[1]-b[1]);
	  n = nx*nx+ny*ny+nz*nz;
	  n = sqrt(n);
	  nx /= n; ny /= n; nz /= n;
	  fprintf (outfile, "  facet normal %f %f %f\n", nx, ny, nz);
	  }

    fprintf(outfile,"     outer loop\n");
    fprintf(outfile,"       vertex %f %f %f\n",(a[0]+xmin)*XSCALE,
					       (a[1]+ymin)*YSCALE,
					       (a[2]+zmin)*ZSCALE);
    fprintf(outfile,"       vertex %f %f %f\n",(b[0]+xmin)*XSCALE,
					       (b[1]+ymin)*YSCALE,
					       (b[2]+zmin)*ZSCALE);
    fprintf(outfile,"       vertex %f %f %f\n",(c[0]+xmin)*XSCALE,
					       (c[1]+ymin)*YSCALE,
					       (c[2]+zmin)*ZSCALE);
    fprintf(outfile,"     endloop\n");
    fprintf(outfile,"  endfacet\n");
}

void STLDisplayDevice::write_header (void) {
    fprintf (outfile, "solid molecule\n");
}

void STLDisplayDevice::write_trailer (void) {
    fprintf (outfile, "endsolid\n");
    msgInfo << "Only triangles in the present scene have been processed.\n";

}

void STLDisplayDevice::min_check(float *a, float *b, float *c) {
    if (a[0] < xmin) (xmin = a[0]);
    if (b[0] < xmin) (xmin = b[0]);
    if (c[0] < xmin) (xmin = c[0]);
    if (a[1] < ymin) (ymin = a[1]);
    if (b[1] < ymin) (ymin = b[1]);
    if (c[1] < ymin) (ymin = c[1]);
    if (a[2] < zmin) (zmin = a[2]);
    if (b[2] < zmin) (zmin = b[2]);
}

void STLDisplayDevice::pre_render(void *cmdlist) {
   char *cmdptr = (char *)cmdlist;
   int tok, cmdsize;

   xmin = 9999999;  // Init mins to some ridiculously high number
   ymin = 9999999;
   zmin = 9999999;
   MSGDEBUG(3, "STLDisplayDevice: pre-rendering command list." << sendmsg);

   if (!cmdlist) return;

   dataBlock = NULL;
   while ((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
      float *fp;
      int *ip;

      if (tok == DLINKLIST) {
	 cmdptr += sizeof (int);
	 cmdptr = *((char **) cmdptr);
	 tok = ((int *)cmdptr)[0];
       }

       cmdsize = ((int *)cmdptr)[1];
       cmdptr += 2*sizeof(int);

       fp = (float *)cmdptr;
       ip = (int *)cmdptr;

       switch (tok) {
       case DTRIANGLE:
       min_check (fp, fp+3, fp+6);
       break;
       case DTRIANGLE_I:
       min_check (dataBlock + ip[1], 
		  dataBlock + ip[2], 
		  dataBlock + ip[3]);
       break;
       }

      cmdptr += cmdsize;
   }
}


void STLDisplayDevice::render(void *cmdlist) {

		// First find minimum values in data if
		// we are to translate to make data start
		// from zero.  This is a common requirement
		// in STL programs.
    if (MAKE_POSITIVE) 
       pre_render (cmdlist);
    else {
       xmin = 0.0;
       ymin = 0.0;
       zmin = 0.0;
    }
    xmin = -xmin;
    ymin = -ymin;
    zmin = -zmin;
      

    char *cmdptr = (char *)cmdlist;
    int tok, cmdsize;

    MSGDEBUG(3, "STLDisplayDevice: rendering command list." << sendmsg);

    if (!cmdlist) return;

    // scan through the list, getting each command and executing it, until
    // the end of commands token is found
    dataBlock = NULL;
    while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
	float *fp;
	int *ip;

	if (tok == DLINKLIST) {
	    cmdptr += sizeof(int);	 // step forward one (no length here)
	    cmdptr = *((char **) cmdptr);// and go to the next link
	    tok = ((int *)cmdptr)[0];	 // this is guaranteed to be neither
	}				// a DLINKLIST nor DLASTCOMMAND

	cmdsize = ((int *)cmdptr)[1];
	cmdptr += 2*sizeof(int);

	fp = (float *)cmdptr;
	ip = (int *)cmdptr;

	switch (tok) {
	    case DPOINT:
		break;
	    case DPOINT_I:
		break;
	    case DLINE:
		break;
	    case DLINE_I:
		break;
	    case DSPHERE:
		break;
	    case DSPHERE_I:
		break;
	    case DTRIANGLE:
		STLtriangle(fp, fp+3, fp+6);
		break;
	    case DTRIANGLE_I:
		STLtriangle(dataBlock + ip[1],
			 dataBlock + ip[2],
			 dataBlock + ip[3]);
		break;
	    case DSQUARE:
		break;
	    case DSQUARE_I:
		break;
	    case DCYLINDER:
#ifdef USE_SLOW_CYLINDERS
#else
#endif
		break;
	    case DCYLINDER_I:
		break;
	    case DCONE:
		break;
	    case DCONE_I:
		break;
	    case DTEXTPOS:
		break;
	    case DTEXTPOS_I:
		break;
	    case DTEXT:
		break;
	    case DCOLORINDEX:
		break;
	    case DCOLORRGB:
		break;
	    case DPICKPOINT:
		break;
	    case DPICKPOINT_I:
		break;
	    case DPICKLINE:
		break;
	    case DPICKLINE_I:
		break;
	    case DPICKBOX:
		break;
	    case DPICKBOX_I:
		break;
	    case DLIGHTONOFF:
		break;
	    case DMATERIALS:
		break;
	    case DSPHERERES:
		break;
	    case DSPHERETYPE:
		break;
	    case DLINESTYLE:
		break;
	    case DLINEWIDTH:
		break;
	    case DPUSH:
		break;
	    case DPOP:
		break;
	    case DLOAD:
		break;
	    case DMULT:
		break;
	    case DTRANS:
		break;
	    case DSCALE:
		break;
	    case DROT:
		break;
	    case DCOLORDEF:
		break;
	    case DLIGHTDEF:
		break;
	    case DCLEAR:
		break;
	    case DMOREDATA:
	    case DNEWDATA:
		// set the current drawing data block
#ifdef VMDCAVE
		dataBlock = (float *)cmdptr;	// point to current position in list
#else
		dataBlock = ((float **)cmdptr)[0];  // point to given memory loc
#endif
		break;
	}

	// update position in array
	cmdptr += cmdsize;
    }
}



