/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

#include <string.h>
#include "SortableArray.h"


////////////////////  constructor
template<class T>
SortableArray<T>::SortableArray(int s, float rsf) {
  // save factor by which to increase array if necessary
  resizeFactor = rsf;
  
  // initialize values for current array size and max array size
  currSize = 0;
  sz = (s > 0 ? s : 10);
  
  // allocate space for maximum size of array
  data = new T[sz];
}


////////////////////  destructor
template<class T>
SortableArray<T>::~SortableArray(void) {
  // free up the current array, if necessary
  if(data)
    delete [] data;
}
  

///////////////////
///////////////////  public routines
///////////////////


// return the nth item; assume N is >= 0
template<class T>
T& SortableArray<T>::item(int N) {

  // check and see if this is attempting to access an item larger than
  // the array.  If so, extend the max size of the array by resizeFactor,
  // and return the value at the Nth position (which will be basically
  // random).
  if(N >= sz) {			// extend size of array if necessary
  
    // calc new size of array
    int newsize = (int)((float)N * resizeFactor + 0.5);

    // allocate new space, and copy over previous copy.  We only need to
    // copy over the first currSize elements, since currSize is the max
    // index that has been accessed (read OR write).  Then delete old
    // storage.
    T *newdata = new T[newsize];
    if(data) {
      memcpy((void *)newdata, (void *)data, currSize * sizeof(T));
      delete [] data;
    }
    
    // save new values
    data = newdata;
    sz = newsize;
  }
  
  // remember what the maximum index reached so far is
  if(N >= currSize)
    currSize = N + 1;
    
  // return element at Nth position
  return data[N];
}


// insert new item after the specified index.  Return index of new item.
// if specified index is < 0, item put at beginning of list.
// if specified index is >= num(), item put at end of list.
template<class T>
int SortableArray<T>::insert_after(int n, const T& val) {
  int thissize = currSize;
  int newindx = n + 1;
  if(n < 0)
    newindx = 0;
  else if(n >= currSize)
    newindx = currSize;

  for(int i=thissize; i > newindx; i--)
    item(i) = item(i-1);
  data[newindx] = val;
    
  return newindx;
}


// insert new item before the specified index.  Return index of new item.
// if specified index is < 0, item put at beginning of list.
// if specified index is >= num(), item put at end of list.
template<class T>
int SortableArray<T>::insert_before(int n, const T& val) {
  int thissize = currSize;
  int newindx = n;
  if(n < 0)
    newindx = 0;
  else if(n >= currSize)
    newindx = currSize;
    
  for(int i=thissize; i > newindx; i--)
    item(i) = item(i-1);
  data[newindx] = val;

  return newindx;
}


// remove the Mth ... Nth items: move all items lower in the
// array up to fill the empty slots.
// If both arguments are < 0, removes ALL items.
// If the second argument is < 0, just removes item m
template<class T>
void SortableArray<T>::remove(int m, int n) {
  if(m < currSize && n < currSize) {
    if((m < 0 && n < 0) || (m >= 0 && n < 0) || (m > 0 && n > 0 && m <= n)) {
      int N = n, M = (m >= 0 ? m : 0);
      if(m < 0 && n < 0)
	N = (currSize - 1);
      else if(n < 0)
        N = M;
      else
	N = n;
      int removed = N - M + 1;
      for(int i=N+1; i < currSize; i++)
        data[i - removed] = data[i];
      currSize -= removed;
    }
  }
}


// scan the array until the first item that matches in the array is
// found.  Return the index if found, (-1) otherwise.
template<class T>
int SortableArray<T>::find(const T& val) {
  int i, retval = (-1);
  for(i=0; i < currSize; i++) {
    if(data[i] == val) {
      retval = i;
      break;
    }
  }
  return retval;
}


// auxilliary function for qsort
template<class T>
void SortableArray<T>::swap (int i, int j)
{
 T temp;

 temp = data[i];
 data[i]=data[j];
 data[j]=temp;
}

// sorts elements based on a specialied field.  To change
// the sort, change the field which is used as a basis for
// comparison.  The '>' and '<' symbols are overloaded.
template<class T>
void SortableArray<T>::qsort (int left, int right)
{
  int i, last;

  if (left >= right)    // fewer than two elements
     return;
  swap (left, (left + right) /2);
  last = left;
  for (i = left+1; i <=right; i++)
    if (data[i] > data[left]) 
     swap (++last, i);
  swap (left, last);
  qsort (left, last-1);
  qsort (last+1, right);
}



// This is an internal sort
template<class T>
void SortableArray<T>::isort () {
   int i, j;
   T temp;
   for (i = currSize-2; i>=0; i--) {  // jan 16 added = to >=
     j = i+1;
     temp = data[i];
     while ((j < currSize) && (temp < data[j])) {
       data[j-1] = data[j];
       j = j+1;
     }
     data[j-1] = temp;
   }
}

