/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: SymbolTable.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.7 $	$Date: 1996/11/12 02:48:52 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  Stores the functions available to get info from a molecule
 *  Calls the atom selection parser to create a parse tree
 *
 ***************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include "builtin.h"  // this is from the GNU C++ class library
                     //  it is used to catch the regex error function
#include "SymbolTable.h"
#include "ParseTree.h"

extern "C" int yyparse();

// converted from a template version; see SymbolTable.h


////////// bypasses normal error checking for use with GRegex.[Ch]
// Normally, when a regex finds an error during the compile,  the
// 'lib_error_handler' gets called.  What I do is swap the normal
// handler for a new one, which is active only during the compile.
// This can be used to catch the errors and store them into local
// (static) variables.  When the compile returns, I check those
// variable to see if there was an error, and I replaces the handler
// with the original function.
//   There seems to be no other way to do this.

// is 0 (FALSE) if there was an error
static symboltable_regex_error = 1;

// contains the error message
static GString symboltables_regex_message;

// reset the error value to 1
void symboltable_regex_reset(void)
{
  symboltable_regex_error = 1;
  symboltables_regex_message = "";
}

// returns the value of the error value.  Also, if there is a pointer,
// sets it to the error string
int symboltable_regex_check(const char ** s)
{
   if (s) {
      *s = symboltables_regex_message;
   }
   return symboltable_regex_error;
}

// temporarily replaces the error function so as to catch error
// and redirect the output to a string
void symboltable_regex_error_fctn(const char *s1, const char *s2)
{
   symboltables_regex_message = s1;
   symboltables_regex_message +=" : ";
   symboltables_regex_message +=s2;
   symboltable_regex_error = 0;  // FALSE means error
}

// hold the previous error handle
static two_arg_error_handler_t symboltable_regex_oldfctn = NULL;

// swaps the normal error handler with the temporary one, and
// vice versa
void symboltable_regex_swap(void)
{
   if (symboltable_regex_oldfctn == NULL ) {
      symboltable_regex_oldfctn = lib_error_handler;
      lib_error_handler = symboltable_regex_error_fctn;
   } else {
      lib_error_handler = symboltable_regex_oldfctn;
      symboltable_regex_oldfctn = NULL;
   }
}

/////////////////////////////////////////////////////////////
/// Keep track of the GRegex used to detect keywords.  The
///   index of the GRegex is identical to the index of the
///   function in the SymbolTableElement array which accesses
///   (or converts) the data.

SymbolTableName::SymbolTableName(void) {
   regex = NULL;
   is_a = SymbolTableName::NOTHING;
}

SymbolTableName::SymbolTableName(const char *newpattern, const char *vis,
				 symdesc new_desc,
				 symtype new_takes, symtype new_returns) {
   // ensures that a complete match is made
   pattern = "^";
   pattern += newpattern;
   pattern += "$";
   visible = vis;
   is_a = new_desc;
   takes_a = new_takes;
   returns_a = new_returns;
   set();
}
SymbolTableName::~SymbolTableName(void) {
   unset();
}

int SymbolTableName::set(const char **err)
{
   symboltable_regex_reset();
   symboltable_regex_swap();
   regex = new GRegex(pattern);
   symboltable_regex_swap();
   if (!regex) {
      *err = "Couldn't make a GRegex.  Out of memory???\n";
      return 0;
   }
   int i = symboltable_regex_check(err);
   if (!i) {
      unset();
   }
   return i;
}

// get rid of the compiled string
void SymbolTableName::unset(void)
{
   if (regex) delete regex;
   regex = NULL;
}

///////////////////////// SymbolTableElement ///////////
/// keeps track of the member function and global function pointers
SYMBOLTABLE_TEMP_HDR
SymbolTableElement SYMBOLTABLE_TEMP_REF::SymbolTableElement(void)
{
   fctn = NULL;
   set_fctn = NULL;
}

// this is for functions like 'cos', which are global and
// do not need to know the number of elements
SYMBOLTABLE_TEMP_HDR
SymbolTableElement SYMBOLTABLE_TEMP_REF::
SymbolTableElement(void *new_fctn)
{
   fctn = new_fctn;
   set_fctn = NULL;
}

SYMBOLTABLE_TEMP_HDR
SymbolTableElement SYMBOLTABLE_TEMP_REF::
SymbolTableElement( int (SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, int *, int *),
		    int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, int *, int *))
{
   keyword_int     = new_fctn;
   set_keyword_int = set_fctn;
}

SYMBOLTABLE_TEMP_HDR
SymbolTableElement SYMBOLTABLE_TEMP_REF::
SymbolTableElement( int ( SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, double *, int*),
		    int ( SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, double *, int*))
{
   keyword_double     = new_fctn;
   set_keyword_double = set_fctn;
}

SYMBOLTABLE_TEMP_HDR
SymbolTableElement SYMBOLTABLE_TEMP_REF::
SymbolTableElement(
    int ( SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, GString *, int *),
    int ( SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, GString *, int *))
{
   keyword_string = new_fctn;
   set_keyword_string = set_fctn;
}

SYMBOLTABLE_TEMP_HDR
SymbolTableElement SYMBOLTABLE_TEMP_REF::
SymbolTableElement(
    int ( SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, int *),
    int ( SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, int *, int *))
{
   keyword_single = new_fctn;
   set_keyword_single = set_fctn;
}
SYMBOLTABLE_TEMP_HDR
SymbolTableElement SYMBOLTABLE_TEMP_REF::
SymbolTableElement(
		   int ( SYMBOLTABLE_TEMP_FCTN*new_fctn)
		     (int argc, const char ** argv, int *, int, int *),
		   int ( SYMBOLTABLE_TEMP_FCTN*new_set_fctn)
		     (int argc, const char ** argv, int *, int, int *))
{
   keyword_stringfctn = new_fctn;
   set_fctn = new_set_fctn;
}


////////////////// SymbolTable /////////////////
// tracks both the names (GRegex) and the function pointers

SYMBOLTABLE_TEMP_HDR
SymbolTable SYMBOLTABLE_TEMP_REF::SymbolTable(void) {
   // some compiliers will warn if there is a destructor with no
   // constructor, so I put this here to supress that warning
}

SYMBOLTABLE_TEMP_HDR
SymbolTable SYMBOLTABLE_TEMP_REF::~SymbolTable(void)
{
   int num = names.num();
   for (int i=0; i<num; i++) {
      names[i]->unset();    // get rid of the compiled GRegex
      delete names[i];
      delete fctns[i];
   }
   // and the resize array is deallocated here ...
}

// add a global function
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::add_function(const char *pattern,
		    const char *visible, void *fctn,
		    SymbolTableName::symtype takes,
		    SymbolTableName::symtype returns)
{
   SymbolTableName *tmpname = new SymbolTableName(pattern, visible,
						  SymbolTableName::FUNCTION,
						  takes, returns);
   if (!tmpname->regex) {
      delete tmpname;
      return 0;
   }
   SymbolTableElement SYMBOLTABLE_TEMP_REF *tmpele = new SymbolTableElement SYMBOLTABLE_TEMP_REF(fctn);
   names.append(tmpname);
   fctns.append(tmpele);
   return 1;
}


// keywords only fill a pre-allocated list with the given data type
#define add_keyword_macro(name,datadesc, accessfctn, dataident, datafmt)      \
SYMBOLTABLE_TEMP_HDR							      \
int SymbolTable SYMBOLTABLE_TEMP_REF::name(const char *pattern,		      \
const char *visible, datadesc, accessfctn)		              	      \
{									      \
   SymbolTableName *tmpname = new SymbolTableName(pattern, visible,           \
						  SymbolTableName::dataident, \
						  SymbolTableName::datafmt,   \
						  SymbolTableName::datafmt);  \
   if (!tmpname->regex) { /* then there was an error */			      \
      delete tmpname;							      \
      return 0;								      \
   }									      \
   SymbolTableElement SYMBOLTABLE_TEMP_REF *tmpfctn = new		      \
        SymbolTableElement SYMBOLTABLE_TEMP_REF(fctn, set_fctn);	      \
   names.append(tmpname);						      \
   fctns.append(tmpfctn);						      \
   return 1;								      \
}

add_keyword_macro(add_keyword, 
		  int (SYMBOLTABLE_TEMP_FCTN*fctn)(int, int*, int*),
		  int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, int*, int*),
		  KEYWORD, IS_INT);
add_keyword_macro(add_keyword, 
		  int (SYMBOLTABLE_TEMP_FCTN*fctn)(int, double*, int*) ,
		  int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, double*, int*) ,
                  KEYWORD, IS_FLOAT);
add_keyword_macro(add_keyword, 
		  int (SYMBOLTABLE_TEMP_FCTN*fctn)(int, GString*, int*) ,
		  int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, GString*, int*) ,
                  KEYWORD, IS_STRING)


// single words are like 'water' and 'protein'
// all they do is adjust the flags appropriately
// NOTE: a singleword must return_a IS_INT for the set_keyword to work
add_keyword_macro(add_singleword, 
		  int (SYMBOLTABLE_TEMP_FCTN*fctn)(int, int*),
		  int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, int*, int*),
		  SINGLEWORD, IS_INT)


add_keyword_macro(add_stringfctn, 
		  int (SYMBOLTABLE_TEMP_FCTN*fctn)(int, const char **, 
						   int *,
						   int, int *),
		  int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, const char **, 
						   int *,
						   int, int *),
		  STRINGFCTN, IS_STRING)

/// find the index that matches the word, or return -1
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::find_attribute( char *s)
{
  for (int i=names.num()-1; i>=0; i--) {
    if (names[i]->regex->match(s, strlen(s)) != -1) {
      return i;
    }
  }
  return -1;
}
// is the given element changable?  Returns TRUE on yes, FALSE on no
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::is_changeable( int fctnidx)
{
  if (!(fctnidx >= 0 && fctnidx < fctns.num())) {
    return FALSE;
  }
  SymbolTableElement PARSETREE_TEMP_REF *fctn=fctns[fctnidx];
#if PARSETREE_TEMP == 1
  if ( (obj ->* (namesfctn->set_fctn))) {
#else
  if (fctn->set_fctn) {
#endif
    return TRUE;
  }
  return FALSE;
}




///////////////// get the information from the given keyword

int symboltable_int_compare(const void *x, const void *y)
{
   return *((int *) x) - *((int *) y);
}
    
int symboltable_double_compare(const void *x, const void *y)
{
   double delta = *((double *) x) - *((double *) y);
   if (delta < 0) {
      return -1;
   }
   if (delta == 0) {
      return 0;
   }
   return 1;
}
int symboltable_string_compare(const void *x, const void *y)
{
   return compare( *((GString *) x), *((GString *) y));
}
    
// return it into an integer array    
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::extract_keyword_info
(char *name, int num, int *iptr, int *flgs, int sorted,
 SymbolTableName::symtype sort_as)
{
  int i = find_attribute(name);
  if (i == -1) return 1;
  extract_keyword_info(i, num, iptr, flgs, sorted, sort_as);
  return 0;
}
SYMBOLTABLE_TEMP_HDR
void SymbolTable SYMBOLTABLE_TEMP_REF::extract_keyword_info
(int fctnidx, int num, int *iptr, int *flgs, int sorted,
 SymbolTableName::symtype sort_as)
{
   // is this a keyword?
   if (names[fctnidx]->is_a != SymbolTableName::KEYWORD) {
      // then it is all 0
      for (int i=0; i<num; i++) {
	 iptr[i]=0;
      }
      return;
   }
   // the function that contains the values
   SymbolTableElement PARSETREE_TEMP_REF *fctn=fctns[fctnidx];
   // and its type
   SymbolTableName::symtype has_type = names[fctnidx]-> returns_a;
   if (has_type == SymbolTableName::IS_INT) {  // then no conversion needed
#if PARSETREE_TEMP == 1
      (obj->*(fctn->keyword_int))(num, iptr, flgs);
#else
      (fctn->keyword_int)(num, iptr, flgs);
#endif
      if (sorted) {
	 if (sort_as != SymbolTableName::IS_STRING) {
	    qsort(iptr, num, sizeof(int), symboltable_int_compare);
	 }
	 else 
	 {
	    // I don't like doing this, but some things (like resid) need
	    // to be done this way
	    GString *tmp = new GString[num];
	    char s[50];
	    int i;
	    for (i=0; i<num; i++) {
	       sprintf(s, "%d", iptr[i]);
	       tmp[i] = s;
	    }
	    qsort(tmp, num, sizeof(GString), symboltable_string_compare);
	    for ( i=0; i<num; i++) {
	       iptr[i] = atoi((const char *) tmp[i]);
	    }
	    delete [] tmp;
	 }
      }
      return;
   }
   if (has_type == SymbolTableName::IS_FLOAT) {
      // make some temp space
      double *tmpspace = new double[num];
      // nice and simple
      extract_keyword_info(fctnidx, num, tmpspace, flgs, sorted);
      // convert to int
      for (int i=0; i<num; i++) {
	 iptr[i] = (int) tmpspace[i];
      }
      
      delete [] tmpspace;
      return;
   }
   if (has_type == SymbolTableName::IS_STRING) {
      GString *tmpspace = new GString[num];
      extract_keyword_info(fctnidx, num, tmpspace, flgs, sorted);
      for (int i=0; i<num; i++) {
	 iptr[i] = atoi( (const char *) tmpspace[i]);
      }
      delete [] tmpspace;
      return;
   }
}

// return it into a double array    
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::extract_keyword_info
(char *name, int num, double *dptr, int *flgs, int sorted,
 SymbolTableName::symtype sort_as)
{
  int i = find_attribute(name);
  if (i == -1) return 1;
  extract_keyword_info(i, num, dptr, flgs, sorted, sort_as);
  return 0;
}
SYMBOLTABLE_TEMP_HDR
void SymbolTable SYMBOLTABLE_TEMP_REF::extract_keyword_info
(int fctnidx, int num, double *dptr, int *flgs, int sorted,
 SymbolTableName::symtype sort_as)
{
   // is this a keyword?
   if (names[fctnidx]->is_a != SymbolTableName::KEYWORD) {
      // then it is all 0
      for (int i=0; i<num; i++) {
	 dptr[i]=0.0;
      }
      return;
   }
   // the function that contains the values
   SymbolTableElement PARSETREE_TEMP_REF *fctn=fctns[fctnidx];
   // and its type
   SymbolTableName::symtype has_type = names[fctnidx]-> returns_a;
   if (has_type == SymbolTableName::IS_FLOAT) {  // then no conversion needed
#if PARSETREE_TEMP == 1
      (obj->*(fctn->keyword_double))(num, dptr, flgs);
#else
      (fctn->keyword_double)(num, dptr, flgs);
#endif
      if (sorted) {
	 if (sort_as != SymbolTableName::IS_STRING) {
	    qsort(dptr, num, sizeof(double), symboltable_double_compare);
	 }
	 else
	 {
	    GString *tmp = new GString[num];
	    char s[50];
	    int i;
	    for (i=0; i<num; i++) {
	       sprintf(s, "%lf", dptr[i]);
	       tmp[i] = s;
	    }
	    qsort(tmp, num, sizeof(GString), symboltable_string_compare);
	    for ( i=0; i<num; i++) {
	       dptr[i] = atoi((const char *) tmp[i]);
	    }
	    delete [] tmp;
	 }
      }
      return;
   }
   if (has_type == SymbolTableName::IS_INT) {
      // make some temp space
      int *tmpspace = new int[num];
      // slow, but simple
      extract_keyword_info(fctnidx, num, tmpspace, flgs, sorted);
      // convert to double
      for (int i=0; i<num; i++) {
	 dptr[i] = (double) tmpspace[i];
      }
      delete [] tmpspace;
      return;
   }
   if (has_type == SymbolTableName::IS_STRING) {
      GString *tmpspace = new GString[num];
      extract_keyword_info(fctnidx, num, tmpspace, flgs, sorted);
      for (int i=0; i<num; i++) {
	 dptr[i] = atof( (const char *) tmpspace[i]);
      }
      delete [] tmpspace;
      return;
   }
}

// return it into a string array    
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::extract_keyword_info
(char *name, int num, GString *sptr, int *flgs, int sorted,
 SymbolTableName::symtype sort_as)
{
  int i = find_attribute(name);
  if (i == -1) return 1;
  extract_keyword_info(i, num, sptr, flgs, sorted, sort_as);
  return 0;
}
SYMBOLTABLE_TEMP_HDR
void SymbolTable SYMBOLTABLE_TEMP_REF::extract_keyword_info
(int fctnidx, int num, GString *sptr, int *flgs, int sorted,
 SymbolTableName::symtype sort_as)
{
   // is this a keyword?
   if (names[fctnidx]->is_a != SymbolTableName::KEYWORD) {
      // then it is all ""
      for (int i=0; i<num; i++) {
	 sptr[i]="";
      }
      return;
   }
   // the function that contains the values
   SymbolTableElement PARSETREE_TEMP_REF *fctn=fctns[fctnidx];
   // and its type
   SymbolTableName::symtype has_type = names[fctnidx] -> returns_a;
   if (has_type == SymbolTableName::IS_STRING) {  // then no conversion needed
#if PARSETREE_TEMP == 1
      (obj->*(fctn->keyword_string))(num, sptr, flgs);
#else
      (fctn->keyword_string)(num, sptr, flgs);
#endif
      if (sorted) {
	 if (sort_as == SymbolTableName::IS_INT) {
	    int *tmp = new int[num];
	    int i;
	    for (i=0; i<num; i++) {
	       tmp[i] = atoi( (const char *) sptr[i]);
	    }
	    qsort(tmp, num, sizeof(int), symboltable_int_compare);
	    char s[40];
	    for ( i=0; i<num; i++) {
	       sprintf(s, "%d", tmp[i]);
	       sptr[i] = s;
	    }
	    delete [] tmp;
	 } else if (sort_as == SymbolTableName::IS_FLOAT) {
	    double *tmp = new double[num];
	    int i;
	    for ( i=0; i<num; i++) {
	       tmp[i] = atof( (const char *) sptr[i]);
	    }
	    qsort(tmp, num, sizeof(double), symboltable_double_compare);
	    char s[40];
	    for ( i=0; i<num; i++) {
	       sprintf(s, "%lf", tmp[i]);
	       sptr[i] = s;
	    }
	    delete [] tmp;
	 } else {
	    qsort(sptr, num, sizeof(GString), symboltable_string_compare);
	 }
      }
      return;
   }
   if (has_type == SymbolTableName::IS_INT) {
      // make some temp space
      int *tmpspace = new int[num];
      // slow, but simple
      extract_keyword_info(fctnidx, num, tmpspace, flgs, sorted);
      // convert to a string
      char s[50];
      for (int i=0; i<num; i++) {
	if (flgs[i]) {
	  sprintf(s, "%d", tmpspace[i]);
	  sptr[i] = s;
	}
      }
      delete [] tmpspace;
      return;
   }
   if (has_type == SymbolTableName::IS_FLOAT) {
      double *tmpspace = new double[num];
      extract_keyword_info(fctnidx, num, tmpspace, flgs, sorted);
      char s[50];
      for (int i=0; i<num; i++) {
	if (flgs[i]) {
	  sprintf(s, "%lf", tmpspace[i]);
	  sptr[i] = s;
	}
      }
      delete [] tmpspace;
      return;
   }
}
//////////////////////////////////////////////////////
///        set_keyword_info
/// this is the inverse of "extract_keyword_info"
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(char *name, int num, int *iptr, int *flgs)
{
  int i = find_attribute(name);
  if (i == -1) return 1;
  set_keyword_info(i, num, iptr, flgs);
  return 1;
}
SYMBOLTABLE_TEMP_HDR
void SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(int fctnidx, int num, int *iptr, int *flgs)
{
  // is this a keyword or singleword?
  if (names[fctnidx]->is_a != SymbolTableName::KEYWORD &&       // set int
      names[fctnidx]->is_a != SymbolTableName::SINGLEWORD) { // set boolean
    // ignore
    return;
  }
  // the function used to set the values
  SymbolTableElement PARSETREE_TEMP_REF *fctn = fctns[fctnidx];
  // and its type
  // NOTE:  This works for SINGLEWORD since it is defined to return an int!!
  SymbolTableName::symtype has_type = names[fctnidx] -> returns_a;
  if (has_type == SymbolTableName::IS_INT) { // no conversion needed
    if (is_changeable(fctnidx)) {
#if PARSETREE_TEMP == 1
      (obj->*(fctn->set_keyword_int))(num, iptr, flgs);
#else
      (fctn->set_keyword_int)(num, iptr, flgs);
#endif
    } else {
fprintf(stderr, 
"Cannot modify values for entry '%s' -- no associated function'",
(const char *) names[fctnidx]->visible);
    }
    return;
  }
  if (has_type == SymbolTableName::IS_FLOAT) {
    // make some temp space
    double *tmpspace = new double[num];
    // copy the data to it
    for (int i=0; i<num; i++) {
      tmpspace[i] = (double) iptr[i];
    }
    // and call the correct function
    set_keyword_info(fctnidx, num, tmpspace, flgs);
    delete [] tmpspace;
    return;
  }

  if (has_type == SymbolTableName::IS_STRING) {
    GString *tmpspace = new GString[num];
    char s[10];
    for (int i=0; i<num; i++) {
      sprintf(s, "%d", iptr[i]);
      tmpspace[i] = s;
    }
    set_keyword_info(fctnidx, num, tmpspace, flgs);
    delete [] tmpspace;
    return;
  }
}

////////// for doubles
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(char *name, int num, double *dptr, int *flgs)
{
  int i = find_attribute(name);
  if (i == -1) return 1;
  set_keyword_info(i, num, dptr, flgs);
  return 1;
}
SYMBOLTABLE_TEMP_HDR
void SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(int fctnidx, int num, double *dptr, int *flgs)
{
  // is this a keyword?
  if (names[fctnidx]->is_a != SymbolTableName::KEYWORD &&
      names[fctnidx]->is_a != SymbolTableName::SINGLEWORD) {
    // ignore
    return;
  }
  // the function used to set the values
  SymbolTableElement PARSETREE_TEMP_REF *fctn = fctns[fctnidx];
  // and its type
  SymbolTableName::symtype has_type = names[fctnidx] -> returns_a;
  if (has_type == SymbolTableName::IS_FLOAT) { // no conversion needed
    if (is_changeable(fctnidx)) {
#if PARSETREE_TEMP == 1
      (obj->*(fctn->set_keyword_double))(num, dptr, flgs);
#else
      (fctn->set_keyword_double)(num, dptr, flgs);
#endif
    } else {
fprintf(stderr, 
"Cannot modify values for entry '%s' -- no associated function'",
(const char *) names[fctnidx]->visible);
    }
    return;
  }
  if (has_type == SymbolTableName::IS_INT) {
    // make some temp space
    int *tmpspace = new int[num];
    // copy the data to it
    for (int i=0; i<num; i++) {
      tmpspace[i] = (int) dptr[i];
    }
    // and call the correct function
    set_keyword_info(fctnidx, num, tmpspace, flgs);
    delete [] tmpspace;
    return;
  }

  if (has_type == SymbolTableName::IS_STRING) {
    GString *tmpspace = new GString[num];
    char s[30];
    for (int i=0; i<num; i++) {
      sprintf(s, "%lf", dptr[i]);
      tmpspace[i] = s;
    }
    set_keyword_info(fctnidx, num, tmpspace, flgs);
    delete [] tmpspace;
    return;
  }
}


////////// for GStrings and char *
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(char *name, int num, GString *sptr, int *flgs)
{
  int i = find_attribute(name);
  if (i == -1) return 1;
  set_keyword_info(i, num, sptr, flgs);
  return 1;
}
SYMBOLTABLE_TEMP_HDR
void SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(int fctnidx, int num, GString *sptr, int *flgs)
{
  // is this a keyword?
  if (names[fctnidx]->is_a != SymbolTableName::KEYWORD &&
      names[fctnidx]->is_a != SymbolTableName::SINGLEWORD) {
    // ignore
    return;
  }
  // the function used to set the values
  SymbolTableElement PARSETREE_TEMP_REF *fctn = fctns[fctnidx];
  // and its type
  SymbolTableName::symtype has_type = names[fctnidx] -> returns_a;
  if (has_type == SymbolTableName::IS_STRING) { // no conversion needed
    if (is_changeable(fctnidx)) {
#if PARSETREE_TEMP == 1
      (obj->*(fctn->set_keyword_string))(num, sptr, flgs);
#else
      (fctn->set_keyword_string)(num, sptr, flgs);
#endif
    } else {
fprintf(stderr, 
"Cannot modify values for entry '%s' -- no associated function'",
(const char *) names[fctnidx]->visible);
    }
    return;
  }
  if (has_type == SymbolTableName::IS_INT) {
    // make some temp space
    int *tmpspace = new int[num];
    // copy the data to it
    for (int i=0; i<num; i++) {
      tmpspace[i] = atoi((const char *) sptr[i]);
    }
    // and call the correct function
    set_keyword_info(fctnidx, num, tmpspace, flgs);
    delete [] tmpspace;
    return;
  }

  if (has_type == SymbolTableName::IS_FLOAT) {
    double *tmpspace = new double[num];
    for (int i=0; i<num; i++) {
      tmpspace[i] = atof( (const char *) sptr[i]);
    }
    set_keyword_info(fctnidx, num, tmpspace, flgs);
    delete [] tmpspace;
    return;
  }
}
//////////////////// The char ** versions
SYMBOLTABLE_TEMP_HDR
int SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(char *name, int num, char **sptr, int *flgs)
{
  GString *tmp = new GString[num];
  int i;
  for ( i=0; i<num; i++) {
    if (flgs[i]) {
      tmp[i] = sptr[i];
    }
  }
  i = set_keyword_info(name, num, tmp, flgs);
  delete [] tmp;
  return i;
}
SYMBOLTABLE_TEMP_HDR
void SymbolTable SYMBOLTABLE_TEMP_REF::set_keyword_info
(int fctnidx, int num, char **sptr, int *flgs)
{
  GString *tmp = new GString[num];
  for (int i=0; i<num; i++) {
    if (flgs[i]) {
      tmp[i] = sptr[i];
    }
  }
  set_keyword_info(fctnidx, num, tmp, flgs);
  delete [] tmp;
}

//////////////////////////////////////////////////////
///  call the parser
SYMBOLTABLE_TEMP_HDR
ParseTree SYMBOLTABLE_TEMP_REF *SymbolTable SYMBOLTABLE_TEMP_REF::
parse(char *s) {
//   printf("Sending:%s\n", s);
   if (!s) return NULL;
   atomparser_yystring = s;
   atomparser_symbols = &(names[0]);  // get the pure array
   atomparser_numsymbols = names.num();
   yyparse();
   if (atomparser_result) {
      return new ParseTree SYMBOLTABLE_TEMP_REF(this, atomparser_result);
   }
   return NULL;
}

SYMBOLTABLE_TEMP_HDR
ParseTree SYMBOLTABLE_TEMP_REF *SymbolTable SYMBOLTABLE_TEMP_REF::parse(const char *s) {
   char *temps = strdup(s);
   ParseTree SYMBOLTABLE_TEMP_REF *ret = parse(temps);
   free(temps);
   return ret;
}



