/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: TclCommands.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.21 $	$Date: 1997/03/19 00:00:05 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Tcl <--> VMD interface commands used for the analysis and 
 * manipulation of structures
 *
 ***************************************************************************/

#include <stdlib.h> 
#include <malloc.h>
#include <errno.h>
#include "tcl.h"
#include "config.h" // for things like the version, authors, etc
#include "MoleculeList.h"
#include "TclCommands.h"
#include "CoorPDB.h"
extern MoleculeList *moleculeList;
#include "UIText.h"
extern UIText *uiText;

#include "ParseTree.h"
extern SymbolTable atomSelParser;

#define SIMPLE_TCL_OPT(string,result)       \
if (!strcmp(argv[1], string)) {             \
  Tcl_AppendResult(interp, result, NULL);   \
  return TCL_OK;                            \
}


// append t to s and return a pointer to the terminating '\0'
static char * my_strcat(char *s, const char *t)
{
  while (*s) {  // go to end of string
    s++;
  }
  while (*t) {  // copy to the end
    *s++ = *t++;
  }
  *s = 0;       // terminate the string
  return s;
}

// add 'f' to the end of the string
char *tcl_append_double(char *s, double f, int add_space)
{
  char tmp[TCL_DOUBLE_SPACE];
  Tcl_PrintDouble(uiText -> tclInterp, f, tmp);
  // find the end of s
  while (*s) {
    s++;
  }
  // and append
  if (add_space) {
    *s++ = ' ';
  }
  *s = 0;
  //  strcat(s, tmp);
  //  while (*s) {
  //    s++;
  //  }
  //  return s;
  return my_strcat(s, tmp);
}

////// vmdinfo

int vmdinfo_tcl(ClientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
  if (argc == 2) {
    SIMPLE_TCL_OPT("version", VMDVERSION);
    SIMPLE_TCL_OPT("versionmsg", VERSION_MSG);
    SIMPLE_TCL_OPT("authors", VMD_AUTHORS);
    SIMPLE_TCL_OPT("arch", VMD_ARCH);
    SIMPLE_TCL_OPT("options", VMD_OPTIONS);
    SIMPLE_TCL_OPT("www", VMD_HOMEPAGE);
    SIMPLE_TCL_OPT("wwwhelp", VMD_HELPPAGE);
  }
  Tcl_AppendResult(interp, 
"vmdinfo: version | versionmsg | authors | arch | options | www | wwwhelp",
		   NULL);
  return TCL_ERROR;
}

// given a string ("first", "last", "now", or a value)
// return the timestep value in *val
// returns TCL_ERROR if there was a problem
// on error, if val > 0, the value of s wasn't understood
//           if val < 0, the value was negative
static int tcl_get_frame_value(Tcl_Interp *interp, char *s, int *val)
{
  *val = 1;
  if (!strcmp(s, "last")) {
    *val = AtomSel::TS_LAST;
  }
  if (!strcmp(s, "first")) {
    *val = 0;
  }
  if (!strcmp(s, "now")) {
    *val = AtomSel::TS_NOW;
  }
  if (*val == 1) {
    int new_frame;
    if (Tcl_GetInt(interp, s, &new_frame) != TCL_OK) {
      return TCL_ERROR;
    }
    *val = new_frame;
    if (new_frame < 0) {
      return TCL_ERROR;
    }
  }
  return TCL_OK;
}

/***************** override some of the vector routines for speed ******/
/* These should be the exact C equivalent to the corresponding Tcl    */
/* vector commands */

// Function:  vecadd v1 v2 {v3 ...}
//  Returns: the sum of vectors; must all be the same length
//  The increase in speed from Tcl to C++ is 4561 / 255 == 18 fold
int proc_vecadd(ClientData, Tcl_Interp *interp, int argc, 
		       char *argv[])
{
  if (argc == 1) {
    interp -> result = "no value given for parameter \"x\" to \"vecadd\"";
    return TCL_ERROR;
  }
  if (argc == 2) {
    interp -> result = "no value given for parameter \"y\" to \"vecadd\"";
    return TCL_ERROR;
  }
  int num;
  char **data;
  if (Tcl_SplitList(interp, argv[1], &num, &data) != TCL_OK) {
    return TCL_ERROR;
  }
  double *sum = new double[num];
  int i;
  for (i=0; i<num; i++) {
    if (Tcl_GetDouble(interp, data[i], sum+i) != TCL_OK) {
      delete [] sum;
      free(data);
      return TCL_ERROR;
    }
  }
  free(data);
  // do the sums on the rest
  int num2;
  for (int term=2; term < argc; term++) {
    if (Tcl_SplitList(interp, argv[term], &num2, &data) != TCL_OK) {
      delete [] sum;
      return TCL_ERROR;
    }
    if (num != num2) {
      interp -> result = "vecadd: two vectors don't have the same size";
      delete [] sum;
      free(data);
      return TCL_ERROR;
    }
    for (i=0; i<num; i++) {
      double df;
      if (Tcl_GetDouble(interp, data[i], &df) != TCL_OK) {
	delete [] sum;
	free(data);
	return TCL_ERROR;
      }
      sum[i] += df;
    }
  }

  // and return the result
  char s[TCL_DOUBLE_SPACE];
  for (i=0; i<num; i++) {
    Tcl_PrintDouble(interp, sum[i], s);
    Tcl_AppendElement(interp, s);
  }
  free(data);
  delete [] sum;
  return TCL_OK;
}

// Function:  vecsub  v1 v2
//  Returns:   v1 - v2
int proc_vecsub(ClientData, Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc == 1) {
    interp -> result = "no value given for parameter \"x\" to \"vecsub\"";
    return TCL_ERROR;
  }
  if (argc == 2) {
    interp -> result = "no value given for parameter \"y\" to \"vecsub\"";
    return TCL_ERROR;
  }
  int num1, num2;
  float *data1, *data2;
  if (tcl_get_vector("vecsub: ", interp, argv[1], &num1, &data1) != TCL_OK) {
    return TCL_ERROR;
  }
  if (tcl_get_vector("vecsub: ", interp, argv[2], &num2, &data2) != TCL_OK) {
    delete [] data1;
    return TCL_ERROR;
  }
  if (num1 != num2) {
    interp -> result = "vecadd: two vectors don't have the same size";
    delete [] data1;
    delete [] data2;
    return TCL_ERROR;
  }
  // do the subtraction and return the result
  char s[TCL_DOUBLE_SPACE];
  for (int i=0; i<num1; i++) {
    Tcl_PrintDouble(interp, data1[i] - data2[i], s);
    Tcl_AppendElement(interp, s);
  }
  return TCL_OK;
}


// Function: vecscale
//  Returns: scalar * vector or vector * scalar
// speedup is 1228/225 = 5.5 fold
int proc_vecscale(ClientData, Tcl_Interp *interp, int argc, 
		       char *argv[])
{
  if (argc == 1) {
    interp -> result = "no value given for parameter \"c\" to \"vecscale\"";
    return TCL_ERROR;
  }
    
  if (argc == 2) {
    interp -> result = "no value given for parameter \"v\" to \"vecscale\"";
    return TCL_ERROR;
  }
  if (argc != 3) {
    interp -> result = "called \"vecscale\" with too many arguments";
    return TCL_ERROR;
  }
    
  int num1, num2;
  char **data1, **data2;
  if (Tcl_SplitList(interp, argv[1], &num1, &data1) != TCL_OK) {
    return TCL_ERROR;
  }
  if (Tcl_SplitList(interp, argv[2], &num2, &data2) != TCL_OK) {
    free(data1);
    return TCL_ERROR;
  }
  int result = TCL_OK;
  if (num1 == 0 || num2 == 0) {
    result = TCL_ERROR;
    interp -> result = "vecscale: parameters must have data";
  } else if (num1 != 1 && num2 != 1) {
    result = TCL_ERROR;
    interp -> result = "vecscale: one parameter must be a scalar value";
  } else {
    char *scalar, **vector;
    int num;
    if (num1 == 1) {
      scalar = data1[0];
      vector = data2;
      num = num2;
    } else {
      scalar = data2[0];
      vector = data1;
      num = num1;
    }
    char s[TCL_DOUBLE_SPACE];
    double val1, val2;
    if (Tcl_GetDouble(interp, scalar, &val1) != TCL_OK) {
      result = TCL_ERROR;
    } else {
      for (int i=0; i<num; i++) {
	if (Tcl_GetDouble(interp, vector[i], &val2) != TCL_OK) {
	  interp -> result = "vecscale: vector contains a non-number";
	  result = TCL_ERROR;
	  break;
	}
	Tcl_PrintDouble(interp, val1 * val2, s);
	Tcl_AppendElement(interp, s);
      }
    }
  }
  free(data1);
  free(data2);
  return result;
}


//  Function: transoffset
//   Returns: the transformation correspoding to a vector offset
int proc_transoffset(ClientData, Tcl_Interp *interp, int argc, 
		     char *argv[])
{
  if (argc != 2) {
    interp -> result = "transoffset: takes one parameter, an offset vector";
    return TCL_ERROR;
  }
  // get the vector
  int num;
  float *data;
  if (tcl_get_vector("transoffset: ", interp, argv[1], &num, &data) != 
      TCL_OK) {
    return TCL_ERROR;
  }
  // don't check the size (the script version doesn't)
  Matrix4 t;
  switch (num) {
  case 3: t.mat[3][2] = data[2];  // YES, these fall through!
  case 2: t.mat[3][1] = data[1];
  case 1: t.mat[3][0] = data[0];
  case 0: break;
  }
  delete [] data;
  tcl_append_matrix(interp, t);
  return TCL_OK;
  
}

/// Given a string with a matrix in it, return the matrix
// returns TCL_OK if good
// If bad, returns TCL_ERROR and sets the interp->result to the error message
// The name of the function should be passed in 'fctn' so the error message
// can be constructed correctly
int tcl_get_matrix(char *fctn, Tcl_Interp *interp, 
			  char *s, Matrix4 *mat)
{ 
  int num_rows;
  char **data_rows;
  if (Tcl_SplitList(interp, s, &num_rows, &data_rows) != TCL_OK) {
    sprintf(interp -> result, "%s: badly formed matrix", fctn);
    return TCL_ERROR;
  }
  if (num_rows != 4) {
    free(data_rows);
    sprintf(interp -> result, "%s: need a 4x4 matrix", fctn);
    return TCL_ERROR;
  }
  int num_row[4];
  char **data_row[4];
  data_row[0] = data_row[1] = data_row[2] = data_row[3] = NULL;
  if (Tcl_SplitList(interp, data_rows[0], num_row+0, data_row+0) != TCL_OK ||
      num_row[0] != 4 ||
      Tcl_SplitList(interp, data_rows[1], num_row+1, data_row+1) != TCL_OK ||
      num_row[1] != 4 ||
      Tcl_SplitList(interp, data_rows[2], num_row+2, data_row+2) != TCL_OK ||
      num_row[2] != 4 ||
      Tcl_SplitList(interp, data_rows[3], num_row+3, data_row+3) != TCL_OK ||
      num_row[3] != 4) {
    free(data_rows);
    if (data_row[0]) free(data_row[0]);
    if (data_row[1]) free(data_row[1]);
    if (data_row[2]) free(data_row[2]);
    if (data_row[3]) free(data_row[3]);
    Tcl_AppendResult(interp, fctn, ": poorly formed matrix", NULL);
    return TCL_ERROR;
  }
  free(data_rows);
  // now get the numbers
  double tmp;
  int ret_val = TCL_OK;
  for (int i=0; i<4; i++) {
    for (int j=0; j<4; j++) {
      if (Tcl_GetDouble(interp, data_row[i][j], &tmp) != TCL_OK) {
	ret_val = TCL_ERROR;
	sprintf(interp -> result, "%s: non-numeric in matrix", fctn);
      } else {
	mat -> mat[j][i] = tmp;  // Matrix4 is transpose to Tcl's matrix
      }
    }
  }
  free(data_row[0]);
  free(data_row[1]);
  free(data_row[2]);
  free(data_row[3]);
  return ret_val;
}

// append the matrix into the -> result field of the interp
void tcl_append_matrix(Tcl_Interp *interp, const Matrix4 &mat)
{
  char s[TCL_DOUBLE_SPACE];

  for (int i=0; i<4; i++) {
    Tcl_AppendResult(interp, "{", NULL);
    for (int j=0; j<4; j++) {
      Tcl_PrintDouble(interp, mat.mat[j][i], s);
      Tcl_AppendResult(interp, s, (j != 3 ? " " : ""), NULL);
    }
    Tcl_AppendResult(interp, (i != 3 ? "} " : "}"), NULL);
  }
}


// Given a string with a vector in it, get the vector
// YOU must delete [] the vector (in "result") when finished
// returns TCL_OK if good
// If bad, returns TCL_ERROR and sets the interp->result to the error message
// The name of the function should be passed in 'fctn' so the error message
// can be constructed correctly
int tcl_get_vector(char *fctn, Tcl_Interp *interp, 
			  char *s, int *num, float **result)
{
  *result = NULL;
  *num = 0;
  char **data;
  if (Tcl_SplitList(interp, s, num, &data) != TCL_OK) { // is a list
    Tcl_AppendResult(interp, fctn, ": badly formed vector", NULL);
    return TCL_ERROR;
  }
  *result = new float[*num];
  int ret_val = TCL_OK;
  double tmp;
  for (int i=0; i<*num; i++) {
    if (Tcl_GetDouble(interp, data[i], &tmp) != TCL_OK) {  // of numbers
      sprintf(interp->result, "%s: non-numeric in vector", fctn);
      ret_val = TCL_ERROR;
    } else {
      (*result)[i] = tmp;
    }
  }
  free(data);
  if (ret_val == TCL_ERROR) {
    delete [] (*result);
    *result = NULL;
  }
  return ret_val;
}


// speed up the matrix * vector routines -- DIFFERENT ERROR MESSAGES
// THAN THE TCL VERSION
// speedup is nearly 25 fold
int proc_vectrans(ClientData, Tcl_Interp *interp, int argc, 
		  char *argv[])
{
  if (argc == 1) {
    Tcl_AppendResult(interp, "no value given for parameter \"m\" to \"",
		     argv[0], "\"", NULL);
    return TCL_ERROR;
  }
  if (argc == 2) {
    Tcl_AppendResult(interp, "no value given for parameter \"v\" to \"",
		     argv[0], "\"", NULL);
    return TCL_ERROR;
  }
  if (argc > 3) {
    Tcl_AppendResult(interp, "called \"", argv[0], 
		     "\" with too many arguments", NULL);
    return TCL_ERROR;
  }

  // get the matrix data
  Matrix4 mat;
  if (tcl_get_matrix(argv[0], interp, argv[1], &mat) != TCL_OK) {
    return TCL_ERROR;
  }
  // for the vector
  float *vec;
  int vec_size;
  if (tcl_get_vector(argv[0], interp, argv[2], &vec_size, 
		     &vec) != TCL_OK) {
    return TCL_ERROR;
  }
  float vec_data[4];
  if (vec_size == 3) {
    memcpy(vec_data, vec, 3*sizeof(float));
    if (!strcmp(argv[0], "coordtrans")) {
      vec_data[3] = 1;
    } else {
      vec_data[3] = 0;
    }
  } else {
    if (vec_size == 4) {
      memcpy(vec_data, vec, 4*sizeof(float));
    } else {
      Tcl_AppendResult(interp, argv[0], ": vector must be of size 3 or 4");
      delete [] vec;
      return TCL_ERROR;
    }
  }
  delete [] vec;

  // vector data is in vec_data
  float result[4];
  mat.multpoint4d(vec_data, result);
  // return it
  if (vec_size == 3) {
    char s[TCL_DOUBLE_SPACE];
    Tcl_PrintDouble(interp, result[0], s);
    Tcl_AppendElement(interp, s);
    Tcl_PrintDouble(interp, result[1], s);
    Tcl_AppendElement(interp, s);
    Tcl_PrintDouble(interp, result[2], s);
    Tcl_AppendElement(interp, s);
  } else {
    char s[TCL_DOUBLE_SPACE];
    Tcl_PrintDouble(interp, result[0], s);
    Tcl_AppendElement(interp, s);
    Tcl_PrintDouble(interp, result[1], s);
    Tcl_AppendElement(interp, s);
    Tcl_PrintDouble(interp, result[2], s);
    Tcl_AppendElement(interp, s);
    Tcl_PrintDouble(interp, result[3], s);
    Tcl_AppendElement(interp, s);
  }
  return TCL_OK;
}

// Function: transmult m1 m2 ... mn
//  Returns: the product of the matricies
// speedup is 136347 / 1316 = factor of 104
int proc_transmult(ClientData, Tcl_Interp *interp, int argc, 
		   char *argv[])
{
  // make there there are at least two values
  if (argc <= 1) {
    interp -> result = "no value given for parameter \"mx\" to \"transmult\"";
    return TCL_ERROR;
  }
  if (argc == 2) {
    interp -> result = "no value given for parameter \"my\" to \"transmult\"";
    return TCL_ERROR;
  }
  // Get the first matrix
  Matrix4 mult;
  if (tcl_get_matrix("transmult: ", interp, argv[1], &mult) != TCL_OK) {
    return TCL_ERROR;
  }
  int i = 2;
  Matrix4 tmp;
  while (i < argc) {
    if (tcl_get_matrix("transmult: ", interp, argv[i], &tmp) != TCL_OK) {
      return TCL_ERROR;
    }
    mult.multmatrix(tmp);
    i++;
  }
  tcl_append_matrix(interp, mult);
  return TCL_OK;
}


////////////////////////////////////////////////////////////////////////
// given a string, return the indicated molecule.
// String can be a number or 'top'

static Molecule *find_molecule(Tcl_Interp *interp, char *text)
{
  int molid = -1;
  if (!strcmp(text, "top")) {
    if (moleculeList->top()) {
      molid = moleculeList->top()->id();
    } else {
      Tcl_AppendResult(interp, "There is no 'top' molecule ", NULL);
      return NULL;
    }
  } else {
    if (Tcl_GetInt(interp, text, &molid) != TCL_OK) {
      Tcl_AppendResult(interp, "Not valid molecule id ", text, NULL);
      return NULL;
    }
  }
  // here I have 'molid', so get the given molecule 
  Molecule *mol = moleculeList -> molecule( moleculeList -> 
					    mol_index_from_id(molid));  
  if (!mol) {
    Tcl_AppendResult(interp, "Cannot find molecule ", text, NULL);
  }
  return mol;
}

///// tcl interface to the AtomSel object

// keep track of the generated AtomSel objects
static Tcl_HashTable internal_tcl_atomsel_hash;
static int internal_tcl_atomsel_init = 0;

// forward definitions
static int access_tcl_atomsel(ClientData my_data, Tcl_Interp *interp,
		       int argc, char *argv[]);
static void remove_tcl_atomsel(ClientData my_data);

// given the interpreter and attribute string, construct the array
// mapping from attribute to atomSelParser index
static int split_tcl_atomsel_info(Tcl_Interp *interp, char *opts, 
				  int *num, int **mapping,
				  char ***ret_attribs)
{
  *num = 0;
  *mapping = NULL;

  // make the list of attributes
  char **attribs;
  int num_attribs;
  if (Tcl_SplitList(interp, opts, &num_attribs, &attribs) != TCL_OK) {
    Tcl_AppendResult(interp, "cannot split attributes list", NULL);
    return TCL_ERROR;
  }

  // verify that each attrib is a valid KEYWORD or SINGLEWORD
  // in the atomSelParser
  int *info_index = new int[num_attribs];
  for (int i=0; i<num_attribs; i++) {
    // search for a match to the attribute
    int j = atomSelParser.find_attribute(attribs[i]);

    if (j == -1) { // the name wasn't found, so complain
      Tcl_AppendResult(interp, "cannot find attribute '", 
		       attribs[i], "'", NULL);
      delete [] info_index;
      free(attribs);
      return TCL_ERROR;
    }
    // make sure this is a KEYWORD or SINGLEWORD
    if (atomSelParser.names[j]->is_a != SymbolTableName::KEYWORD &&
	atomSelParser.names[j]->is_a != SymbolTableName::SINGLEWORD) {
      Tcl_AppendResult(interp, "'", attribs[i], 
		       "' is not a keyword or singleword", NULL);
      delete [] info_index;
      free(attribs);
      return TCL_ERROR;
    }
    info_index[i] = j; // make the mapping from attrib to atomSelParser index
  }

  //  free(attribs);
  *ret_attribs = attribs;
  // return the mapping
  *mapping = info_index;
  *num = num_attribs;
  return TCL_OK;
}
				    


// the Tcl command is "atomselect".  It generates 'local' (with upproc)
// functions which return information about the AtomSel selection
// Format is: atomselect <molecule id> <text>
int make_tcl_atomsel(ClientData, Tcl_Interp *interp, int argc, char *argv[])
{
  // the number of "atomselect%u" commands generated
  static unsigned int internal_tcl_atom_count = 0;

  // return a list of all the undeleted selection
  if (argc == 2 && !strcmp(argv[1], "list")) {
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;
    for (entryPtr = Tcl_FirstHashEntry(&internal_tcl_atomsel_hash, &search);
	 entryPtr != NULL;
	 entryPtr = Tcl_NextHashEntry(&search)) {
      Tcl_AppendElement(interp, Tcl_GetHashKey(&internal_tcl_atomsel_hash, 
					       entryPtr));
    }
    return TCL_OK;
  }

  // return a list of the available keywords in the form
  if (argc == 2 && !strcmp(argv[1], "keywords")) {
    for (int i=0; i<atomSelParser.names.num(); i++) {
      Tcl_AppendElement(interp, (char *) (const char *) 
			atomSelParser.names[i] -> visible);
    }
    return TCL_OK;
  }

  // return all the symbol tabke information for the available keywords
  // in the form  {visiblename regex is takes}, where
  //   "is" is one of "int", "float", "string"
  //   "takes" is one of "keyword", "function", "boolean", "sfunction"
  if (argc == 2 && !strcmp(argv[1], "symboltable")) {
    char *pis, *ptakes;
    // go through the parser, one by one
    for (int i=0; i< atomSelParser.names.num(); i++) {
      Tcl_AppendResult(interp, i==0?"":" ", "{", NULL);
      // what kind of function is this?
      switch (atomSelParser.names[i] -> is_a) {
      case SymbolTableName::KEYWORD: ptakes = "keyword"; break;
      case SymbolTableName::FUNCTION: ptakes = "function"; break;
      case SymbolTableName::SINGLEWORD: ptakes = "boolean"; break;
      case SymbolTableName::STRINGFCTN: ptakes = "sfunction"; break;
      default: ptakes = "unknown"; break;
      }
      // what does it take?
      switch (atomSelParser.names[i] -> takes_a) {
      case SymbolTableName::IS_INT : pis = "int"; break;
      case SymbolTableName::IS_FLOAT : pis = "float"; break;
      case SymbolTableName::IS_STRING : pis = "string"; break;
      default: pis = "unknown"; break;
      }
      // append to the result string
      Tcl_AppendElement(interp, (char *) (const char *) atomSelParser.names[i] 
			-> visible);
      Tcl_AppendElement(interp, (char *) (const char *) atomSelParser.names[i] 
			-> pattern);
      Tcl_AppendElement(interp, pis);
      Tcl_AppendElement(interp, ptakes);
      Tcl_AppendResult(interp, "}", NULL);
    }
    return TCL_OK;
  }

  if (!((argc == 3) || (argc == 5 && !strcmp(argv[3], "frame")))) {
    interp->result = 
      "atomselect <molId> <selection string> [frame <n>]\n"
      "atomselect [list|keywords|symboltable]"
      ;
    return TCL_ERROR;
  }
  int frame = AtomSel::TS_NOW;
  if (argc == 5) { // get the frame number
    int val;
    if (tcl_get_frame_value(interp, argv[4], &val) != TCL_OK) {
      interp -> result = "atomselect: bad frame number in input, must be "
	"'first', 'last', 'now', or a non-negative number";
      return TCL_ERROR;
    }
    frame = val;
  }
      
  // get the molecule id
  Molecule *mol = find_molecule(interp, argv[1]);
  if (!mol) {
    Tcl_AppendResult(interp, " in atomselect's 'molId'", NULL);
    return TCL_ERROR;
  }
  // do the selection 
  AtomSel *atomSel = new AtomSel(moleculeList);
  atomSel -> which_frame = frame;
  if (atomSel->change(argv[2]) == AtomSel::NO_PARSE) {
    Tcl_AppendResult(interp, "atomselect: cannot parse selection text: ",
		     argv[2], NULL);
    return TCL_ERROR;
  }
  // do the search
  if (atomSel->find(mol) < 0) {
    // something went wrong
    interp -> result = "select: unknown error in AtomSel.find";
    return TCL_ERROR;
  }

  // At this point the data is okay so construct the new function

  // add this to the hash table
  if (internal_tcl_atomsel_init == 0) {
    internal_tcl_atomsel_init = 1;
    Tcl_InitHashTable(&internal_tcl_atomsel_hash, TCL_STRING_KEYS);
  }
  // make the name
  char newname[30];
  sprintf(newname, "atomselect%d", internal_tcl_atom_count++);

  // it is guaranteeded unique
  {
    int is_new;
    Tcl_HashEntry *entryPtr = Tcl_CreateHashEntry(&internal_tcl_atomsel_hash,
						  newname, &is_new);
    Tcl_SetHashValue(entryPtr, atomSel);
  }

  // make the new proc
  Tcl_CreateCommand(interp, newname, access_tcl_atomsel, 
		    (ClientData) atomSel, 
		    (Tcl_CmdDeleteProc *) remove_tcl_atomsel);

  // here I need to change the context ...
  Tcl_VarEval(interp, "upproc 0 ", newname, NULL);

  // return the new function name and return it
  Tcl_AppendElement(interp, newname);
  return TCL_OK;
}

// given the tcl variable string, get the selection
AtomSel *tcl_commands_get_sel(char *str)
{
  if (internal_tcl_atomsel_init == 0) {
    internal_tcl_atomsel_init = 1;
    Tcl_InitHashTable(&internal_tcl_atomsel_hash, TCL_STRING_KEYS);
  }
  Tcl_HashEntry *entryPtr = Tcl_FindHashEntry(&internal_tcl_atomsel_hash, 
					      str);
  if (entryPtr == NULL) return NULL;
  return (AtomSel *) Tcl_GetHashValue(entryPtr);
}

// improve the speed of 'move' and 'moveby'
// needs a selection and a matrix
//  Applies the matrix to the coordinates of the selected atoms
int proc_vmd_atomselect_move(ClientData, Tcl_Interp *interp, int argc, 
			     char *argv[])
{
  if (argc != 3) {
    interp -> result = "atomselection move: needs a selection and a matrix";
    return TCL_ERROR;
  }
  // get the selection
  AtomSel *sel = tcl_commands_get_sel(argv[1]);
  if (!sel) {
    interp -> result = "atomselection move: not a valid selection";
    return TCL_ERROR;
  }
  if (!sel -> molecule()) {
    interp -> result = "atomselection move: molecule was deleted";
      return TCL_ERROR;
  }
  // get the frame
  float *framepos = sel -> coordinates();
  if (!framepos) {
    interp -> result = "atomselection move: invalid/ no coordinates in "
      "selection";
    return TCL_ERROR;
  }

  // get the matrix
  Matrix4 mat;
  if (tcl_get_matrix("atomselection move:", interp, argv[2], &mat) !=
      TCL_OK) {
    return TCL_ERROR;
  }
  // and apply it to the coordinates
  int num = sel -> num_atoms;
  float tmp[3];
  for (int i=0; i<num; i++) {
    if (sel -> on[i]) {
      mat.multpoint3d(framepos + 3*i, tmp);
      copy(framepos + 3*i, tmp);
    }
  }
  ((DrawMolecule *) sel -> molecule()) -> force_recalc();
  return TCL_OK;
}
// and the same for the vector offset
//  Applies the vector to the coordinates of the selected atoms
int proc_vmd_atomselect_moveby(ClientData, Tcl_Interp *interp, int argc, 
			       char *argv[])
{
  if (argc != 3) {
    interp -> result = "atomselection moveby: needs a selection and a vector";
    return TCL_ERROR;
  }
  // get the selection
  AtomSel *sel = tcl_commands_get_sel(argv[1]);
  if (!sel) {
    interp -> result = "atomselection moveby: not a valid selection";
    return TCL_ERROR;
  }
  if (!sel -> molecule()) {
    interp -> result = "atomselection moveby: molecule was deleted";
      return TCL_ERROR;
  }
  // get the frame
  float *framepos = sel -> coordinates();
  if (!framepos) {
    interp -> result = "atomselection moveby: invalid/ no coordinates in "
      "selection";
    return TCL_ERROR;
  }

  // get the matrix
  float *vect;
  int num_vect;
  if (tcl_get_vector("atomselection moveby:", interp, argv[2], 
		     &num_vect, &vect) != TCL_OK) {
    return TCL_ERROR;
  }
  if (num_vect != 3) {
    interp -> result = "atomselection moveby: translation vector can only "
      "be of length 3";
    free(vect);
    return TCL_ERROR;
  }
  // and apply it to the coordinates
  int num = sel -> num_atoms;
  for (int i=0; i<num; i++) {
    if (sel -> on[i]) {
      add(framepos + 3*i, framepos + 3*i, vect);
    }
  }
  free(vect);
  ((DrawMolecule *) sel -> molecule()) -> force_recalc();
  return TCL_OK;
}


// methods related to a selection
//0  num       -- number of atoms selected
//1  list      -- list of atom indicies
//2  molid     -- id of the molecule used
//3  text      -- the selection text
//4  get {options}  -- return a list of the listed data for each atom
//5  set {options} {{values1} ... {valuesn}} -- set the listed values
//6  type      -- returns "atomselect"
//20 frame     -- returns the value of the frame (or 'now' or 'last')
//21 frame <num> -- sets the frame value given the name or number
///// these are defered to other Tcl functions
//7  moveby {x y z}    -- move by a given {x y z} offset
//8  lmoveby {{x y z}} -- move by a list of {x y z} offsets, 1 per atom
//9  moveto {x y z}    -- move to a given {x y z} offset
//10 lmoveto {{x y z}  -- same as 'set {x y z}'
/////
//11 move {transformation}   -- takes a 4x4 transformation matrix
/////
//12 delete    -- same as 'rename $sel {}'
//13 global    -- same as 'upproc #0 $argv[0]'
//14 uplevel L -- same as 'upproc $argv[1] $argv[0]'
#define CHECK_MATCH(string,val) if(!strcmp(argv[1],string)){option=val;break;}

int access_tcl_atomsel(ClientData my_data, Tcl_Interp *interp,
		       int argc, char *argv[]) {

  AtomSel *atomSel = (AtomSel *) my_data;
  if (atomSel == NULL) {
    interp -> result = "atomselect access without data!";
    return TCL_ERROR;
  }
  atomSel->use();
  int option = -1;
  char *outfile_name = NULL;  // for 'writepdb'
  while (1) {
    if (argc == 2) {
      CHECK_MATCH("num", 0);
      CHECK_MATCH("list", 1);
      CHECK_MATCH("molindex", 2);
      CHECK_MATCH("molid", 2);
      CHECK_MATCH("text", 3);
      CHECK_MATCH("type", 6);
      CHECK_MATCH("delete", 12);
      CHECK_MATCH("global", 13);
      CHECK_MATCH("frame", 20);
      CHECK_MATCH("writepdb", 22);
    } else if (argc == 3) {
      CHECK_MATCH("get", 4);
//      CHECK_MATCH("moveby", 7);   // these now pass via the "extended"
//      CHECK_MATCH("lmoveby", 8);  // Tcl functionality
//      CHECK_MATCH("moveto", 9);
//      CHECK_MATCH("lmoveto", 10);
//      CHECK_MATCH("move", 11);
      CHECK_MATCH("uplevel", 14);
      CHECK_MATCH("frame", 21);
      CHECK_MATCH("writepdb", 23);
    } else if (argc == 4) {
      CHECK_MATCH("set", 5);
    }
    // see if it is an "extended" function of the form
    // vmd_atomselect_$1   // yes, $1, not $0, which is the selection name
    {
      GString s = "vmd_atomselect_";
      s += argv[1];
      Tcl_CmdInfo info;
      if (Tcl_GetCommandInfo(interp, (char *) (const char *) s, &info)) {
	// construct the options list
	char *tmp = argv[1];  // get the 'firstword'
	argv[1] = argv[0];    // make the selection the second word
	argv[0] = (char *) (const char *)s;  // name of the new command
	int result = (*info.proc)(info.clientData, interp, argc, argv);
	argv[0] = argv[1];
	argv[1] = tmp;
	return result;
      }
    }
    if (argc != 1) {
      // gave some wierd keyword
      Tcl_AppendResult(interp, "atomselection: improper method: ", argv[1],
		       "\n", NULL);
    }
    // Now list the available options
    Tcl_AppendResult(interp, 
	     "[num|list|molid|text|type|delete|global|frame|writepdb]\n",
	     "get <attributes> {for parameter list, use 'atomselect list'\n",
	     "set <attributes> <list in the same form as returned from get>\n",
	     "frame <new frame value>\n",
	     "[moveto|moveby] <3 vector>\n",
	     "[lmoveto|lmoveby] <x> <y> <z>\n",
	     "move <4x4 transforamtion matrix>\n",
	     "writepdb <|filename>   {default is to the screen}",
		     NULL);
    return TCL_ERROR;
  }

  switch(option) {
  case 0: { // num
    sprintf(interp->result, "%d", atomSel -> selected);
    return TCL_OK;
  }
  case 1: { // list
    char s[10];
    for (int i=0; i<atomSel->num_atoms; i++) {
      if (atomSel->on[i]) {
	sprintf(s, "%d", i);
	Tcl_AppendElement(interp, s);
      } 
    }
    return TCL_OK;
  }
  case 2: { // molid
    sprintf(interp -> result, "%d", atomSel->molid);
    return TCL_OK;
  }
  case 3: { // text
    Tcl_AppendElement(interp, atomSel -> cmdStr);
    return TCL_OK;
  }
  case 20: { // frame
    switch (atomSel->which_frame) {
      case AtomSel::TS_LAST: sprintf(interp -> result, "last"); break;
      case AtomSel::TS_NOW : sprintf(interp -> result, "now"); break;
      default:
	sprintf(interp -> result, "%d", atomSel->which_frame);
    }
    return TCL_OK;
  }
  case 21: { // frame <num>
    int val;
    if (tcl_get_frame_value(interp, argv[2], &val) != TCL_OK) {
      Tcl_AppendResult(interp, "atomsel: frame '", argv[2], "' invalid; ",
	 "please use a number >=0 or 'first', 'last', or 'now'", NULL);
      return TCL_ERROR;
    }
    atomSel -> which_frame = val;
    return TCL_OK;
  }
  case 4: { // get
    // check that the molecule exists
    int index = moleculeList->mol_index_from_id(atomSel -> molid);
    if (index == -1) {
      sprintf(interp->result, "atomsel: get: was molecule %d deleted?",
	      atomSel->molid);
      return TCL_ERROR;
    }
    int num_atoms = atomSel -> num_atoms;
    // get the mapping
    int *mapping;
    int num_mapping;
    char **attribs;
    if (split_tcl_atomsel_info(interp, argv[2], &num_mapping, 
			       &mapping, &attribs) != TCL_OK) {
      Tcl_AppendResult(interp, ": in atomsel: get:", NULL);
      return TCL_ERROR;
    }

    // get the requested information
    {
      GString **attribs_data = (GString **) malloc(num_mapping *
						   sizeof(GString *));
      int i;
      for (i=0; i<num_mapping; i++) {
	// space for the data
	attribs_data[i] = new GString[num_atoms];

	// if this is a boolean singleword, special case it
	if (atomSelParser.names[mapping[i]]->is_a == 
	    SymbolTableName::SINGLEWORD) {
	  // get the boolean state
	  parsetree_nodestring = attribs[i]; // hack for $ and @ references
	  SymbolTableElement *fctn=atomSelParser.fctns[mapping[i]];
	  int *flgs = new int[num_atoms];
	  memcpy(flgs, atomSel->on, num_atoms * sizeof(int));
	  fctn->keyword_single(num_atoms, flgs);
	  for (int j=0; j<num_atoms; j++) {
	    if (atomSel -> on[j]) {
	      if (flgs[j]) {
		attribs_data[i][j] = "1";
	      } else {
		attribs_data[i][j] = "0";
	      }
	    }
	  }
	  delete [] flgs;
	} else { // then this is a keyword, and I already have routines to use
	  parsetree_nodestring = attribs[i]; // hack for $ and @ references
	  atomSelParser.extract_keyword_info(mapping[i], num_atoms,
					     attribs_data[i], atomSel -> on);
	}
      }

      // return the information to Tcl
      for (i=0; i<num_atoms; i++) {
	int first = 0;
	if (atomSel->on[i]) {
	  Tcl_AppendResult(interp, (first++ ? "{" : " {"), NULL);
	  for (int j=0; j<num_mapping; j++) {
	    Tcl_AppendElement(interp, (char *) (const char *)
			      attribs_data[j][i]);
	  }
	  Tcl_AppendResult(interp, "}", NULL);
	}
      }
      // free the arrays
      for (i=0; i<num_mapping; i++) {
	delete [] attribs_data[i];
      }
      free(attribs_data);
      free(attribs);
      delete [] mapping;
    }
    return TCL_OK;
  }
  case 5: {  // set
    // check that the molecule exists
    int index = moleculeList->mol_index_from_id(atomSel -> molid);
    if (index == -1) {
      sprintf(interp->result, "atomsel: set: was molecule %d deleted?",
	      atomSel->molid);
      return TCL_ERROR;
    }
    Molecule *mol = moleculeList->molecule(index);

    // get the mapping from keywords to function index
    int *mapping;
    int num_mapping;
    char **attribs;
    if (split_tcl_atomsel_info(interp, argv[2], &num_mapping, 
			       &mapping, &attribs) != TCL_OK) {
      Tcl_AppendResult(interp, ": in atomsel: set:", NULL);
      return TCL_ERROR;
    }

    // at this point I know the values of "mapping" are good functions
    // now check that the are "writable"
    {
      int problem = 0;
      for (int i=0; i<num_mapping; i++) {
	if (!atomSelParser.is_changeable(mapping[i])) {
	  if (!problem) {
	    Tcl_AppendResult(interp, 
			     "atomsel object: set: data not modifiable:",
			     NULL);
	    problem = 1;
	  }
	  Tcl_AppendResult(interp, " ", (const char *) 
			   atomSelParser.names[mapping[i]] -> visible, NULL);
	}
      }
      if (problem) {
	free(attribs);
	delete [] mapping;
	return TCL_ERROR;
      }
    }

    // well, what do you know.  I _can_ change the data.

    // Convert the data from {a1 b1 c1 ... } {a2 b2 c3 ...}  ...
    // into {a1 a2 a3 ...} {b1 b2 b3 ... } ...
    // reconstruct the lists into values
    char **outer_list;
    int num_outer_list;
    if (Tcl_SplitList(interp, argv[3], &num_outer_list, &outer_list)
	!= TCL_OK) {
      interp -> result = "atomselect: set: Bad data list";
      free(attribs);
      delete [] mapping;
      return TCL_ERROR;
    }
    // make sure there is enough data
    if (num_outer_list != atomSel->selected) {
      if (num_outer_list == 1) { // if there is one element, copy 'n' times
	free(outer_list);
	char *s = new char[atomSel->selected * (strlen(argv[3])+3)+1];
	s[0] = 0;
	char *ss = s;
	for (int i=0; i<atomSel->selected; i++) {
	  ss = my_strcat(ss, "{");
	  ss = my_strcat(ss, argv[3]);
	  ss = my_strcat(ss, "} ");
	}
	// and make a new split
	if (Tcl_SplitList(interp, s, &num_outer_list, &outer_list)
	    != TCL_OK) {
	  interp -> result = "atomselect: set: Bad data list (while trying "
	    "to make copies)";
	  free(attribs);
	  delete [] mapping;
	  delete [] s;
	  return TCL_ERROR;
	}
	delete [] s;  // otherwise, I now have n copies
      } else {
	sprintf(interp->result, 
		"atomselect: set: %d elements in list, but %d in selection",
		num_outer_list, atomSel->selected);
	free(attribs);
	delete [] mapping;
	free(outer_list);
	return TCL_ERROR;
      }
    }
    // So the {a1 b1 ...} {a2 b2 ...} lists are broken up.
    // Break each up into another list and rearrange and save the parts
    int is_error = 0;
    char ***data = (char ***) malloc(sizeof(char **) * num_outer_list);
    {
      int i;
      for (i=0; i<num_outer_list; i++) {
	data[i] = NULL;
      }
      for (i=0; i<num_outer_list; i++) {
	int num;
	Tcl_SplitList(interp, outer_list[i], &num, data+i);
	if (num != num_mapping) {
	  sprintf(interp->result, "atomselect: set: data element %d "
		  "has %d terms (instead of %d)", i, num, num_mapping);
	  is_error = 1;
	  break;
	}
      }
    }
    if (! is_error) {
      char **list = (char **) malloc(sizeof(char *) * atomSel->num_atoms);
      
      // set the pointers to the correct offsets
      for (int j=0; j<num_mapping; j++) {
	int count = 0;
	int i;
	for (i=0; i<atomSel -> num_atoms; i++) {
	  if (atomSel -> on[i]) {
	    list[i] = data[count++][j];
	  } else {
	    list[i] = NULL;
	  }
	}
	parsetree_nodestring = attribs[i]; // hack for $ and @ references
	atomSelParser.set_keyword_info(mapping[j], atomSel -> num_atoms, 
				       list, atomSel->on);
      }
      free(list);
    }


    // here I delete the SplitList arrays
    {
      for (int i=0; i<num_outer_list; i++) {
	if (data[i] != NULL) free(data[i]);
      }
      free(data);
      free(outer_list);
      free(attribs);
      delete [] mapping;
    }
    return TCL_OK;
  }
  case 6: // type
    interp->result = "atomselect";
    return TCL_OK;
  case 7: // moveby
    // proc vmd_atomselect_moveby {sel vect} {
    //    set coords ""
    //    foreach v [$sel get {x y z}] [
    //       lappend coords [vecadd $v $vect]
    //    }
    //    $sel set {x y z} $coords
    // }
    return Tcl_VarEval(interp, "vmd_atomselect_moveby {", argv[0], "} {", 
		       argv[2], "}", NULL);
  case 8: // lmoveby
    return Tcl_VarEval(interp, "vmd_atomselect_lmoveby {", argv[0], "} {", 
		       argv[2], "}", NULL);
  case 9: // moveto
    return Tcl_VarEval(interp, "vmd_atomselect_moveto {", argv[0], "} {", 
		       argv[2], "}", NULL);
  case 10: // lmoveto
    return Tcl_VarEval(interp, "vmd_atomselect_lmoveto {", argv[0], "} {", 
		       argv[2], "}", NULL);
  case 11: // move {transformation}
    return Tcl_VarEval(interp, "vmd_atomselect_move {", argv[0], "} {",
		       argv[2], "}", NULL);
  case 12: // delete
    return Tcl_VarEval(interp, "rename ", argv[0], " {}", NULL);
  case 13: // global
    return Tcl_VarEval(interp, "upproc #0 ", argv[0], NULL);
  case 14: // uplevel
    return Tcl_VarEval(interp, "upproc ", argv[1], " ", argv[0], NULL);

  case 23: {   // writepdb <name>
    // if outfile_name != NULL, case 22 opens a file
    outfile_name = argv[2];
  } // NOTE: deliberate fall through !
  case 22: {   // writepdb
    // outfile_name == NULL => stdout
    // otherwise, open a new file

    // check that the molecule exists
    int index = moleculeList->mol_index_from_id(atomSel -> molid);
    Molecule *mol = moleculeList->molecule(index);
    if (index == -1 || !mol) {
      sprintf(interp->result, "atomsel: writepdb: was molecule %d deleted?",
	      atomSel->molid);
      return TCL_ERROR;
    }

    // check that the selection's timestep exists
    Timestep *ts = NULL;
    switch (atomSel -> which_frame) {
    case AtomSel::TS_NOW: ts = mol -> current(); break;
    case AtomSel::TS_LAST: ts = mol -> item(mol -> Animation::num()-1); break;
    default: ts = mol -> item(atomSel -> which_frame); break;
    }
    if (!ts) {
      if (atomSel -> which_frame >= 0) {
	sprintf(interp->result, "atomsel: writepdb: frame %d out of range "
		"for molecule %d", atomSel -> which_frame, atomSel -> molid);
      } else {
	sprintf(interp->result, "atomsel: writepdb: no coordinate frames "
		"in molecule %d", atomSel -> molid);
      }
      return TCL_ERROR;
    }

    // open the file (if need be)
    FILE *outfile;
    if (outfile_name) {
      outfile = fopen(outfile_name, "w");
      if (!outfile) {
	Tcl_AppendResult(interp, "atomsel: writepdb: cannot open file '",
			 outfile_name, "'\n", strerror(errno), NULL);
	return TCL_ERROR;
      }
    } else {
      outfile = stdout;
    }

    // write the selected elements to the PDB 
    fprintf(outfile,
	    "REMARK %d atoms from VMD's atomselect writepdb output\n", 
	    atomSel -> selected);
    int count = 1;  // PDB files start at one
    for (int i=0; i < atomSel -> num_atoms; i++) {
      if (atomSel -> on[i]) {
	write_pdb_record(mol, ts, i, outfile, count++);
      }
    }
    fprintf(outfile, "END\n");
    if (outfile_name) {
      fclose(outfile);
    }
  }
  return TCL_OK;
    
  default:
    break;
  }
  interp->result = "atomselect: error: major wierdness!";
  return TCL_ERROR;
}

      // an "atomselect%u" is to be deleted, remove it from the hash

void remove_tcl_atomsel(ClientData my_data) {
  AtomSel *atomSel = (AtomSel *) my_data;
  if (atomSel == NULL) {
    msgErr << "atomselect deletion without data!" << sendmsg;
    return;
  }
  // find it in the hash table

    Tcl_HashSearch searchPtr;
    Tcl_HashEntry *hash;
    for (hash = Tcl_FirstHashEntry(&internal_tcl_atomsel_hash, &searchPtr);
	 hash != NULL; hash = Tcl_NextHashEntry(&searchPtr)) {
	   if (Tcl_GetHashValue(hash) == atomSel) {
	     break;
	   }
    }
    if (hash == NULL) {
      msgErr << "atomselect deletion cannot find data in table!" << sendmsg;
      return;
    }
    delete atomSel;
    Tcl_DeleteHashEntry(hash);
    //    msgErr << "Ich bin hier!" << sendmsg;
}





// Function:  molinfo
// Option  :  molinfo num
//  Returns:   number of molecules
// Option  :  molinfo index <int>
//  Returns:   molecule id of the nth molecule (starting at index = 0)
// Option  :  molinfo list
//  Returns:   list of all molecule ids
// Option  :  molinfo top
//  Returns:   molecule id of the 'top' molecule
// Option  :  molinfo {molecule number} get <data>
//  Returns:   the given data for that molecule
// Option  :  molinfo {molecule number} set <data> <data fields>
//  Does (okay, this isn't a 'info' thing): sets the data field(s)

int molecule_tcl(ClientData , Tcl_Interp *interp, int argc, char *argv[]) 
{
  // what does it want?
  if (argc == 2) {

// Option  :  molinfo num
//  Returns:   number of molecules
    if (!strcmp(argv[1], "num")) {
      sprintf(interp->result, "%d", moleculeList->num());
      return TCL_OK;
    }

// Option  :  molinfo list
//  Returns:   list of all molecule ids
    if (!strcmp(argv[1], "list")) {
      if (moleculeList -> num() <= 0) {
	return TCL_OK;
      }
      char s[20];
      sprintf(s, "%d", moleculeList->molecule(0)->id());
      Tcl_AppendResult(interp, s, (char *) NULL);
      for (int i=1; i<moleculeList -> num(); i++) {
	sprintf(s, "%d", moleculeList->molecule(i)->id());
	Tcl_AppendResult(interp, " ", s, (char *) NULL);
      }
      return TCL_OK;
    }
// Option  :  molinfo top
//  Returns:   molecule id of the 'top' molecule
    if (!strcmp(argv[1], "top")) {
      if (moleculeList->top()) {
	sprintf(interp->result, "%d", moleculeList->top()->id());
      } else {
	interp->result = "-1";
      }
      return TCL_OK;
    }
    
    // otherwise, I don't know
    Tcl_AppendResult(interp, "molinfo: couldn't understand '",
		     argv[1], "'", NULL);
    return TCL_ERROR;
  } // end of commands with only one option

  if (argc == 3) { // commands with two options
    int val;
    if (Tcl_GetInt(interp, argv[2], &val) != TCL_OK) {
      return TCL_ERROR;
    }
// Option  :  molecule index <int>
//  Returns:   molecule id of the nth molecule (starting at index = 0)
    if (!strcmp(argv[1], "index")) {
      if (moleculeList->molecule(val)) {
	sprintf(interp->result, "%d", moleculeList->molecule(val)->id());
      } else {
	interp -> result = "-1";
      }
      return TCL_OK;
    }
    Tcl_AppendResult(interp, "molinfo: couldn't understand '",
		     argv[1], "'", NULL);
    return TCL_ERROR;
  }

// Option  :  molinfo {molecule number} get <data> [frame <number>]
//  Returns:   the given data for that molecule
  if ((argc == 4 && !strcmp(argv[2], "get")) ||
      (argc == 6 && !strcmp(argv[2], "get") && !strcmp(argv[4], "frame"))) {
    int frame_num;
    if (argc == 4) {
      frame_num = AtomSel::TS_NOW;
    } else {
      if (tcl_get_frame_value(interp, argv[5], &frame_num) != TCL_OK) {
	interp -> result = "atomselect: bad frame number in input, must be "
	  "'first', 'last', 'now', or a non-negative number";
	return TCL_ERROR;
      }
    }
    int val;
    // get the molecule name recursively
    if (!strcmp(argv[1], "top")) {
      if (Tcl_VarEval(interp, argv[0], " top", NULL) != TCL_OK ||
	  Tcl_GetInt(interp, interp->result, &val) != TCL_OK     ) {
	return TCL_ERROR;
      }
    } else {
      if (Tcl_GetInt(interp, argv[1], &val) != TCL_OK) {
	return TCL_ERROR;
      }
    }
    Tcl_ResetResult(interp);

    // split the data into the various terms
    char **list;
    int num_list;
    if (Tcl_SplitList(interp, argv[3], &num_list, &list) != TCL_OK) {
      return TCL_ERROR;
    }
    // and return the information
    int result = molinfo_get(val, num_list, list, interp, frame_num);
    free(list);
    return result;
  }
// Option  :  molinfo {molecule number} set <data> <new data> [frame <number>]
//  Does   :   sets the given data for that molecule
  if ((argc == 5 && !strcmp(argv[2], "set")) ||
      (argc == 7 && !strcmp(argv[2], "set") && !strcmp(argv[5], "frame"))) {
    // get the frame number
    int frame_num;
    if (argc == 5) {
      frame_num = AtomSel::TS_NOW;
    } else {
      if (tcl_get_frame_value(interp, argv[6], &frame_num) != TCL_OK) {
	interp -> result = "atomselect: bad frame number in input, must be "
	  "'first', 'last', 'now', or a non-negative number";
	return TCL_ERROR;
      }
    }
    
    int val;
    if (!strcmp(argv[1], "top")) {
      if (Tcl_VarEval(interp, argv[0], " top", NULL) != TCL_OK ||
	  Tcl_GetInt(interp, interp->result, &val) != TCL_OK     ) {
	return TCL_ERROR;
      }
    } else {
      if (Tcl_GetInt(interp, argv[1], &val) != TCL_OK) {
	return TCL_ERROR;
      }
    }
    Tcl_ResetResult(interp);

    // make sure the two lists have the same number of terms
    char **list1, **list2;
    int num_list1, num_list2;
    if (Tcl_SplitList(interp, argv[3], &num_list1, &list1) != TCL_OK) {
      return TCL_ERROR;
    }
    if (Tcl_SplitList(interp, argv[4], &num_list2, &list2) != TCL_OK) {
      free(list1);
      return TCL_ERROR;
    }
    if (num_list1 != num_list2) {
      free(list1); free(list2);
      interp -> result = 
	"molinfo: set: argument and value lists have different sizes";
      return TCL_ERROR;
    }

    // call the 'set' routine
    int result = molinfo_set(val, num_list1, list1, list2, interp, frame_num);

    free(list1); free(list2);
    return result;
  }

  // There's been an error; find out what's wrong
  interp -> result = "molinfo: called with unknown command";
  if (argc >= 3) {
    if (!strcmp(argv[2], "get")) {
      interp -> result = "molinfo: incorrect format for 'get'";
    } else if (!strcmp(argv[2], "set")) {
      interp -> result = "molinfo: incorrect format for 'set'";
    }
  } else if (argc >= 2) {
    if (!strcmp(argv[1], "get") || strcmp(argv[1], "set")) {
      interp -> result = "molinfo: missing molecule number";
    }
  }

  return TCL_ERROR;
}

//////// the "feedback [message|confirm|choice] <options>" command
//// message
// message "text1" ["text2" ["text3"]]
//// alert
// message "text1" ["text2" ["text3"]]
//// confirm 
// confirm "text1" ["text2" ["text3"]]
//// choice
// choice "text1" "text2" "text3" "button1" ["button2" ["button3"]]
//// input
// input [description [default]]
//// filename
// filename [message [directory [pattern [default]]]]
#include "startup.h"


#ifdef VMDFORMS
#include "forms.h"
 // change in the latest forms library
#if FL_INCLUDE_VERSION > 81
static int fl_show_question_old(char *s1, char *s2, char *s3) {
  char *s;
  s = new char[strlen(s1)+strlen(s2)+strlen(s3)+5];
  s[0] = 0;
  if (s1) strcpy(s, s1);
  if (s2) {strcat(s, s2); strcat(s, "\n");}
  if (s3) {strcat(s, s3); strcat(s, "\n");}
  int i = fl_show_question(s, 1);
  delete [] s;
  return i;
}
#define fl_show_question(s1,s2,s3) fl_show_question_old(s1,s2,s3)
#endif

#if FL_INCLUDE_VERSION > 81
 // another change in the forms library, grrrrr....
static int fl_show_choice_old(char *s1, char *s2, char *s3, int n,
			      char *c1, char *c2, char *c3) {
  char *s;
  s = new char[strlen(s1)+strlen(s2)+strlen(s3)+5];
  s[0] = 0;
  if (s1) strcpy(s, s1);
  if (s2) {strcat(s, s2); strcat(s, "\n");}
  if (s3) {strcat(s, s3); strcat(s, "\n");}
  int i = fl_show_choice(s, n, c1, c2, c3, 0);
  delete [] s;
  return i;
}
#define fl_show_choice(s1,s2,s3,n,c1,c2,c3) \
        fl_show_choice_old(s1,s2,s3,n,c1,c2,c3)
#endif

#else

// make everything a text input
#define fl_show_message(s1, s2, s3) text_fl_show_message(s1, s2, s3) 
#define fl_show_alert(s1, s2, s3, c) text_fl_show_alert(s1, s2, s2)
#define fl_show_question(s1, s2, s3) text_fl_show_question(s1, s2, s3)
#define fl_show_choice(s1,s2,s3,n,b1,b2,b3) \
           text_fl_show_choice(s1,s2,s3,n,b1,b2,b3)
#define fl_show_input(s1,s2) text_fl_show_input(s1,s2)
#define fl_show_fselector(msg,dir,pat,def) \
          text_fl_show_input(msg, def)
#endif  // VMDFORMS

// do everything in cin/cout text
static void text_fl_show_message(const char *s1, const char *s2, 
				 const char *s3) {
  cout << s1 << "\n" << s2 << "\n" << s3 << "\n";
  char s[101];
  cout << "(press enter to continue) " << flush;
  fgets(s, 100, stdin);
}
static void text_fl_show_alert(const char *s1, const char *s2,
			       const char *s3) {
  cout << s1 << "\n" << " - - - - - - - - - - - - - - -\n"
       << s2 << "\n" << s3 << "\n";
  char s[100];
  cout << "(press enter to continue) " << flush;
  fgets(s, 100, stdin);
}
static int text_fl_show_question(const char *s1, const char *s2, 
				 const char *s3) {
  cout << s1 << "\n" << s2 << "\n" << s3 << "\n";
  char c;
  while (1) {
    cout << "(Enter Y or N):" << flush;
    cin >> c;
    if (c == 'Y' || c == 'y') return 1;
    if (c == 'N' || c == 'n') return 0;
  }
}
static int text_fl_show_choice(char *s1, char *s2, char *s3,
			       int n, char *b1, char *b2, char *b3)
{
  cout << s1 << "\n" << s2 << "\n" << s3 << "\n";
  cout << "Pick one of the following:\n";
  if (n > 0) {
    cout << "1. " << b1 << "\n";
  }
  if (n > 1) {
    cout << "2. " << b2 << "\n";
  }
  if (n > 2) {
    cout << "3. " << b3 << "\n";
  }
  char s[100];
  int i;
  while (1) {
    cout << "Choose a number from 1 to " << n << " : " << flush;
    cin >> s;
    i = atoi(s);
    if (n == 0) return 1;
    if (i > 0 && i <= n) return i;
  }
}

#define text_fl_show_fselector(msg,dir,pat,def) \
           text_fl_show_input(msg, def)
static const char * text_fl_show_input(char *s1, char *s2)
{
  static char s[201];
  if (!s1[0]) {
    cout << "Enter a string [" << s2 << "]: " << flush;
  } else {
    cout << s1 << " [" << s2 << "]: " << flush;
  }
  fgets(s, 200, stdin); // chop off that "\n"
  { char *t = s; while (*t) {if (*t == '\n') {*t = 0; break;} t++;}}
  if (!s[0]) {
    strncpy(s, s2, 200);
  }
  s[200] = 0;
  return s;
}

int feedback_tcl(ClientData , Tcl_Interp *interp, int argc, char **argv)
{
  int use_text = (which_display == DISPLAY_TEXT);

  if (argc < 2) {
    interp -> result = "feedback [message | alert | confirm | choice | "
      "input | filename] <options>";
    return TCL_ERROR;
  }
  int msg = -1;
  if (!strcmp(argv[1], "message")) msg = 0;
  if (!strcmp(argv[1], "alert")) msg = 1;
  if (!strcmp(argv[1], "confirm")) msg = 2;
  if (!strcmp(argv[1], "choice")) msg = 3;

  if (msg >=0 && msg < 3) {
    if (argc > 5) {
      sprintf(interp -> result, "feedback: %s: takes only up to three lines",
	      msg == 0 ? "message" : 
	      msg == 1 ? "alert" :
	      "confirm"
	      );
      return TCL_ERROR;
    }
    char *s[3];
    if (argc < 5) s[2] = ""; else s[2] = argv[4];
    if (argc < 4) s[1] = ""; else s[1] = argv[3];
    if (argc < 3) s[0] = ""; else s[0] = argv[2];
    int ret_val = 0;
    switch (msg) {
    case 0: 
      if (use_text) text_fl_show_message(s[0], s[1], s[2]);
      else fl_show_message(s[0], s[1], s[2]); 
      break;
    case 1: 
      if (use_text) text_fl_show_alert(s[0], s[1], s[2]);
      else fl_show_alert(s[0], s[1], s[2], 1);      // '1' centers form
      break;
    case 2: 
      if (use_text) ret_val = text_fl_show_question(s[0], s[1], s[2]);
      else ret_val = fl_show_question(s[0], s[1], s[2]);
      break;
    }
    interp -> result = (ret_val == 0 ? "0" : "1");
    return TCL_OK;
  }

  //  do "choice" here -- must have three text statements, though
  // number of buttons is optional
  if (msg == 3) {
    if (argc < 6) {
      interp -> result = "feedback: choice: must have 3 text lines "
	"and at least one button";
      return TCL_ERROR;
    }
    if (argc > 8) {
      interp -> result = "feedback: choice: must have 3 text lines "
	"and no more than 3 buttons";
      return TCL_ERROR;
    }
    int ret_val;
    if (use_text) ret_val = text_fl_show_choice(argv[2], argv[3], argv[4],
				 argc - 5, argv[5], argv[6], argv[7]) - 1;
    else  ret_val = fl_show_choice(argv[2], argv[3], argv[4],
				   argc - 5, argv[5], argv[6], argv[7]) - 1;
    sprintf(interp -> result, "%d", ret_val);
    return TCL_OK;
  }

  // get a line of input
  if (!strcmp(argv[1], "input")) {
    if (argc < 2 || argc > 4) {
      interp -> result = "feedback: input: ?description? "
	"?default value?";
      return TCL_ERROR;
    }
    const char *s1;
    char *s[2];
    if (argc < 4) s[1] = ""; else s[1] = argv[3];
    if (argc < 3) s[0] = ""; else s[0] = argv[2];
    if (use_text) s1 = text_fl_show_input(s[0], s[1]);
    else s1 = fl_show_input(s[0], s[1]);

    Tcl_AppendElement(interp, (char *) s1);
    return TCL_OK;
  }

  // get a file name
  if (!strcmp(argv[1], "filename")) {
    if (argc > 6) {
      interp -> result = "feedback: filename: ?message? ?directory? "
	"?pattern? ?default?";
      return TCL_ERROR;
    }
    char *msg, *dir, *pat, *def;
    if (argc < 6) def = ""; else def = argv[5];
    if (argc < 5) pat = ""; else pat = argv[4];
    if (argc < 4) dir = ""; else dir = argv[3];
    if (argc < 3) msg = "Enter file name "; else msg = argv[2];
    const char *s1;
    if (use_text) s1 = text_fl_show_fselector(msg, dir, pat, def);
    else s1 = fl_show_fselector(msg, dir, pat, def);
    if (s1 == NULL) {
      interp -> result = "(no file)";        // why not?
      return TCL_ERROR;
    }
    Tcl_AppendElement(interp, (char *) s1);
    return TCL_OK;
  }    

  Tcl_AppendResult(interp, "feedback: don't understand parameter: '",
		   argv[1], "'", NULL);
  return TCL_ERROR;
}


/////////////////////////// Electron Density Maps
#include "ReadEDM.h"
static Tcl_HashTable internal_tcl_edm_hash;
static int internal_tcl_edm_init = 0;
void remove_tcl_edm(ClientData my_data);
int access_tcl_edm(ClientData my_data, Tcl_Interp *interp,
		   int argc, char *argv[]);

// read a file and create a local variable
/// options are:  load "filename"
//                list
int make_tcl_edm(ClientData, Tcl_Interp *interp, int argc, char *argv[])
{
  static unsigned int internal_tcl_edm_count = 0;
  if (argc == 2 && !strcmp(argv[1], "list")) {
    if (!internal_tcl_edm_init) {  // make sure there is a hash table
      interp -> result = "";
      return TCL_OK;
    }
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;
    for (entryPtr = Tcl_FirstHashEntry(&internal_tcl_edm_hash, &search);
	 entryPtr != NULL;
	 entryPtr = Tcl_NextHashEntry(&search)) {
      Tcl_AppendElement(interp, Tcl_GetHashKey(&internal_tcl_edm_hash,
					       entryPtr));
    }
    return TCL_OK;
  }
  if (argc != 3 || strcmp(argv[1], "load")) {
    interp->result =
  "edm: incorrect usage:  should be \"edm list\" or \"edm load <filename>\"";
    return TCL_ERROR;
  }
  // read the file
  char *filename = argv[2];
  edm_t *new_edm = read_edm(filename);
  if (!new_edm) {
    Tcl_AppendResult(interp, "edm: couldn't read file ", filename, NULL);
    return TCL_ERROR;
  }
  
  // if it exists, create the Tcl command and save the hash
  if (internal_tcl_edm_init == 0) {
    internal_tcl_edm_init = 1;
    Tcl_InitHashTable(&internal_tcl_edm_hash, TCL_STRING_KEYS);
  }
  // make the name
  char newname[30];
  sprintf(newname, "vmd_edm_data%d", internal_tcl_edm_count++);

  {  // the %d should make this unique
    int is_new;
    Tcl_HashEntry *entryPtr = Tcl_CreateHashEntry(&internal_tcl_edm_hash,
						  newname, &is_new);
    Tcl_SetHashValue(entryPtr, new_edm);
  }
  // make the new proc
  Tcl_CreateCommand(interp, newname, access_tcl_edm,
		    (ClientData) new_edm, 
		    (Tcl_CmdDeleteProc *) remove_tcl_edm);

  // localize the context
  Tcl_VarEval(interp, "upproc 0 ", newname, NULL);
  // and return the new function name
  Tcl_AppendElement(interp, newname);
  return TCL_OK;
}

// given a grid, find the min/max values
static void edm_find_min_max(Grid<float> *grid, float &mn, float &mx)
{
  float c;
  int nx, ny, nz;
  grid->numrange(&nx, &ny, &nz);
  mn = mx = grid -> item(0,0,0);
  for (int i=0; i<nx; i++) {
    for (int j=0; j<ny; j++) {
      for (int k=0; k<nz; k++) {
	c = grid -> item(i,j,k);
	if (c < mn) mn = c; else if (c > mx) mx = c;
      }
    }
  }
}

// manipulate the data
int access_tcl_edm(ClientData my_data, Tcl_Interp *interp,
		   int argc, char *argv[])
{
  // make sure it exists
  edm_t *my_edm = (edm_t *) my_data;
  if (my_edm == NULL) {
    interp -> result = "edm without data!";
    return TCL_ERROR;
  }
  // the commands are "info", "data {x y z}", and "delete"
  if (argc == 2) {
    if (!strcmp(argv[1], "info")) {
      char s[3*TCL_DOUBLE_SPACE + 3];
      s[0] = 0;
      tcl_append_double(s, my_edm -> minx, 0);
      tcl_append_double(s, my_edm -> miny, 1);
      tcl_append_double(s, my_edm -> minz, 1);
      Tcl_AppendElement(interp, s);
      s[0] = 0;
      tcl_append_double(s, my_edm -> maxx, 0);
      tcl_append_double(s, my_edm -> maxy, 1);
      tcl_append_double(s, my_edm -> maxz, 1);
      Tcl_AppendElement(interp, s);
      int nx, ny, nz;
      my_edm -> grid -> numrange(&nx, &ny, &nz);
      sprintf(s, "%d %d %d", nx, ny, nz);
      Tcl_AppendElement(interp, s);
      s[0] = 0;
      tcl_append_double(s, my_edm -> alpha, 0);
      tcl_append_double(s, my_edm -> beta, 1);
      tcl_append_double(s, my_edm -> gamma, 1);
      Tcl_AppendElement(interp, s);

      // find the min/max value
      float a, b;
      edm_find_min_max(my_edm -> grid, a, b);
      s[0] = 0;
      tcl_append_double(s, a, 0);
      tcl_append_double(s, b, 1);
      Tcl_AppendElement(interp, s);
	    
      return TCL_OK;
    }
    if (!strcmp(argv[1], "delete")) {
      return Tcl_VarEval(interp, "rename ", argv[0], " {}", NULL);
    }
    Tcl_AppendResult(interp, "edm: don't understand the parameter '", 
		     argv[1], "'", NULL);
    return TCL_ERROR;
  } // argc == 2

  // only thing allowed here are "data {x y z}"
  // and "write_isobuild <filename>"
  if (argc < 2) {
    interp->result = "edm: need parameters: \"info\", \"delete\", "
      "\"data {i j k}\", or \"write_isobuild\"";
    return TCL_ERROR;
  }
  if (argc !=3 || (strcmp(argv[1], "data") && 
		   strcmp(argv[1], "write_isobuild"))) {
    Tcl_AppendResult(interp, "edm: don't understand the parameter '", 
		     argv[1], "'", NULL);
    return TCL_ERROR;
  }

  if (!strcmp(argv[1], "write_isobuild")) {
    // get the output filename
    char *filename = argv[2];
    // open the file for output
    FILE *outfile = fopen(filename, "w");
    if (!outfile) {
      Tcl_AppendResult(interp, "edm: cannot open file for output '",
		       filename, "': ", strerror(errno), NULL);
      return TCL_ERROR;
    }
    // write the data in 'float' format
    int nx, ny, nz;
    my_edm -> grid -> numrange(&nx, &ny, &nz);

    // Print the header info: 3 == float
    fprintf(outfile, "%d %d %d %d\n", 3, nx, ny, nz);
    // and go through the rest of the data
    {
      float f;
      for (int k=0; k<nz; k++) {
	for (int i=0; i<nx; i++) {
	  for (int j=0; j<ny; j++) {
	    f = my_edm->grid->item(i,j,k);
	    fwrite(&f, sizeof(float), 1, outfile);
	  }
	}
      }
    }
    // that should be it, so close the file and report success
    fclose(outfile);
    return TCL_OK;
  }
  //// get the data
  {
    // get the i, j, k value
    char **list;
    int num;
    if (Tcl_SplitList(interp, argv[2], &num, &list) != TCL_OK) {
      return TCL_ERROR;
    }
    // there should be three elements
    if (num != 3) {
      interp -> result = 
	"edm: there should be three elements for the data coordinates";
      free(list);
      return TCL_ERROR;
    }
    // convert them to values
    int i, j, k;
    if (Tcl_GetInt(interp, list[0], &i) != TCL_OK ||
	Tcl_GetInt(interp, list[1], &j) != TCL_OK ||
	Tcl_GetInt(interp, list[2], &k) != TCL_OK) {
      free(list);
      return TCL_ERROR;
    }
    // check the range
    int nx, ny, nz;
    my_edm -> grid -> numrange(&nx, &ny, &nz);
    if (i < 0 || j < 0 || k < 0 ||
	i >= nx || j >= ny || k >= nz) {
      interp -> result = "edm: value out of range";
      return TCL_ERROR;
    }
    // Now everything is okay
    char s[TCL_DOUBLE_SPACE];
    Tcl_PrintDouble(interp, my_edm -> grid -> item(i, j, k), s);
    Tcl_AppendElement(interp, s);
    return TCL_OK;
  } // got the data
}

void remove_tcl_edm(ClientData my_data) {
  edm_t *my_edm = (edm_t *) my_data;
  if (!my_edm) {return ;}
  // find it in the hash table
    Tcl_HashSearch searchPtr;
    Tcl_HashEntry *hash;
    for (hash = Tcl_FirstHashEntry(&internal_tcl_edm_hash, &searchPtr);
	 hash != NULL; hash = Tcl_NextHashEntry(&searchPtr)) {
	   if (Tcl_GetHashValue(hash) == my_edm) {
	     break;
	   }
    }
    if (hash == NULL) {
      msgErr << "edm: deletion cannot find data in table!" << sendmsg;
      return;
    }
    delete my_edm;
    Tcl_DeleteHashEntry(hash);
}


