/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Timer.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 1997/03/13 17:51:33 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *      Timer.C constains the implementation of the class Timer which
 * provides a generic way to time things.  Each Timer keeps a cumulative
 * time.  It can be started, stopped, or cleared.
 *      On the SGIs, the clock time is obtained by calls to the routine
 * gettimeofday() and the user, system, and total cpu times are obtained
 * using the routine times().  This code was taken from namd and was
 * originally written by Mark Nelson.
 *
 ***************************************************************************/

#include <unistd.h>
#include <sys/types.h>
#include <time.h>
#include <sys/times.h>
#include <sys/time.h>
#include "Inform.h"
#include "Timer.h"

//  TIMERON and TIMEROFF define the current state of a Timer
#define TIMEROFF	0
#define TIMERON		1

/************************************************************************/
/*									*/
/*			FUNCTION Timer					*/
/*									*/
/*	This is the constructor for the class Timer.  It sets the timer */
/*  status to TIMEROFF and clears all the values.  It also makes a call */
/*  to sysconf to determine the number of clock ticks per second for    */
/*  use with the call times()						*/
/*									*/
/************************************************************************/

Timer::Timer()

{
	cpu_speed=sysconf(_SC_CLK_TCK);
	timer_state=TIMEROFF;
	clear();
}
/*			END OF FUNCTION Timer				*/

/************************************************************************/
/*									*/
/*			FUNCTION clear					*/
/*									*/
/*	clear sets all of the accumulated times for this timer to 0.	*/
/*  It is intended to only be used on a stopped timer.  If it is used	*/
/*  on a running timer, a warning message is printed, the timer is      */
/*  stopped and all of its values are cleared.				*/
/*									*/
/************************************************************************/

void Timer::clear()

{
	// Check to see if the timer if on
	if (timer_state == TIMERON)
	{
		//  Clearing a running Timer
		msgWarn << "TRIED TO CLEAR A RUNNING TIMER! "
		         << "SETTING ALL VALUES TO 0 AND STOPPING TIMER"
			 << sendmsg;

		timer_state=TIMEROFF;
	}
	//  Set all of the accumulated values to 0
	current_secs=0;
	current_usecs=0;
	current_user_time=0;
	current_system_time=0;
}
/*			END OF FUNCTION clear				*/

/************************************************************************/
/*									*/
/*			FUNCTION start					*/
/*									*/
/*	start a Timer timing.  This will start adding time elapsed to   */
/*  the current accumulated values of the timer.  If you try to start   */
/*  a timer that is already running, a warning message is printed	*/
/*									*/
/************************************************************************/

void Timer::start()

{
	struct tms tmsbuf;	//  Values from call to times
	struct timeval tvbuf;	//  Values from call to gettimeofday
	struct timezone tzbuf;  //  Timezone values from gettimeofday
				//  These values aren't used for anything

	//  Check to see if the timer is already running
	if (timer_state==TIMERON)
	{
		msgWarn << "TRIED TO START A RUNNING TIMER! "
		         << "CONTINUING UNCHANGED"
			 << sendmsg;

		return;
	}

	//  Get the current time values from the system
	gettimeofday(&tvbuf, &tzbuf);
	times(&tmsbuf);

	//  Set the state of the Timer
	timer_state=TIMERON;

	//  Set the starting values to the current time
	last_secs=tvbuf.tv_sec;
	last_usecs=tvbuf.tv_usec;
	last_user_time=tmsbuf.tms_utime;
	last_system_time=tmsbuf.tms_stime;
}
/*			END OF FUNCTION start				*/

/************************************************************************/
/*									*/
/*				FUNCITON stop				*/
/*									*/
/*	stop stops a Timer from accumulating time.  If you try and stop */
/*  a stopped Timer, a warning message is printed			*/
/*									*/
/************************************************************************/

void Timer::stop()

{
	struct tms tmsbuf;
	struct timeval tvbuf;
	struct timezone tzbuf;

	if (timer_state==TIMEROFF)
	{
	  //		msgWarn << "TRIED TO STOP A STOPPED TIMER! "
	  //	         << "CONTINUING UNCHANGED"
	  //			 << sendmsg;

		return;
	}

	gettimeofday(&tvbuf, &tzbuf);
	times(&tmsbuf);

	timer_state=TIMEROFF;
	current_secs+=tvbuf.tv_sec-last_secs;
	current_usecs+=tvbuf.tv_usec-last_usecs;
	current_user_time+=tmsbuf.tms_utime-last_user_time;
	current_system_time+=tmsbuf.tms_stime-last_system_time;
}
/*			END OF FUNCTION stop				*/

/************************************************************************/
/*									*/
/*				FUNCITON reset				*/
/*									*/
/*	reset resets a Timer by 1) clearing it if it is off, or 2)      */
/*  stopping, clearing, and restarting it if it is on                   */
/*									*/
/************************************************************************/

void Timer::reset()

{
	if (timer_state == TIMEROFF)
	{
		clear();
	}
	else
	{
		stop();
		clear();
		start();
	}
}
/*			END OF FUNCTION reset				*/

/************************************************************************/
/*									*/
/*			FUNCTION clock_time				*/
/*									*/
/*	clock_time returns the current amount of real (clock) time	*/
/*  accumulated by this timer.  If the timer is stopped, this is just	*/
/*  the total accumulated time.  If the timer is running, this is the	*/
/*  accumulated time + the time since the timer was last started.	*/
/*									*/
/************************************************************************/

float Timer::clock_time()

{
	struct timeval tvbuf;	// Values from gettimeofday
	struct timezone tzbuf;	// Timezone values from gettimeofday
				// these values aren't used for anything
	long seconds;		// seconds elapsed
	long useconds;		// useconds (mirco-seconds) elapsed
	float ret_val;		// number of seconds elpased 

	if (timer_state==TIMEROFF)
	{
		//  Timer is currently off, so just return
		//  accumulated time
		seconds=current_secs;
		useconds=current_usecs;
	}
	else
	{
		//  Timer is currently running, so add the elapsed
		//  time since the timer was last started to the
		//  accumulated time
		gettimeofday(&tvbuf, &tzbuf);

	        seconds = current_secs+tvbuf.tv_sec-last_secs;
	        useconds = current_usecs+tvbuf.tv_usec-last_usecs;
	}

	//  Adjust for the fact that the useconds may be negative.
	//  If they are, take away 1 second and add 1 million
	//  microseconds until they are positive
	while (useconds < 0)
	{
		useconds = useconds + 1000000;
		seconds = seconds - 1;
	}

	//  Convert into floating point number of seconds
	ret_val = (float) seconds + (float) useconds/1000000.0;

	return(ret_val);
}
/*			END OF FUNCTION clock_time			*/

/************************************************************************/
/*									*/
/*			FUNCTION user_time				*/
/*									*/
/*	user_time reports the current amount of user cpu time           */
/*   accumulated by this Timer.  If the timer is currently off, 	*/
/*   this is just the accumulated time.  If the Timer is running, this  */
/*   is the accumulated time plust the time since the timer was last    */
/*   started.								*/
/*									*/
/************************************************************************/

float Timer::user_time()

{
	float ret_val;		//  Return value	
	struct tms tmsbuf;	//  Structure for times call

	if (timer_state==TIMEROFF)
	{
		//  Timer is off, just return accumulated time
		ret_val = current_user_time;
	}
	else
	{
		//  Timer is on, add current running time to
		//  accumulated time
		times(&tmsbuf);

		ret_val = current_user_time+tmsbuf.tms_utime-last_user_time;
	}

	//  Convert from clock ticks to seconds using the 
	//  cpu_speed value obtained by the constructor
	ret_val = ret_val/cpu_speed;

	return(ret_val);
}
/*			END OF FUNCTION user_time			*/

/************************************************************************/
/*									*/
/*			FUNCTION system_time				*/
/*									*/
/*	system_time reports the current amount of system cpu time       */
/*   accumulated by this Timer.  If the timer is currently off, 	*/
/*   this is just the accumulated time.  If the Timer is running, this  */
/*   is the accumulated time plust the time since the timer was last    */
/*   started.								*/
/*									*/
/************************************************************************/

float Timer::system_time()

{
	float ret_val;		//  Return value
	struct tms tmsbuf;	//  Structure for times() call

	if (timer_state==TIMEROFF)
	{
		//  Timer is off, just return accumulated time
		ret_val = current_system_time;
	}
	else
	{
		//  Timer is on, return accumulated plus current
		times(&tmsbuf);

		ret_val = current_system_time+tmsbuf.tms_stime-last_system_time;
	}

	//  Convert from clock ticks to seconds using the 
	//  cpu_speed value obtained by the constructor
	ret_val = ret_val/cpu_speed;

	return(ret_val);
}
/*			END OF FUNCTION system_time			*/

