/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Timestep.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.11 $	$Date: 1996/10/10 00:17:38 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Timestep class, which stores coordinates, energies, etc. for a
 * single timestep.
 *
 * Note: As more data is stored for each step, it should go in here.  For
 * example, H-Bonds could be calculated each step.
 ***************************************************************************/

#include <math.h>
#include "Timestep.h"
#include "Inform.h"
#include "Atom.h"

// There may be a bug in the code that maintains the "extra"
// information.  Since that isn't (yet) used in VMD, make sure
// the data exists and is never deleted.
static float *timestep_ignore_dynamics = NULL;
static int timestep_maxn = 0;

///////////////////////////  constructor  

Timestep::Timestep(int n, float DT, float *) {

  MSGDEBUG(2,"Creating new timestep with " << n << " atoms, dt=" << DT);
  MSGDEBUG(2,sendmsg);

  for(int i=0; i < TSENERGIES; energy[i++] = 0.0);
  num = n;
  dt = DT;
  pos = new float[ATOMCOORDS * num];

  // special code to check the strange case
  if (timestep_maxn < num) {
    timestep_maxn = n;
    if (timestep_ignore_dynamics) {
      delete [] timestep_ignore_dynamics;
    }
    timestep_ignore_dynamics = NULL;
  }
  if (!timestep_ignore_dynamics) {
    timestep_ignore_dynamics = new float[ATOMEXTRA_DYNAMIC * num];
  }
  data = timestep_ignore_dynamics;
  needDataDelete = FALSE;


  scale_factor = 1.0;
  numPatches = 0;
  patchData = NULL;

  Initialized = FALSE;
}


// copy constructor

Timestep::Timestep(const Timestep& ts) {
  num = ts.num;
  pos = new float[ATOMCOORDS * num];
  memcpy(pos, ts.pos, ATOMCOORDS * num * sizeof(float));

  data = timestep_ignore_dynamics;
  needDataDelete = FALSE;

  memcpy(energy, ts.energy, sizeof(ts.energy));

  numPatches = ts.numPatches;
  if (numPatches > 0) {
    patchData = new float[ numPatches * TSPATCHDATA ];
    memcpy(patchData, ts.patchData, numPatches * TSPATCHDATA *
	   sizeof(float));
  } else {
    patchData = NULL;
  }

  memcpy(minpos, ts.minpos, sizeof(minpos));
  memcpy(maxpos, ts.maxpos, sizeof(maxpos));
  memcpy(mindata, ts.mindata, sizeof(mindata));
  memcpy(maxdata, ts.maxdata, sizeof(maxdata));
  memcpy(minpatch, ts.minpatch, sizeof(minpatch));
  memcpy(maxpatch, ts.maxpatch, sizeof(maxpatch));
  maxrad = ts.maxrad;
  memcpy(COV, ts.COV, sizeof(COV));
  scale_factor = ts.scale_factor;
  dt = ts.dt;
  Initialized = ts.Initialized;

}

///////////////////////////  destructor  

Timestep::~Timestep(void) {
  if(needDataDelete && data)
    delete [] data;

  if (pos)
    delete [] pos;
    
  if(patchData)
    delete [] patchData;
}

//////////////////////////  public routines  

// create storage to store n patches; returns pointer to beginning
// of data block
float *Timestep::create_patch_storage(int n) {

  // delete old storage, if necessary
  if(patchData) {
    // see if we already have the proper size for patches
    if(numPatches == n)
      return patchData;

    // number differs; remove old storage
    delete [] patchData;
    patchData = NULL;
    numPatches = 0;
  }
  
  // create new storage, if n is legal
  if(n > 0) {
    patchData = new float[n * TSPATCHDATA];
    numPatches = n;
  }
  
  return patchData;
}


// calculate the max/min values for all the quantities, and anything else
// required based on atom coordinates, etc.  Used for scaling
// and translating purposes.
void Timestep::init(void) {
  int i, j;
  float *mpos = pos, *mdata = data, *mpatch = patchData;

  // only do this if there are atoms
  if(!num)
    return;		

  // inialize min/max positions
  for(i=0; i < ATOMCOORDS; i++) {
    minpos[i] = maxpos[i] = pos[i];		// take values for first atom
    COV[i] = 0.0;
  }  
  for(i=0; i < ATOMEXTRA_DYNAMIC; i++) {
    mindata[i] = maxdata[i] = data[i];		// take values for first atom
  }

  // calc ranges for all atoms
  for(i=0; i < num; i++) {
  
    for(j=0; j < ATOMCOORDS; j++) {
      COV[j] += mpos[j];
      if(mpos[j] < minpos[j])
        minpos[j] = mpos[j];
      else if(mpos[j] > maxpos[j])
        maxpos[j] = mpos[j];
    }

    for(j=0; j < ATOMEXTRA_DYNAMIC; j++) {
      if(mdata[j] < mindata[j])
        mindata[j] = mdata[j];
      else if(mdata[j] > maxdata[j])
        maxdata[j] = mdata[j];
    }

    mpos += ATOMCOORDS;
    mdata += ATOMEXTRA_DYNAMIC;
  }
  
  // calculate ranges for patch data
  if(numPatches > 0 && mpatch) {
  
    // initialize min and max values
    for(i=0; i < TSPATCHDATA; i++)
      minpatch[i] = maxpatch[i] = mpatch[i];
      
    // calc ranges
    for(i=0; i < numPatches; i++) {

      for(j=0; j < TSPATCHDATA; j++) {
        if(mpatch[j] < minpatch[j])
	  minpatch[j] = mpatch[j];
	else if(mpatch[j] > maxpatch[j])
	  maxpatch[j] = mpatch[j];
      }
      
      mpatch += TSPATCHDATA;
    }
  }

  // calculate center-of-volume and scale factor
  scale_factor = maxpos[0] - minpos[0];
  for(i=0; i < ATOMCOORDS; i++) {
    COV[i] /= (float)num;
    if((maxpos[i] - minpos[i]) > scale_factor)
      scale_factor = maxpos[i] - minpos[i];
  }
  scale_factor = 1.5 / scale_factor;

  // calculate enclosing sphere radius
  maxrad = 0.0;
  float newrad = 0.0;
  mpos = pos;
  for(i=0; i < num; i++) {
    float x = *(mpos++) - COV[0];
    float y = *(mpos++) - COV[1];
    float z = *(mpos++) - COV[2];
    newrad = sqrtf(x*x + y*y + z*z);
    if(newrad > maxrad)
      maxrad = newrad;
  }

  MSGDEBUG(3,"Timestep ranges calculated: for " << num << " atoms:"<< sendmsg);
  MSGDEBUG(3,"  Item       Min       Max" << sendmsg);
  MSGDEBUG(3,"--------- --------- -----------" << sendmsg);
  for(i=0; i < ATOMCOORDS; i++) {
    MSGDEBUG(3,"  pos[" << i << "]     " << minpos[i] << "\t " << maxpos[i]);
    MSGDEBUG(3,sendmsg);
  }
  for(i=0; i < ATOMEXTRA_DYNAMIC; i++) {
    MSGDEBUG(3," data[" << i << "]     " << mindata[i] << "\t " << maxdata[i]);
    MSGDEBUG(3,sendmsg);
  }
  if(numPatches > 0) {
    for(i=0; i < TSPATCHDATA; i++) {
      MSGDEBUG(3,"  patch[" << i << "]   " << minpatch[i] << "\t ");
      MSGDEBUG(3,maxpatch[i] << sendmsg);
    }
  }
  
  Initialized = TRUE;
}

// reset coords and related items to 0
void Timestep::zero_values(void)
{
  if (num <= 0) return;
  
  int i;
  for (i=0; i<3*num; i++) {
    pos[i] = 0.0;
  }
  for(i=0; i < TSENERGIES; energy[i++] = 0.0);
  scale_factor = 1.0;
  init();
}

