/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Timestep.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1997/03/23 10:23:12 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Timestep class, which stores coordinates, energies, etc. for a
 * single timestep.
 *
 * Note: As more data is stored for each step, it should go in here.  For
 * example, H-Bonds could be calculated each step.
 ***************************************************************************/
#ifndef TIMESTEP_H
#define TIMESTEP_H

#include "Atom.h"

//
// useful defines
//

// energy terms and temperature stored for each timestep

// TSENERGIES must be the last element.  It indicates the number
// energies.  (TSE_TOTAL is the total energy).  If you add fields here
// you should also add the lines in MolInfo.C so you can get access to
// the fields from Tcl.
enum { TSE_BOND, TSE_ANGLE, TSE_DIHE, TSE_IMPR, TSE_VDW, TSE_COUL,
       TSE_HBOND, TSE_KE, TSE_PE, TSE_TEMP, TSE_TOTAL, TSE_VOLUME,
       TSE_PRESSURE, TSE_EFIELD, TSENERGIES};

// patch data values stored; again, TSPATCHDATA must be the last item
// This information is not (yet?) available to Tcl.
enum {TSP_X, TSP_Y, TSP_Z, TSP_XLEN, TSP_YLEN, TSP_ZLEN, TSP_LOAD,
      TSP_ATOMS, TSP_NODE, TSPATCHDATA};


class Timestep {

private:
  // do we need to deallocate the space for the 'data' array?
  int needDataDelete;

// while this is a class, almost all the members are public
public:
  // number of atoms this timestep is for
  int num;
  
  // coordinates of all the atoms, as (x,y,z), (x,y,z), ...
  float *pos;
  
  // extra data for each atom: temperature and B-factors (this may just
  // be a pointer from some other object)
  float *data;
  
  // energy for this step; by default, all 0
  float energy[TSENERGIES];
  
  // number of patches for this timestep; if zero, none or not relevant
  int numPatches;
  
  // data for each patch: position of corner, size, and other values
  float *patchData;

  // minimum and maximum values for position and data and patch data
  float minpos[ATOMCOORDS];
  float maxpos[ATOMCOORDS];
  float mindata[ATOMEXTRA_DYNAMIC];
  float maxdata[ATOMEXTRA_DYNAMIC];
  float minpatch[TSPATCHDATA];
  float maxpatch[TSPATCHDATA];
  float maxrad;			// calculated relative to COV

  // center of volume position
  float COV[ATOMCOORDS];

  // scale factor to get the coordinate set to fix in (-1 ... 1) box
  float scale_factor;

  // time required to compute this timestep .... if unknown, should be 0
  float dt;

  // has this timestep been initialized?
  int Initialized;

public:
  // constructor: # atoms, time used to compute this timestep,
  // and 'data' items; if latter is null, space is allocated.
  Timestep(int n, float DT, float *newdata = NULL);

  // copy constructor
  Timestep(const Timestep& ts);
  
  // destructor
  ~Timestep(void);
  
  // create storage to store n patches; returns pointer to beginning
  // of data block
  float *create_patch_storage(int);

  // calculate the max/min values for all the quantities, and anything else
  // required based on atom coordinates, etc.  Used for scaling
  // and translating purposes.
  void init(void);

  // set the coords to 0, then re-init
  void zero_values(void);
};

#endif

