/***************************************************************************
 *cr
 *cr		(C) Copyright 1995 The Board of Trustees of the
 *cr			    University of Illinois
 *cr			     All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: TokenDisplayDevice.C,v $
 *	$Author: dalke $		$Locker:  $		   $State: Exp $
 *	$Revision: 1.6 $	      $Date: 1997/03/19 04:06:07 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * FileRenderer type for listing all tokens in the display list
 *
 ***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "TokenDisplayDevice.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"

// Utility function to encode a point into a string for use in printf()s
// Up to 8 points may be used in an expression before overwriting
//  previously generated strings.
static const char *pt(float *p) {
    static const int nbuf = 8;
    static char buf[nbuf][50];
    static int bufptr = 0;

    bufptr = (bufptr + 1) % nbuf;   // use next buffer in round-robin fashion
    sprintf(buf[bufptr], "%g %g %g", p[0], p[1], p[2]);
    return buf[bufptr];
}

///////////////////////// constructor 

// constructor ... call the parent with the right values
TokenDisplayDevice::TokenDisplayDevice(void) : 
  FileRenderer("Token", "token.txt", "true") {
}

///////////////////////// protected nonvirtual routines

// draw a point
void TokenDisplayDevice::point(float *p) {
    fprintf(outfile, "%-16s%s\n", "point", pt(p));
}

// draw a sphere
void TokenDisplayDevice::sphere(float *xyzr) {
    fprintf(outfile, "%-16s%s\t%g\n", "sphere", pt(xyzr), xyzr[3]);
}

// draw a line (cylinder) from a to b
void TokenDisplayDevice::line(float *a, float*b) {
    fprintf(outfile, "%-16s%s\t%s\n", "line", pt(a), pt(b));
}

// draw a cylinder (lists endpoints, resolution, and radius)
void TokenDisplayDevice::cylinder(float *a, float *b, float radius,
				  int filled) {
  fprintf(outfile, "%-16s%s\t%s\t%g\t%d\n", "cylinder", 
	  pt(a), pt(b), radius, filled);
}

#ifdef QWEQWE
// draw a precomputed cylinder (lists edges normals & vertices for each edge)
void TokenDisplayDevice::prcylinder(const char *tok, int num, float *edges) {
    fprintf(outfile, "%-16s%d\n", tok, num);
    for (int i = 0; i < num; i++) {
	fprintf(outfile, "\t\t%s\t%s\t%s\n", pt(edges), pt(edges+3), pt(edges+6));
	edges += 9;
    }
}
#endif // QWEQWE

// draw a cone
void TokenDisplayDevice::cone(float *a, float *b, float radius) {
    fprintf(outfile, "%-16s%s\t%s\t%g\n", "cone", pt(a), pt(b), radius);
}

// draw a triangle
void TokenDisplayDevice::triangle(float *a, float *b, float *c,
				  float *n1, float *n2, float *n3) {
    fprintf(outfile, "%-16s%s\t%s\n", "triangle", pt(a), pt(n1));
    fprintf(outfile, "\t\t%s\t%s\n", pt(b), pt(n2));
    fprintf(outfile, "\t\t%s\t%s\n", pt(c), pt(n3));
}

// draw a square
void TokenDisplayDevice::square(float *norm, float *a, float *b, 
				float *c, float *d) {
    fprintf(outfile, "%-16s%s\n", "square", pt(norm));
    fprintf(outfile, "\t\t%s\t%s\n", pt(a), pt(b));
    fprintf(outfile, "\t\t%s\t%s\n", pt(c), pt(d));
}

// other object information
void TokenDisplayDevice::set_line_width(int new_width)
{
  fprintf(outfile, "linewidth\t%d\n", new_width);
}
void TokenDisplayDevice::set_line_style(int new_style)
{
  fprintf(outfile, "linestyle\t%d\n", new_style);
}
void TokenDisplayDevice::set_sphere_res(int new_res)
{
  fprintf(outfile, "sphereres\t%d\n", new_res);
}
void TokenDisplayDevice::set_sphere_style(int new_style)
{
  fprintf(outfile, "spheretype\t%d\n", new_style);
}




// matrix stack operation
void TokenDisplayDevice::prmatop(const char *tok, const Matrix4& mat) {
    fprintf(outfile, "%-16s\n", tok);
    for (int row = 0; row < 4; row++) {
	fprintf(outfile, "\t\t%g %g %g %g\n",
		mat.mat[row][0], mat.mat[row][1], 
		mat.mat[row][2], mat.mat[row][3]);
    }
}

void TokenDisplayDevice::push(void) {
  fprintf(outfile, "push\n");
}
void TokenDisplayDevice::pop(void) {
  fprintf(outfile, "pop\n");
}
void TokenDisplayDevice::load(Matrix4& mat) {
  prmatop("loadmat", mat);
}
void TokenDisplayDevice::multmatrix(Matrix4& mat) {
  prmatop("multmat", mat);
}
void TokenDisplayDevice::translate(float x, float y, float z) {
  float tmp[3];
  tmp[0] = x; tmp[1] = y; tmp[2] = z;
  fprintf(outfile, "trans\t\t%s\n", pt(tmp));
}
void TokenDisplayDevice::rot(float ang, char axis) {
  fprintf(outfile, "rot\t\t%g %c\n", ang, axis);
}
void TokenDisplayDevice::scale(float x, float y, float z) {
  float tmp[3];
  tmp[0] = x; tmp[1] = y; tmp[2] = z;
  fprintf(outfile, "scale\t\t%s\n", pt(tmp));
}

// set the color index
void TokenDisplayDevice::set_color(int index)
{
  fprintf(outfile, "colorindex\t%d\n", index);
}

// materials
void TokenDisplayDevice::activate_materials(void)
{
  fprintf(outfile, "materials\t%d\n", 1);
}
void TokenDisplayDevice::deactivate_materials(void)
{
  fprintf(outfile, "materials\t%d\n", 0);
}


// deal with text
void TokenDisplayDevice::text_position(float x, float y, float z)
{
  float tmp[3];
  tmp[0] = x; tmp[1] = y; tmp[2] = z;
  fprintf(outfile, "textpos\t\t%s\n", pt(tmp));
}

void TokenDisplayDevice::text(char *s)
{
  fprintf(outfile, "text\t\t%s\n", s);
}

// deal with comment
void TokenDisplayDevice::comment(char *s)
{
  fprintf (outfile, "comment\t\t%s\n", s);
}


// picking objects
void TokenDisplayDevice::pick_point(float *xyz, int id)
{
  fprintf(outfile, "pickpoint\t%s %d\n", pt(xyz), id);
}

///////////////////// public virtual routines

// initialize the file for output
void TokenDisplayDevice::write_header(void) {

  // file for Token raytracer.

  fprintf(outfile, "# Token dump of VMD Scene\n");

  fprintf(outfile, "# View\n");
  fprintf(outfile, "eye %s\n", pt(eyePos));
  fprintf(outfile, "at %s\n", pt(eyeDir));
  fprintf(outfile, "up %s\n", "0 1 0");
  
  fprintf(outfile, "# Light Definitions\n");
  for (int i = 0; i < DISP_LIGHTS; i++) {
    if (lightDefined[i] && lightOn[i]) {
      fprintf(outfile, "light color %s\n", pt(lightColor[i]));
      fprintf(outfile, "light direction %s\n", pt(lightPos[i]));
    }
  }
  
  fprintf(outfile, "background color %s\n", pt(backColor));

  fprintf(outfile, "# Token list\n");
}


// clean up after yourself
void TokenDisplayDevice::write_trailer(void) {
  fprintf(outfile, "# End of tokens\n");
  msgInfo << "Token file generation finished" << sendmsg;
}

#ifdef QWEQWE
void TokenDisplayDevice::render(void *cmdlist) {
    char *cmdptr = (char *)cmdlist;
    int tok, cmdsize;

    MSGDEBUG(3, "TokenDisplayDevice: rendering command list." << sendmsg);

    if (!cmdlist) return;

    // scan through the list, getting each command and executing it, until
    // the end of commands token is found
    dataBlock = NULL;
    while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
	float *fp;
	int *ip, n;

	if (tok == DLINKLIST) {
	    cmdptr += sizeof(int);	 // step forward one (no length here)
	    cmdptr = *((char **) cmdptr);// and go to the next link
	    tok = ((int *)cmdptr)[0];	 // this is guaranteed to be neither
	}				// a DLINKLIST nor DLASTCOMMAND

	cmdsize = ((int *)cmdptr)[1];
	cmdptr += 2*sizeof(int);

	fp = (float *)cmdptr;
	ip = (int *)cmdptr;

	switch (tok) {
	    case DPOINT:
		prpoint("point", fp);
		break;
	    case DPOINT_I:
		fp = dataBlock + ip[0];
		prpoint("point_i", fp);
		break;
	    case DLINE:
		prline("line", fp, fp+3);
		break;
	    case DLINE_I:
		prline("line_i", dataBlock + ip[0], dataBlock + ip[1]);
		break;
	    case DSPHERE:
		prsphere("sphere", fp, fp[3]);
		break;
	    case DSPHERE_I:
		prsphere("sphere_i", dataBlock + int(fp[0]), fp[1]);
		break;
	    case DTRIANGLE:
		prtriangle("triangle", fp, fp+3, fp+6, fp+9, fp+12, fp+15);
		break;
	    case DTRIANGLE_I:
		prtriangle("triangle_i",
			 dataBlock + ip[1],
			 dataBlock + ip[2],
			 dataBlock + ip[3],
			 dataBlock + ip[0],
			 dataBlock + ip[0],
			 dataBlock + ip[0]);
		break;
	    case DSQUARE:
		prsquare("square", fp, fp+3, fp+6, fp+9, fp+12);
		break;
	    case DSQUARE_I:
		prsquare("square_i",
		       dataBlock + ip[0],
		       dataBlock + ip[1],
		       dataBlock + ip[2],
		       dataBlock + ip[3],
		       dataBlock + ip[4]);
		break;
	    case DCYLINDER:
#ifdef USE_SLOW_CYLINDERS
		prcylinder("cylinder", fp, fp+3, int(fp[7]), fp[6]);
#else
		prcylinder("cylinder", int(fp[7]), fp+8);
#endif
		break;
	    case DCYLINDER_I:
		prcylinder("cylinder",
			 dataBlock + int(fp[0]),
			 dataBlock + int(fp[1]),
			 int(fp[3]),
			 fp[2] );
		break;
	    case DCONE:
		// draw a cone of given radius and resolution
		prcone("cone", fp, fp+3, int(fp[7]), fp[6]);
		break;
	    case DCONE_I:
		prcone("cone_i",
		     dataBlock + int(fp[0]),
		     dataBlock + int(fp[1]),
		     int(fp[3]),
		     fp[2] );
		break;
	    case DTEXTPOS:
		fprintf(outfile, "textpos\t\t%s\n", pt(fp));
		break;
	    case DTEXTPOS_I:
		fprintf(outfile, "textpos_i\t%s\n", pt(dataBlock + ip[0]));
		break;
	    case DTEXT:
		fprintf(outfile, "text\t\t%s\n", cmdptr);
		break;
	    case DCOLORINDEX:
		// set the current color to the given color index ... assumes the
		// color has already been defined
		n = ip[0];

		fprintf(outfile, "colorindex\t%d\n", n);
		break;
	    case DCOLORRGB:
		fprintf(outfile, "colorrgb\t%s\n", pt(fp));
		break;
	    case DPICKPOINT:
		fprintf(outfile, "pickpoint\t%s %d\n", pt(fp), ip[3]);
		break;
	    case DPICKPOINT_I:
		fprintf(outfile, "pickpoint_i\t%s %d\n",
		    pt(dataBlock + ip[0]), ip[1]);
		break;
	    case DPICKLINE:
		fprintf(outfile, "pickline\n");
		break;
	    case DPICKLINE_I:
		fprintf(outfile, "pickline_i\n");
		break;
	    case DPICKBOX:
		fprintf(outfile, "pickbox\n");
		break;
	    case DPICKBOX_I:
		fprintf(outfile, "pickbox_i\n");
		break;
	    case DLIGHTONOFF:
		fprintf(outfile, "lightonoff\t%d %d\n", ip[0], ip[1]);
		break;
	    case DMATERIALS:
		fprintf(outfile, "materials\t%d\n", ip[0]);
		break;
	    case DSPHERERES:
		fprintf(outfile, "sphereres\t%d\n", ip[0]);
		break;
	    case DSPHERETYPE:
		fprintf(outfile, "spheretype\t%d\n", ip[0]);
		break;
	    case DLINESTYLE:
		fprintf(outfile, "linestyle\t%d\n", ip[0]);
		break;
	    case DLINEWIDTH:
		fprintf(outfile, "linewidth\t%d\n", ip[0]);
		break;
	    case DPUSH:
		fprintf(outfile, "push\n");
		break;
	    case DPOP:
		fprintf(outfile, "pop\n");
		break;
	    case DLOAD:
		prmatop("loadmat", fp);
		break;
	    case DMULT:
		prmatop("multmat", fp);
		break;
	    case DTRANS:
		fprintf(outfile, "trans\t\t%s\n", pt(fp));
		break;
	    case DSCALE:
		fprintf(outfile, "scale\t\t%s\n", pt(fp));
		break;
	    case DROT:
		fprintf(outfile, "rot\t\t%g %c\n", fp[0], 'x' + int(fp[1]));
		break;
	    case DCOLORDEF:
		// define a new color
		fprintf(outfile, "colordef\t%d\t", int(fp[0]));
		for (n = 0; n < COLOR_ITEMS; n++)
		    fprintf(outfile, "%g ", fp[n+1]);
		putc('\n', outfile);
		break;
	    case DLIGHTDEF:
		fprintf(outfile, "lightdef\t%d\t%s\t%s\n",
		    int(fp[0]), pt(fp+1), pt(fp+4));
		break;
	    case DCLEAR:
		fprintf(outfile, "clear\n");
		break;
	    case DMOREDATA:
	    case DNEWDATA:
		// set the current drawing data block
#ifdef VMDCAVE
		dataBlock = (float *)cmdptr;	// point to current position in list
#else
		dataBlock = ((float **)cmdptr)[0];  // point to given memory loc
#endif
		break;
	    default:
		// command not found, help!
		fprintf(outfile, "unknown token %d\n", tok);
	}

	// update position in array
	cmdptr += cmdsize;
    }
}
#endif

