/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Tracker.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.10 $	$Date: 1997/03/19 04:12:20 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Tracker is a device for measuring the 6D spatial position of a device.
 * This is an abstract base class which requires a particular subclass to
 * define how that data is acquired.
 *
 ***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "Tracker.h"

typedef float *floatptr;
typedef Buttons *ButtonsPtr;

// class constructor
// Note: the position and orientation must be initialized by
//         derived classes.
Tracker::Tracker(int num_sensors) {
  int i;

  sensors = num_sensors;
  maxFilterSize = DEF_MAX_FILTER_SIZE;
  scaleFactor = 1.0;		  // to get units in meters; trackers should
				  // report distances in meters.
  trackerPaused = FALSE;
  filterPositionThreshold = 0.1;  // init the threshold to 1 cm

  if(sensors <= 0) {  // make sure you have a valid parameter
    sensors = 0;         //  If not, clear all the variables
    sensorAvailable = NULL;
    position = orientation = NULL;
    filterSize = filterSlot = NULL;
    filterPos = filterOrient = NULL;
    buttons = NULL;
   } else {
      // allocate space for the variables
    sensorAvailable = new int[sensors];
    buttons = new ButtonsPtr[sensors];
    for(i=0; i < sensors; i++) {    
      sensorAvailable[i] = FALSE;
      buttons[i] = NULL;
    }
    position = new float[sensors * 3];  // store current x,y, and z values
    orientation = new float[sensors * 9];  // store current orientation matrix

    // if you are averaging the data, then allocate memory for it
    if(maxFilterSize > 0) {
         // Create a queue for each sensor
      filterSize = new int[sensors];  // keep track of the current size
      filterSlot = new int[sensors];  // point to the queue head
      filterPos = new floatptr[sensors];  // will store the previous positions
      filterOrient = new floatptr[sensors]; // will store the prev. orientations
       // initialized each of the newly allocated variables
      for(i=0; i < sensors; i++) {
        filterSize[i] = 0;    // start off with no data
        filterSlot[i] = 0;    // queue head == 0th element
        filterPos[i] = new float[maxFilterSize * 3];  // actual pos. data
        filterOrient[i] = new float[maxFilterSize * 9]; // actual orient. data
      }
    }  // end of the filter-averaging allocation routine
  } // end of the code for correctly defined (sensors>0) constructor
}  // end of the constructor


// class destructor
Tracker::~Tracker(void) {
  int i;
  if(sensors > 0) {
    delete [] orientation;
    delete [] position;
    delete [] sensorAvailable;
    if(maxFilterSize > 0) {
      delete [] filterSize;
      delete [] filterSlot;
      for(i=0; i < sensors; i++) {
        delete [] filterOrient[i];
        delete [] filterPos[i];
      }
      delete [] filterOrient;
      delete [] filterPos;
      delete [] buttons;
    }
  }
}

// update the sensor position.  This calls the virtual routine for the
// specific type of tracker, and uses a boxcar average to filter the
// data.  If the difference between the newly acquired data
// and the previous average exceeds a specific threshold, the filter
// will cut out.  This makes the system more responsive for fast
// movements
void Tracker::update_sensors(void) {
  float *fpos, *forient, *dpos, *dorient;  // d is for data, f is for dummy
  float prevpos[3];   // use this to store the previous positions
  register int i,j,k;

  // call virtual routine for each station.  If the routine returns FALSE,
  // nothing is done for that station.
  for(i=0; i < sensors; i++) {
    memcpy(prevpos, position + 3*i, 3*sizeof(float)); // store the old position
    dpos = position + 3*i;    // point dpos and dorient to the proper storage
    dorient = orientation + 9*i;  // area in position and orientation arrays
    // on the first element (i==0), aquire all the data (use old data for i>0) 
    // if the value for sensor i is correct (aka valid)
    if(!trackerPaused && 
       my_get_data(i, dpos, dpos + 1, dpos + 2, dorient,(i == 0))) {
        // scale the position just read by the scaling factor
        *dpos *= scaleFactor;
        *(dpos+1) *= scaleFactor;
        *(dpos+2) *= scaleFactor;

//      printf("XYZ=(%f %f %f)  orientDEF=(%f %f %f)\n", dpos[0], dpos[1], dpos[2], dorient[3], dorient[4], dorient[5]);
      if(maxFilterSize > 0) {    
        // if so, check if filtering is enabled then check to see if the 
	// position hasn't moved too far
        if (fabs(prevpos[0] - dpos[0]) + fabs(prevpos[1]-dpos[1]) +
             fabs(prevpos[2]-dpos[2]) < filterPositionThreshold )  {
          // if not, store the data in the filter
          // as pointed to by filterSlot (the queue head)
	  fpos = filterPos[i] + 3 * filterSlot[i];
          memcpy(fpos, dpos, 3*sizeof(*dpos));

          // do the same for the orientation data
	  forient = filterOrient[i] + 9 * filterSlot[i]; 
          memcpy(forient, dorient, 9*sizeof(*dorient));

//          *(fpos++) = *(dpos);      	// copy position data
//          *(fpos++) = *(dpos + 1);
//          *(fpos++) = *(dpos + 2);

//          for(j=0; j < 9; j++)  	// copy the orientation data
//            *(forient++) = *(dorient + j);

          if(filterSize[i] < maxFilterSize)   // grow the queue with aquired data until
               filterSize[i]++;               // it is full

          // increment the queue head and if it is too large, wrap it around to zero
          filterSlot[i] = (filterSlot[i] + 1) % maxFilterSize;


          // average the stored values, and put result in current variables
          fpos    = filterPos[i];
          forient = filterOrient[i];
          for(k=0; k < 3; dpos[k++]    = 0.0)  // zero the accumulators
            ;
          for(k=0; k < 9; dorient[k++] = 0.0)
            ;
          for(j=0; j < filterSize[i]; j++) {    // for each term saved
            for(k=0; k < 3; k++)                 // add the positions
              dpos[k] += *(fpos++);
            for(k=0; k < 9; k++)                 // and velocities
              dorient[k] += *(forient++);
           }
         for(k=0; k < 3; dpos[k++] /= filterSize[i])  // and average the results
            ;
         for(k=0; k < 9; dorient[k++] /= filterSize[i])
            ;
         // end of the averaging/filter routine
        } else {  // however, if the tracker position has moved too far......
         // then cut out the averaging
          filterSlot[i]=0;  // by setting the queue head to zero
          filterSize[i]=0;  // and the queue size to zero and taking the 
			    // values of dpos and dorient to be correct
                            // NB: there may be a tiny shiver for the 
			    // first couple data points if the first 
			    // (uninitialized) read is within the threshold; but
                            // don't worry about it.
        }   // end of the thresholding
      }  // end of the filtering routines
    }  // end of the initialization of working sensors
  }  // end of the loop that goes through the sensors
} // end of update_sensors


// change the current position threshold value for all the sensors
void Tracker::change_position_threshold(float newThreshold) {
  if (newThreshold >=0.0)
    filterPositionThreshold = newThreshold;
}


// change the current scaling factor
void Tracker::set_scale(float newScale) {
  if(newScale >= 0.0)
    scaleFactor = newScale;
}


// read position of sensor
// NOTE: this will call the update routine unless the user specifically
//    requests that no update be done by giving a fourth argument
//    of FALSE.
void Tracker::get_position(int station, float& x, float& y, float& z, 
                           int do_update) {
  register float *pos;
  if(station >= 0 && station < sensors) {
    pos = position + (3*station);
    if(do_update)  update_sensors();
    x = *(pos++);
    y = *(pos++);
    z = *pos;
  }
}

// set the tracker sensor position to a specific value
void Tracker::set_position(int station, float x, float y, float z)
{
   if (station >= 0 && station < sensors) {
      register float *pos = position + 3*station;
      *(pos++) = x;
      *(pos++) = y;
      *(pos++) = z;
   }
}
// apply an offset to a tracker/sensor position
void Tracker::mod_position(int station, float x, float y, float z)
{
   if (station >= 0 && station < sensors) {
      register float *pos = position + 3*station;
      *(pos++) += x;
      *(pos++) += y;
      *(pos++) += z;
   }
}


// NOTE: this will call the update routine unless the user specifically
//    requests that no update be done by giving a fourth argument
//    of FALSE.
void Tracker::get_orientation(int station, float *orient, int do_update) {
  if(station >= 0 && station < sensors) {
    if(do_update)  update_sensors();
    memcpy(orient, orientation + 9*station, 9*sizeof(float));
  }
}

// set the rotation matrix
void Tracker::set_orientation(int station, float *new_orient)
{
   if (station >=0 && station < sensors) {
      memcpy(orientation + 9*station, new_orient, 9*sizeof(float));
   }
}
//  apply a new orientation to the current matrix
void Tracker::mod_orientation(int station, float *mod_orient)
{
   if (station >= 0 && station < sensors) {
      float *orient = orientation + 9*station;
      float tmp[3][3];
      // matrix multiplication
      int i;
      for (i=0; i<3; i++) {
	 for (int k=0; k<3; k++) {
	    tmp[i][k] = 0;
	    for (int j=0; j<3; j++) {
	       tmp[i][k] = orient[3*i+j] * mod_orient[3*j+k];
	    }
	 }
      }
      for (i=0; i<9; i++) {  // apply the result
	 orient[i] = tmp[i/3][i%3];
      }
   }
}
// associate a Buttons object with a specific sensor on this tracker
void Tracker::assign_buttons(int sensor, Buttons *new_buttons) {
  if (sensor >=0 && sensor < sensors) {
    buttons[sensor] = new_buttons;
  }
}

// get the button associated with a sensor, or a NULL
Buttons *Tracker::get_buttons(int sensor) {
  if (sensor >=0 && sensor < sensors) {
    return buttons[sensor];
  }
  return NULL;
}

