/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: startup.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.19 $	$Date: 1997/03/22 06:05:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Startup code for VMD ... used only by Global files, contains variables
 * and routines to read initial data files and scripts, and parse
 * command-line options.
 ***************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "startup.h"
#include "Global.h"
#include "CommandQueue.h"
#include "UIText.h"
#include "CmdMol.h"
#include "MoleculeFile.h"
#include "ConfigList.h"
#include "utilities.h"

//
// externally-accessed variables
//
int which_display;			// type of display to use
float displayHeight, displayDist;	// height and distance of display
int displaySize[2], displayLoc[2];	// X,Y size and position of display
int showTitle;				// how to show the title at start
int have_sigma = FALSE;			// want to connect to sigma?

char *displayTypeNames[NUM_DISPLAY_TYPES] = { "WIN", "CAVE", "TEXT", "CAVEFORMS" };
char *titleTypeNames[NUM_TITLE_TYPES] = { "OFF", "ON" };

//
// flags for whether these variables have been set yet
//
int have_which_display = FALSE;
int have_displayHeight = FALSE, have_displayDist = FALSE;
int have_displaySize = FALSE, have_displayLoc = FALSE;
int have_showTitle = FALSE;

//
// filenames for init and startup files
//
static char *initFileStr = NULL;
static char *startupFileStr = NULL;
static char *beginCmdFile = NULL;

//
// filenames for initial molecule to load
//
static char *pdbFileName = NULL;
static char *psfFileName = NULL;
static char *dcdFileName = NULL;


/////////////////////////  routines  ///////////////////////////////

// look for all environment variables VMD can use, and initialize the
// proper variables.  If an env variable is not found, use a default value.
// ENVIRONMENT VARIABLES USED BY VMD (default values set in config.h):
//	VMDDIR		directory with VMD data files and utility programs
//	VMDTMPDIR	directory in which to put temporary files (def: /tmp)
//	VMDBABELBIN	full path to the babel executable
//	VMDHTMLVIEWER	program to use to display HTML documents (def: Mosaic)
void VMDgetEnvironment(void) {
  char *envtxt;

  // VMDDIR: directory with VMD data files and utility programs
  //		default: DEF_VMDENVVAR
  if(dataPath) delete [] dataPath;
  if((envtxt = getenv("VMDDIR")) != NULL)
    dataPath = stringdup(envtxt);
  else
    dataPath = stringdup(DEF_VMDENVVAR);
  // strip trailing / characters ...
  while(strlen(dataPath) > 0 && dataPath[strlen(dataPath)-1] == '/')
    dataPath[strlen(dataPath)-1] = '\0';

  // VMDTMPDIR: directory in which to put temporary files (def: /tmp)
  //		default: DEF_VMDTMPDIR
  if(myTempDir)  delete [] myTempDir;
  if((envtxt = getenv("VMDTMPDIR")) != NULL)
    myTempDir = stringdup(envtxt);
  else
    myTempDir = stringdup(DEF_VMDTMPDIR);
  // strip trailing / characters ...
  while(strlen(myTempDir) > 0 && myTempDir[strlen(myTempDir)-1] == '/')
    myTempDir[strlen(myTempDir)-1] = '\0';

  // VMDBABELBIN: set up the path to Babel (for file conversion)
  // this is either NULL or executable for at least someone
  //		default: DEF_VMDBABELBIN
  if (vmdBabelBin) delete [] vmdBabelBin;
  if ((envtxt = getenv("VMDBABELBIN")) != NULL)
    vmdBabelBin = stringdup(envtxt);
  else
    vmdBabelBin = stringdup(DEF_VMDBABELBIN);

  struct stat buf;
  if (!stat(vmdBabelBin, &buf) &&
  	buf.st_mode | S_IXUSR ||
	buf.st_mode | S_IXGRP ||
	buf.st_mode | S_IXOTH   ) {
  } else {
    delete [] vmdBabelBin;
    vmdBabelBin = NULL;
  }
  
  // VMDHTMLVIEWER: name of the HTML viewer to use (or NULL if none)
  //		default: DEF_VMDHTMLVIEWER
  if (vmdHTMLViewer) delete [] vmdHTMLViewer;
  if((envtxt = getenv("VMDHTMLVIEWER")) != NULL)
    vmdHTMLViewer = stringdup(envtxt);
  else {
    vmdHTMLViewer = stringdup(DEF_VMDHTMLVIEWER);
    static char tmp[sizeof(DEF_VMDHTMLVIEWER) + 20];
    sprintf(tmp, "VMDHTMLVIEWER=%s", DEF_VMDHTMLVIEWER);
    putenv(tmp);
  }

  // VMDSNAPSHOT: how to call the external snapshot "renderer"
  if((envtxt = getenv("VMDSNAPSHOT")) != NULL)
    vmdSnapshot = stringdup(envtxt);
  else
    vmdSnapshot = stringdup(DEF_VMDSNAPSHOT);
}


// initialize variables which indicate how VMD starts up, and
// parse the command-line options
void VMDparseCommandLine(int argc, char **argv) {

  // initialize variables
  which_display = INIT_DEFDISPLAY;
  showTitle = INIT_DEFTITLE;
  displayHeight = INIT_DEFHEIGHT;
  displayDist = INIT_DEFDIST;
  displaySize[0] = displayLoc[0] = (-1);

  // go through the arguments
  int ev = 1;
  while(ev < argc) {
    if(*(argv[ev]) != '-' || !strupcmp(argv[ev], "-pdb")) {
      if(pdbFileName) {
	msgErr << "More than one PDB file specified.  Using original file '";
	msgErr << pdbFileName << "'." << sendmsg;
      } else {
        if(*(argv[ev]) != '-')
          pdbFileName = stringdup(argv[ev]);
	else if(argc > (ev + 1) && *(argv[ev+1]) != '-')
          pdbFileName = stringdup(argv[++ev]);
        else
          msgErr << "-pdb must also specify a filename." << sendmsg;
      }

    } else if(!strupcmp(argv[ev], "-psf")) {
      if(psfFileName) {
	msgErr << "More than one PSF file specified.  Using original file '";
	msgErr << psfFileName << "'." << sendmsg;
      } else if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
        psfFileName = stringdup(argv[++ev]);
      } else {
        msgErr << "-psf must also specify a filename." << sendmsg;
      }

    } else if(!strupcmp(argv[ev], "-dcd")) {
      if(dcdFileName) {
	msgErr << "More than one DCD file specified.  Using original file '";
	msgErr << dcdFileName << "'." << sendmsg;
      } else if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
        dcdFileName = stringdup(argv[++ev]);
      } else {
        msgErr << "-dcd must also specify a filename." << sendmsg;
      }

    } else if(!strupcmp(argv[ev], "-dist")) {
      if(argc > (ev + 1)) {
        displayDist = atof(argv[++ev]);
	have_displayDist = TRUE;
      } else
        msgErr << "-dist must also specify a distance." << sendmsg;

    } else if(!strupcmp(argv[ev], "-e")) {
      if(argc > (ev + 1)) {
        if(beginCmdFile)  delete [] beginCmdFile;
        beginCmdFile = stringdup(argv[++ev]);
      } else
        msgErr << "-e must also specify a filename." << sendmsg;

    } else if(!strupcmp(argv[ev], "-height")) {
      if(argc > (ev + 1)) {
        displayHeight = atof(argv[++ev]);
	have_displayHeight = TRUE;
      } else
        msgErr << "-height must also specify a distance." << sendmsg;

    } else if(!strupcmp(argv[ev], "-pos")) {
      if(argc > (ev + 2) && *(argv[ev+1]) != '-' && *(argv[ev+2]) != '-') {
        displayLoc[0] = atoi(argv[++ev]);
        displayLoc[1] = atoi(argv[++ev]);
	have_displayLoc = TRUE;
      } else
        msgErr << "-pos must also specify an X Y pair." << sendmsg;

    } else if(!strupcmp(argv[ev], "-size")) {
      if(argc > (ev + 2) && *(argv[ev+1]) != '-' && *(argv[ev+2]) != '-') {
        displaySize[0] = atoi(argv[++ev]);
        displaySize[1] = atoi(argv[++ev]);
	have_displaySize = TRUE;
      } else
        msgErr << "-size must also specify an X Y pair." << sendmsg;

    } else if(!strupcmp(argv[ev], "-init")) {
      // use next argument as init file name
      if(argc > (ev + 1) && *(argv[ev+1]) != '-')
        initFileStr = stringdup(argv[++ev]);
      else
        msgErr << "-init must also have a new file name specified."<< sendmsg;

    } else if(!strupcmp(argv[ev], "-startup")) {
      // use next argument as startup config file name
      if(argc > (ev + 1))
        startupFileStr = stringdup(argv[++ev]);
      else
        msgErr << "-startup must also have a new file name specified."
	       << sendmsg;

    } else if(!strupcmp(argv[ev], "-nt")) {
      // do not print out the program title
      showTitle = TITLE_OFF;
      have_showTitle = TRUE;

    } else if(!strupcmp(argv[ev], "-debug")) {
      // turn on debugging
      msgDebug.on(TRUE);
      if(argc > (ev + 1) && argv[ev+1][0] != '-') {
        // also set the debugging level to the one given
        msgDebug.output_level(atoi(argv[++ev]));
      }

    } else if (!strupcmp(argv[ev], "-dispdev")) {  // startup Display
      ev++;
      have_which_display = TRUE;
      if (!strupcmp(argv[ev], "cave")) {  // use the cave 
        which_display = DISPLAY_CAVE;
      } else if (!strupcmp(argv[ev], "win")) { // use the GLDisplay (default)
        which_display = DISPLAY_WIN;
      } else if (!strupcmp(argv[ev], "text")) { // do not use any display
        which_display = DISPLAY_TEXT;
      } else if (!strupcmp(argv[ev], "caveforms")) { // use Forms and CAVE
        which_display = DISPLAY_CAVEFORMS;  // useful with "-display machine:0"
      } else if (!strupcmp(argv[ev], "none")) { // do not use any display
        which_display = DISPLAY_TEXT;
      } else {
        have_which_display = FALSE;
        msgErr << "-dispdev options are 'win' (default), 'cave', 'caveforms', or 'text | none'";
        msgErr << sendmsg;
      }


    } else if (!strupcmp(argv[ev], "-?") || !strupcmp(argv[ev], "-h")) {
      // print out command-line option summary
      msgInfo << "Available command-line options:" << sendmsg;
      msgInfo << "<filename> (with no - option)     Load specified PDB file\n";
      msgInfo << "\t-dcd <filename>     Load specified DCD trajectory file\n";
      msgInfo << "\t-debug [n]          Turn on debugging, at level n\n";
      msgInfo << "\t-dispdev <win | cave | text | none> Specify display device";
      msgInfo << sendmsg;
      msgInfo << "\t-dist <d>           Distance from origin to screen";
      msgInfo << sendmsg;
      msgInfo << "\t-e <filename>       Execute commands in <filename>";
      msgInfo << "\t-h (or -?)          Display this command-line summary\n";
      msgInfo << "\t-height <h>         Height of display screen";
      msgInfo << sendmsg;
      msgInfo << "\t-init <filename>    Specify initialization file";
      msgInfo << sendmsg;
      msgInfo << "\t-pdb <filename>     Load specified PDB file\n";
      msgInfo << "\t-pos <X> <Y>        Lower-left corner position of display";
      msgInfo << sendmsg;
      msgInfo << "\t-psf <filename>     Load specified PSF structure file\n";
      msgInfo << "\t-nt			No title display at start" << sendmsg;
      msgInfo << "\t-size <X> <Y>       Size of display" << sendmsg;
      msgInfo << "\t-startup <filename> Specify startup script file" << sendmsg;
#ifdef VMDSIGMA
      msgInfo << "\t-sigma              Connect to Sigma MD code" << sendmsg;
#endif
      exit(0);

    } else if (!strupcmp(argv[ev], "-sigma")) {
      // bind to Sigma MD code
#ifdef VMDSIGMA
      have_sigma = TRUE;
#else
      have_sigma = FALSE;
      msgWarn << "-sigma is not supported in this compiled version" << sendmsg;
#endif
    } else {
      // unknown option ... just ignore, could be used by others
      MSGDEBUG(1,"Unknown command-line option '" << argv[ev] << "'" << sendmsg);
    }
    ev++;
  }

  // command-line options have been parsed ... any init status variables that
  // have been given initial values will have flags saying so, and their
  // values will not be changed when the init file(s) is parsed.
}


// process an init file, by reading it in and checking for keywords.  Items
// that have had their value set already will not be changed.
void VMDprocessInitFile(char *fname) {
  ConfigList *configList;
  char *dispArgv[64], *dispStr, *strList;
  int dispArgc;

  // read in the file, which is of the form <keyword> = <value>
  configList = new ConfigList;
  if(! configList->read(fname)) {
    // file not found ... just return
    delete configList;
    return;
  }
  
  // check for proper keywords, and set values if found
  if(!have_which_display && (strList = configList->find("DISPLAY"))) {
    for(int i=0; i < NUM_DISPLAY_TYPES; i++) {
      if(!strupcmp(strList, displayTypeNames[i])) {
	which_display = i;
	break;
      }
    }
  }
  
  if(!have_showTitle && (strList = configList->find("TITLE"))) {
    for(int i=0; i < NUM_TITLE_TYPES; i++) {
      if(!strupcmp(strList, titleTypeNames[i])) {
	showTitle = i;
	break;
      }
    }
  }
  
  if(!vmdBabelBin && (strList = configList->find("BABEL"))) {
    vmdBabelBin = stringdup(strList);
    struct stat buf;
    if (!stat(vmdBabelBin, &buf) &&
	  buf.st_mode | S_IXUSR ||
	  buf.st_mode | S_IXGRP ||
	  buf.st_mode | S_IXOTH   ) {
    } else {
      delete [] vmdBabelBin;
      vmdBabelBin = NULL;
    }
  }

  if(!vmdHTMLViewer && (strList = configList->find("HTMLVIEWER")))
    vmdHTMLViewer = stringdup(strList);

  if(!have_displayHeight && (strList = configList->find("SCRHEIGHT")))
    displayHeight = atof(strList);

  if(!have_displayDist && (strList = configList->find("SCRDIST")))
    displayDist = atof(strList);

  if(!have_displayLoc && (strList = configList->find("SCRPOS"))) {
    dispStr = NULL;
    if((dispStr = str_tokenize(strList, &dispArgc, dispArgv)) != NULL
    		&& dispArgc == 2) {
      displayLoc[0] = atoi(dispArgv[0]);
      displayLoc[1] = atoi(dispArgv[1]);
    } else {
      msgErr << "Illegal SCRPOS initialization setting '" << strList << "'.";
      msgErr << sendmsg;
    }
    if(dispStr)  delete [] dispStr;
  }
  
  if(!have_displaySize && (strList = configList->find("SCRSIZE"))) {
    dispStr = NULL;
    if((dispStr = str_tokenize(strList, &dispArgc, dispArgv)) != NULL
    		&& dispArgc == 2) {
      displaySize[0] = atoi(dispArgv[0]);
      displaySize[1] = atoi(dispArgv[1]);
    } else {
      msgErr << "Illegal SCRSIZE initialization setting '" << strList << "'.";
      msgErr << sendmsg;
    }
    if(dispStr)  delete [] dispStr;
  }

  // done with config list .. delete it
  delete configList;
}



// read in the init file(s).  The order of searching is the following, and EACH
// file found in this list will be processed.
//	1. global data dir as specified by dataPath
//	2. $HOME
//	3. .
// If an init file was specified on the command line, the above three searches
// are skipped, only the specified file is processed.
void VMDreadInit(void) {
  char namebuf[512], *envtxt;
  
  if(!dataPath)
    VMDgetEnvironment();

  // use an init file if previously specified, 
  if(initFileStr) {
    VMDprocessInitFile(initFileStr);
    delete [] initFileStr;
    return;
  }
  
  // 1. look in global data dir for the file.
  strcpy(namebuf, dataPath);
  strcat(namebuf, "/");
  strcat(namebuf, VMD_INITFILE);
  VMDprocessInitFile(namebuf);
  
  // 2. look in home directory
  if (envtxt = getenv("HOME")) {
    strcpy(namebuf, envtxt);
    strcat(namebuf, "/");
    strcat(namebuf, VMD_INITFILE);
    VMDprocessInitFile(namebuf);
  }
  
  // 3. look in current directory
  strcpy(namebuf, VMD_INITFILE);
  VMDprocessInitFile(namebuf);

}



// read in the startup script, execute it, and then execute any other commands
// which might be necessary (i.e. to load any molecules at start)
// This searches for the startup file in the following
// places (and in this order), reading only the FIRST one found:
//		1. Current directory
//		2. Home directory
//		3. 'Default' directory (here, /usr/local/vmd)
// If a name was given in the -startup switch, that file is checked for ONLY.
void VMDreadStartup(void) {
  char namebuf[512], *envtxt;
  int found = FALSE;
  struct stat statbuf;

  if(!dataPath)
    VMDgetEnvironment();

  // check if the file is available
  if(startupFileStr) {	// name specified by -startup
    if(stat(startupFileStr, &statbuf) == 0) {
      found = TRUE;
      strcpy(namebuf, startupFileStr);
    }
    delete [] startupFileStr;
  } else {	// search in different directories, for default file
    // first, look in current dir
    strcpy(namebuf, VMD_STARTUP);
    if(stat(namebuf, &statbuf) == 0) {
      found = TRUE;
    } else {
      // not found in current dir; look in home dir
      if((envtxt = getenv("HOME")) != NULL)
        strcpy(namebuf,envtxt);
      else
        strcpy(namebuf,".");
      strcat(namebuf,"/");
      strcat(namebuf,VMD_STARTUP);
      if(stat(namebuf, &statbuf) == 0) {
        found = TRUE;
      } else {
        // not found in home dir; look in default dir
	strcpy(namebuf, dataPath);
	strcat(namebuf,"/");
	strcat(namebuf,VMD_STARTUP);
        if(stat(namebuf, &statbuf) == 0) {
          found = TRUE;
	}
      }
    }
  }

  //
  // execute any commands needed at start
  //
  
  // read in molecule if requested via command-line switches
  // creates either a regular MoleculeFile or MoleculeSigma depending
  //  on command-line "-sigma" option
  if(pdbFileName || (psfFileName && dcdFileName)) {
    int sftype, cftype;
    char *strfile, *coorfile;
    sftype = (psfFileName ? MoleculeFile::PSF : MoleculeFile::PDB);
    cftype = (dcdFileName ? CoorFileData::DCD : CoorFileData::PDB);
    strfile = (psfFileName ? psfFileName : pdbFileName);
    coorfile = (psfFileName ? (dcdFileName ? dcdFileName 
                                 : pdbFileName) : (char *) NULL);
#ifdef VMDSIGMA
    if (have_sigma)
      commandQueue->append(new CmdMolNew(strfile, sftype, coorfile, cftype,
						CmdMolNew::MOLSIGMA));
    else
#endif
      commandQueue->append(new CmdMolNew(strfile, sftype, coorfile, cftype));
  }

  // after reading in startup file and loading any molecule, the file
  // specified by the -e option is set up to be executed.  But we put the
  // command here FIRST to read the file since reading a new file suspends
  // reading of a previously provided file.
  if(beginCmdFile) {
    if (uiText)
      uiText->read_from_file(beginCmdFile);
  }

  // if the startup file was found, read in the commands there
  if(found) {
    if (uiText)
    uiText->read_from_file(namebuf);
  }
  
  // delete filename storage
  if(pdbFileName)  delete [] pdbFileName;
  if(psfFileName)  delete [] psfFileName;
  if(dcdFileName)  delete [] dcdFileName;
  if(beginCmdFile) delete [] beginCmdFile;
}

