/* algebra.h - vector algebra functions on a sphere
 *
 * $Id: algebra.h,v 1.3 96/02/11 21:28:04 leech Exp $
 *
 * Copyright (C) 1996, Jonathan P. Leech
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is". Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Log:	algebra.h,v $
 * Revision 1.3  96/02/11  21:28:04  leech
 * Add a distance-computing function to ForceLaw.
 * 
 * Revision 1.2  96/02/04  22:46:49  leech
 * Encapsulated force laws in a class with common interface.
 *
 * Revision 1.1  92/04/27  04:25:11  leech
 * Initial revision
 *
 */

#ifndef _jpl_algebra_h
#define _jpl_algebra_h

#include <math.h>
#include "dvector.h"

dvector
    unit_random(),
    normal_component(const dvector &v, const dvector &vref),
    tangent_component(const dvector &v, const dvector &vref);

double
    arclength(const dvector &r1, const dvector &r2);

class ForceLaw {
public:
    enum DistanceFunction { kArcDistance = 0, kEuclidDistance };
    enum FalloffFunction { kSquareLaw = 0, kLinearLaw, kLennartLaw, kMorseLaw };

    ForceLaw(double k, double epsilon, double r0);

    // Accessors
    double GetSoftening() const;
    double GetKForce() const;
    double GetR0() const;
    DistanceFunction GetDistanceFunction() const;
    FalloffFunction GetFalloffFunction() const;

    // Modifiers
    void SetSoftening(double);
    void SetKForce(double);
    void SetR0(double);
    void SetDistanceFunction(DistanceFunction);
    void SetFalloffFunction(FalloffFunction);

    dvector FindForce(const dvector &to, const dvector &from);
    double  Distance(const dvector &to, const dvector &from);

private:
    void SetForceFunction();

    dvector ArcSquare(const dvector &to, const dvector &from);
    dvector ArcLinear(const dvector &to, const dvector &from);
    dvector ArcLennart(const dvector &to, const dvector &from);
    dvector ArcMorse(const dvector &to, const dvector &from);
    dvector EuclidSquare(const dvector &to, const dvector &from);
    dvector EuclidLinear(const dvector &to, const dvector &from);
    dvector EuclidLennart(const dvector &to, const dvector &from);
    dvector EuclidMorse(const dvector &to, const dvector &from);

    double  LennartJones(double r);
    double  MorsePotential(double r);
    double  MorseForce(double r);

    double  k, epsilon, r0;
    DistanceFunction distanceFunction;
    FalloffFunction falloffFunction;
    dvector (ForceLaw::*forceFunction)(const dvector &to, const dvector &from);

    // Cached force terms for Morse law
    void    UpdateTerms();
    double  morseDepth, morseBeta;
};

#endif /*_jpl_algebra_h*/
