############################################################################
#cr                                                                       
#cr            (C) Copyright 1995 The Board of Trustees of the            
#cr                        University of Illinois                         
#cr                         All Rights Reserved                           
#cr                                                                       
############################################################################

############################################################################
# RCS INFORMATION:
#
# 	$RCSfile: fit.tcl,v $
# 	$Author: dalke $	$Locker:  $		$State: Exp $
#	$Revision: 1.2 $	$Date: 1997/03/24 05:29:59 $
#
############################################################################
# DESCRIPTION:
#   Mouse based best fit (RMSD) alignment routines
#
############################################################################

#  align                based on
# ------------------------------------
# (molecule|fragment) (all atoms|heavy atoms|backbone|CA|two atoms)

# the fit starts(or resets) when vmd_mouse_mode changes to "user"
# and the submode is 0

#  There is too much code here; most of it is repetative.  If I used
#  #defines macros I would be much happier (I don't yet know the 
#  equivalent in Tcl)

# set true if already doing a start
set vmd_fit_started 0

# There are 5 modes and 2 types, so: 
#   0 == molecule/all atoms
#   1 == molecule/heavy atoms
#     [...]
#   4 == molecule/two atoms
#   5 == fragment/all atoms
#     [...]
#   9 == fragment/two atoms
# This is used in the mouse popup menu to indicate the fit mode
#  (but only if vmd_fit_started is true)
set vmd_fit_mode 0

# I trace submode since it is changed _after_ mode
trace variable vmd_mouse_submode w vmd_fit_trace

# call this to stop tracing picked atoms
proc vmd_fit_delete_trace {} {
    global vmd_pick_atom vmd_fit_started vmd_fit_started
    trace vdelete vmd_pick_atom w vmd_fit_trace_atom
    set vmd_fit_started 0
}


# call this whenever the vmd_mouse_mode changes
# it sets up the trace to watch mouse picks
proc vmd_fit_trace {args} {
    global vmd_mouse_mode vmd_mouse_submode vmd_fit_started vmd_fit_count
    global vmd_pick_atom
    if [string compare $vmd_mouse_mode "user"] {
	vmd_fit_delete_trace
	return
    }
    if {$vmd_mouse_submode != 0} {
	vmd_fit_delete_trace
	return
    }
    if {$vmd_fit_started == 0} {
	trace variable vmd_pick_atom w vmd_fit_trace_atom
	set vmd_fit_count 0
	set vmd_fit_started 1
    }
    # don't do anything if the fit was already in progress
}


# when an atom is picked, call here
proc vmd_fit_trace_atom {args} {
    global vmd_pick_atom vmd_pick_mol vmd_fit_count vmd_fit_atom vmd_fit_mol
    global vmd_fit_function
    # store the picked atom index and molecule index
    set vmd_fit_atom($vmd_fit_count) $vmd_pick_atom
    set vmd_fit_mol($vmd_fit_count) $vmd_pick_mol
    incr vmd_fit_count
    # if there have been two atoms, do the fit
    if {$vmd_fit_count == 2} {
	# call the function which does the fit
	if [catch {$vmd_fit_function $vmd_fit_mol(0) $vmd_fit_atom(0) $vmd_fit_mol(1) $vmd_fit_atom(1)} errmsg] {
	    puts $errmsg
	}
	# reset in preperation for another fit

	puts "To repeat calculation, select two more atoms"
	puts "To stop alignment, change mouse mode (eg, 'rotate' or press 'r')"
	set vmd_fit_count 0
    } else {
	puts "The first atom has been picked (this selection will move)"
	puts "Now pick the second atom (on the reference selection)"
    }
}

# Initially these will do the alignment and print the RMSD,
# but setting the variable to false means just print the RMSD
set vmd_fit_do_align 1

# these are here for VMD "Mouse.C" to call
proc vmd_fit_do_align_true {} {
    global vmd_fit_do_align
    set vmd_fit_do_align 1
    return
}
proc vmd_fit_do_align_false {} {
    global vmd_fit_do_align
    set vmd_fit_do_align 0
    return
}

########## These are used by the pop up menu and their callbacks
## fit molecule 1 to molecule 2 based on all the atoms in each
proc vmd_fit_molecule_all {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 0
    # This is the function to use when the two atoms are picked
    set vmd_fit_function vmd_fit_do_molecule_all
    # go into fit mode (this resets if need be)
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two molecules based on all atoms."
    } else {
	puts "RMSD of two molecules based on all atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two molecules you wish to compare."
}

proc vmd_fit_do_molecule_all {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    # get the two selections
    puts "You have selected molecules $mol1 and $mol2 to compare."
    set sel1 [atomselect $mol1 {all}]
    set sel2 [atomselect $mol2 {all}]
    # compute the transformation 
    # and apply it to the first selection
    if $vmd_fit_do_align {
	$sel1 move [measure fit $sel1 $sel2 weight mass]
    }
    # print the RMSD between the two selection
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the two molecules is: $rmsd"
}

## fit molecule 1 to molecule 2 based on all the non-hydrogen atoms
# (this assumes hydrogen == name "[1-9]?H.*")
proc vmd_fit_molecule_heavy {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 1
    set vmd_fit_function vmd_fit_do_molecule_heavy
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two molecules based on non-hydrogen atoms"
    } else {
	puts "RMSD of two molecules based on non-hydrogen atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two molecules you wish to compare."
}

proc vmd_fit_do_molecule_heavy {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    puts "You have selected molecules $mol1 and $mol2 to compare."
    set sel1 [atomselect $mol1 {not name "[1-9]?H.*"}]
    set sel2 [atomselect $mol2 {not name "[1-9]?H.*"}]
    if $vmd_fit_do_align {
	[atomselect $mol1 all] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the heavy atoms of the molecules is: $rmsd"
}

## fit molecule 1 to molecule 2 based on all the backbone atoms
proc vmd_fit_molecule_backbone {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 2
    set vmd_fit_function vmd_fit_do_molecule_backbone
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two molecules based on backbone atoms"
    } else {
	puts "RMSD of two molecules based on backbone atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two molecules you wish to compare."
}

proc vmd_fit_do_molecule_backbone {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    puts "You have selected molecules $mol1 and $mol2 to compare."
    set sel1 [atomselect $mol1 backbone]
    set sel2 [atomselect $mol2 backbone]
    if $vmd_fit_do_align {
	[atomselect $mol1 all] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the backbone atoms of the molecules is: $rmsd"
}

## fit molecule 1 to molecule 2 based on all the CA atoms
proc vmd_fit_molecule_ca {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 3
    set vmd_fit_function vmd_fit_do_molecule_ca
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two molecules based on CA atoms"
    } else {
	puts "RMSD of two molecules based on CA atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two molecules you wish to compare."
}

proc vmd_fit_do_molecule_ca {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    puts "You have selected molecules $mol1 and $mol2 to compare."
    set sel1 [atomselect $mol1 {name CA}]
    set sel2 [atomselect $mol2 {name CA}]
    if $vmd_fit_do_align {
	[atomselect $mol1 all] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the CAs of the molecules is: $rmsd"
}

## fit molecule 1 to molecule 2 based on the two atoms
proc vmd_fit_molecule_pick {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 4
    set vmd_fit_function vmd_fit_do_molecule_pick
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two molecules based on successive picked atoms"
    } else {
	puts "RMSD of two molecules based on successive picked atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two molecules you wish to compare."
}

proc vmd_fit_do_molecule_pick {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    puts "You have selected molecules $mol1 and $mol2 to compare."
    set sel1 [atomselect $mol1 "index $atom1"]
    set sel2 [atomselect $mol2 "index $atom2"]
    if $vmd_fit_do_align {
	[atomselect $mol1 all] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the two points (aka 'distance') is: $rmsd"
}

#######
## fit fragment 1 to fragment 2 based on all the atoms in each
proc vmd_fit_fragment_all {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 5
    # This is the function to use when the two atoms are picked
    set vmd_fit_function vmd_fit_do_fragment_all
    # go into fit mode (this resets if need be)
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two fragments based on all atoms"
    } else {
	puts "RMSD of two fragments based on all atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two fragments you wish to compare."
}

proc vmd_fit_do_fragment_all {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    global sel1 sel2
    ## get the two fragments
    # get the fragment number
    set fnum1 [lindex [[atomselect $mol1 "index $atom1"] get fragment] 0]
    # and the fragment
    set sel1 [atomselect $mol1 "fragment $fnum1"]
    # fragment number for atom 2
    set fnum2 [lindex [[atomselect $mol2 "index $atom2"] get fragment] 0]
    # atom 2's fragment
    set sel2 [atomselect $mol2 "fragment $fnum2"]
    # compute the transformation 
    puts "You have selected fragments $fnum1 and $fnum2 to compare." 
    # and apply it to the first selection
    if $vmd_fit_do_align {
	$sel1 move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the two fragments is: $rmsd"
}

## fit fragment 1 to fragment 2 based on all the non-hydrogen atoms
# (this assumes hydrogen == name "[1-9]?H.*")
proc vmd_fit_fragment_heavy {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 6
    set vmd_fit_function vmd_fit_do_fragment_heavy
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two fragments based on non-hydrogen atoms"
    } else {
	puts "RMSD of two fragments based on non-hydrogen atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two fragments you wish to compare."
}

proc vmd_fit_do_fragment_heavy {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    set hydrogen  {"[1-9]?H.*"}
    set fnum1 [lindex [[atomselect $mol1 "index $atom1"] get fragment] 0]
    set sel1 [atomselect $mol1 "fragment $fnum1 and not name $hydrogen"]
    set fnum2 [lindex [[atomselect $mol2 "index $atom2"] get fragment] 0]
    set sel2 [atomselect $mol2 "fragment $fnum2 and not name $hydrogen"]
    puts "You have selected fragments $fnum1 and $fnum2 to compare."
    if $vmd_fit_do_align {
	[atomselect $mol1 "fragment $fnum1"] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the heavy atoms of the molecules is: $rmsd"
}

## fit fragment 1 to fragment 2 based on all the backbone atoms
proc vmd_fit_fragment_backbone {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 7
    set vmd_fit_function vmd_fit_do_fragment_backbone
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two fragments based on backbone atoms"
    } else {
	puts "RMSD of two fragments based on backbone atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two fragments you wish to compare."
}

proc vmd_fit_do_fragment_backbone {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    set fnum1 [lindex [[atomselect $mol1 "index $atom1"] get fragment] 0]
    set sel1 [atomselect $mol1 "fragment $fnum1 and backbone"]
    set fnum2 [lindex [[atomselect $mol2 "index $atom2"] get fragment] 0]
    set sel2 [atomselect $mol2 "fragment $fnum2 and backbone"]
    puts "You have selected fragments $fnum1 and $fnum2 to compare."
    if $vmd_fit_do_align {
	[atomselect $mol1 "fragment $fnum1"] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the backbone of the fragments is: $rmsd"
}

## fit fragment 1 to fragment 2 based on all the CA atoms
proc vmd_fit_fragment_ca {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 8
    set vmd_fit_function vmd_fit_do_fragment_ca
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two fragments based on CA atoms"
    } else {
	puts "RMSD of two fragments based on CA atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two fragments you wish to compare."
}

proc vmd_fit_do_fragment_ca {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    set fnum1 [lindex [[atomselect $mol1 "index $atom1"] get fragment] 0]
    set sel1 [atomselect $mol1 "fragment $fnum1 and name CA"]
    set fnum2 [lindex [[atomselect $mol2 "index $atom2"] get fragment] 0]
    set sel2 [atomselect $mol2 "fragment $fnum2 and name CA"]
    puts "You have selected fragments $fnum1 and $fnum2 to compare."
    if $vmd_fit_do_align {
	[atomselect $mol1 "fragment $fnum1"] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the CA atoms of the fragments is: $rmsd"
}

## fit fragment 1 to fragment 2 based on all the CA atoms
proc vmd_fit_fragment_pick {} {
    global vmd_fit_function vmd_fit_do_align vmd_fit_mode
    set vmd_fit_mode 9
    set vmd_fit_function vmd_fit_do_fragment_pick
    mouse mode user 0
    if $vmd_fit_do_align {
	puts "Best fit of two fragments based on successive picked atoms"
    } else {
	puts "RMSD of two fragments based on successive picked atoms"
    }
    puts "Using the mouse, click on one atom of each of"
    puts "the two fragments you wish to compare."
}

proc vmd_fit_do_fragment_pick {mol1 atom1 mol2 atom2} {
    global vmd_fit_do_align
    set fnum1 [lindex [[atomselect $mol1 "index $atom1"] get fragment] 0]
    set sel1 [atomselect $mol1 "index $atom1"]
    set fnum2 [lindex [[atomselect $mol2 "index $atom2"] get fragment] 0]
    set sel2 [atomselect $mol2 "index $atom2"]
    puts "You have selected fragments $fnum1 and $fnum2 to compare."
    if $vmd_fit_do_align {
	[atomselect $mol1 "fragment $fnum1"] move [measure fit $sel1 $sel2 weight mass]
    }
    set rmsd [measure rmsd $sel1 $sel2 weight mass]
    puts "RMSD between the two points (aka 'distance') is: $rmsd"
}

