############################################################################
#cr                                                                       
#cr            (C) Copyright 1995 The Board of Trustees of the            
#cr                        University of Illinois                         
#cr                         All Rights Reserved                           
#cr                                                                       
############################################################################

############################################################################
# DESCRIPTION:
#
# test various components of the vector/matrix functions
# This file is here for instructive purposes only.
# To use it, source the file.
############################################################################

proc compare {v1 v2} {
    set diff [veclength [vecsub $v1 $v2]]
    set sum  [veclength [vecadd $v1 $v2]]
    if {$sum == 0} {
	return $diff
    }
    return [expr $diff / $sum]
}

proc test_compare {v1 v2} {
    if {[compare $v1 $v2] > .00004} {
	puts [list *********Error value is [compare $v1 $v2]]
	return "Bad"}
    return Ok
}

proc test {phrase v1 v2} {
    set tmp [test_compare $v1 $v2]
    puts "$phrase $tmp" 
}

proc test_matrix {m1 m2} {
    lassign $m1 m11 m12 m13 m14
    lassign $m2 m21 m22 m23 m24
    if {![string compare [test_compare $m11 $m21] Bad] } {return "Bad1"}
    if {![string compare [test_compare $m12 $m22] Bad] } {return "Bad2"}
    if {![string compare [test_compare $m13 $m23] Bad] } {return "Bad3"}
    if {![string compare [test_compare $m14 $m24] Bad] } {return "Bad4"}
    return Ok
}

proc test_mat {phrase m1 m2} {
    set tmp [test_matrix $m1 $m2]
    puts "$phrase $tmp"
}

set v1 {1.1 2.3 4.7}
set v2 {9.8 7.5 6.4}
set v3 {-0.1 -0.2 -0.3}

puts [list test_compare: [test_compare $v1 $v1]]

set sum [vecadd $v1 $v2]
test "Sum: term x: " [lindex $sum 0] 10.9
test "Sum: term y: " [lindex $sum 1] 9.8
test "Sum: term z: " [lindex $sum 2] 11.1

set diff [vecsub $v2 $v1]
test "Diff: term x: " [lindex $diff 0] 8.7
test "Diff: term y: " [lindex $diff 1] 5.2
test "Diff: term z: " [lindex $diff 2] 1.7

test "Veczero 0: " [veczero] {0 0 0}

set sum2 [vecadd $v1 $v2 $v3 $v2]
test "Multiple sums: " $sum2 {20.6 17.1 17.2}

set sum3 [vecadd 1 2 3 4 5 6 7 8 9 10]
test "Single term sums: " $sum3 55

set sum4 [vecadd {1.0 1.1 1.2 1.3 1.4} {2.2 2.3 2.4 2.5 2.6} {10 11 12 13 14}]
test "Five term sums: " $sum4 {13.2 14.4 15.6 16.8 18.0}

set diff2 [vecsub 9.8 1.6]
test "Single term sub: " $diff2 8.2

set diff3 [vecsub {9 8 7 6 5 4 3 2} {6 4 2 0 -2 -4 -6 -8}]
test "Eight term sub: " $diff3 {3 4 5 6 7 8 9 10}

set v1scale [vecscale 5.1 $v2]
test "Vecscale 1: " $v1scale {49.98 38.25 32.64}

set v2scale [vecscale $v2 5.1]
test "Vecscale 2: " $v2scale {49.98 38.25 32.64}

set v3scale [vecscale -1.5 {1 2 3 4 5}]
test "Vecscale 3: " $v3scale {-1.5 -3.0 -4.5 -6.0 -7.5}

set dot [vecdot $v1 $v2]
test "Dot product: " $dot 58.11

set cross1 [veccross $v1 $v2]
test "Cross product 1: " $cross1 {-20.53 39.02 -14.29}
set cross2 [veccross {1 0 0} {0 1 0}]
test "Cross product 2: " $cross2 {0 0 1}

set vlen1 [veclength $v1]
test "Length 1: " $vlen1 5.34696

test "Length 2: " [veclength 3] 3

set vlen3 [veclength {-3 -4}]
test "Length 3: " $vlen3 5

set vlen21 [veclength2 $v1]
test "Length2 1: " $vlen21 [expr $vlen1 * $vlen1]

test "Length2 2: " [veclength2 3] 9

set vlen23 [veclength2 {-3 -4}]
test "Length2 3: " $vlen23 25

set norm1 [vecnorm $v2]
test "Norm 1: " $norm1 {.70496 .53951 .46038}

test "Norm 2: " [vecnorm 1] 1
test "Norm 3: " [vecnorm -3] -1

set invert1 [vecinvert $v2]
test "Invert 1: x: " [lindex $invert1 0] [expr [lindex $v2 0] * -1]
test "Invert 1: y: " [lindex $invert1 1] [expr [lindex $v2 1] * -1]
test "Invert 1: z: " [lindex $invert1 2] [expr [lindex $v2 2] * -1]

set invert2 [vecinvert $v3]
test "Invert 2: " [veclength [vecadd $invert2 $v3]] 0

# define some matricies
set mat1 {{1 0 0 0} {0 1 0 0} {0 0 1 0} {0 0 0 1}}
# a rotation about z by 45 degrees
set mat2 {{0.7071 -0.7071 0 0} 
          {0.7071  0.7071 0 0} 
          {0       0      1 0}
          {0       0      0 1}}

# a rotation about y by -10 degrees
set mat3 {
    { 0.9848 0.0000 -0.1736 0}
    { 0.0000 1.0000  0.0000 0}
    { 0.1736 0.0000  0.9848 0}
    { 0      0       0      1}
}


set tv1 [vectrans $mat1 $v2]
test "Vectrans 1: " $tv1 $v2
set tc1 [coordtrans $mat1 $v2]
test "Coordtrans 1: " $tc1 $v2


set tv2 [vectrans $mat2 {1 0 1.2}]
test "Vectrans 2: " $tv2 {.7071 .7071 1.2}
set tc2 [coordtrans $mat2 {1 0 1.2}]
test "Coordtrans 2: " $tc2 {.7071 .7071 1.2}

set tv3 [vectrans $mat3 {.3 .4 .5}]
test "Vectrans 3: " $tv3 {0.20864 0.400 0.54448}
set tc3 [coordtrans $mat3 {.3 .4 .5}]
test "Coordtrans 3: " $tc3 {0.20864 0.400 0.54448}

# let's make up a matrix
set mat4 {
    {1   2    3    4  }
    {13 13.1 13.2 13.3}
    {-1 -5   -9   -8  }
    {1e1 0  -5e-1  10 }
}

set tv4 [vectrans $mat4 $v2]
test "Vectrans 4: " $tv4 {44 310.13 -104.9}
set tc4 [coordtrans $mat4 $v2]
test "Coordtrans 4:" $tc4 {48 323.43 -112.9}

set tv5 [vectrans $mat4 "$v2 0"]
test "Vectrans 5: " $tv5 {44 310.13 -104.9 94.8}
set tc5 [coordtrans $mat4 "$v2 1"]
test "Coordtrans 5:" $tc5 {48 323.43 -112.9 104.8}

test_mat "Transidentity: " $mat1 [transidentity]

set mat4T [transtranspose $mat4]
test_mat "Transpose 1: " $mat4T {
    {1 13.0 -1 10 }
    {2 13.1 -5  0 }
    {3 13.2 -9 -.5}
    {4 13.3 -8 10 }
}
test_mat "Transtranspose 2: " [transtranspose $mat4T] $mat4

set mult1 [transmult $mat2 $mat2 $mat2 $mat2 $mat2 $mat2 $mat2 $mat2]
test_mat "Transmult 1: " $mult1 $mat1

set mult2 [transmult $mat4 $mat4]
test_mat "Transmult 2: " $mult2 {
    {64     13.2   0.4   46.6}
    {303.1 131.61 86.47 253.63}
    {-137  -22.5  16    -78.5 }
    {110.5  22.5  29.5  144   }
}


set temp1 "35 degrees deg west"
set find_angle1 [find_rotation_value temp1]
test "Find_rotation_value 1: value: " $find_angle1 0.610863
test "Find_rotation_value 1: remainder: " [string compare $temp1 "deg west"] 0
set temp2 "2 rad to nowhere"
set find_angle2 [find_rotation_value temp2]
test "Find_rotation_value 2: value: " $find_angle2 2
test "Find_rotation_value 2: remainder: " [
         string compare $temp2 "to nowhere"] 0

set rotmaty [transaxis y -10]
test_mat "Transaxis 1: " $mat3 $rotmaty
set rotmatx [transaxis x +15 deg]
test_mat "Transaxis 2: " $rotmatx {
    {1.0000 0.0000  0.0000 0}
    {0.0000 0.9659 -0.2588 0}
    {0.0000 0.2588  0.9659 0}
    {0      0       0      1}
}
set rotmatz [transaxis z -2 rad]
test_mat "Transaxis 3: " $rotmatz {
    {-0.4161  0.9093 0.0000 0}
    {-0.9093 -0.4161 0.0000 0}
    { 0.0000  0.0000 1.0000 0}
    { 0       0      0      1}
}

proc test_transvec {count axis dir} {
    if [string compare $dir "-"] {
	set name "Transvec $count: "
	set fctn transvec
    } else {
	set name "Transvecinv $count: "
	set fctn transvecinv
    }

    for {set i 0} {$i < 12} {incr i} {
	set angle [expr $i * 30]
	# make a rotation of ^x to the given angle
	set rotstd [transaxis $axis $dir$angle]
	# reverse it
	set rot [$fctn [vectrans $rotstd {1 0 0 0}]]
	# and test how it compares to where it was expected to be
	set result [test_compare [vectrans $rot {1 0 0 0}] [vectrans $rotstd {1 0 0 0}]]
	if [string compare $result Ok] {
	    return "$name loop $i: Bad"
	}
	return "$name Ok"
    }
}

puts [test_transvec 1 y +]
puts [test_transvec 2 z +]
puts [test_transvec 3 x +]

puts [test_transvec 1 y -]
puts [test_transvec 2 z -]
puts [test_transvec 3 x -]

# now some more complicated (multiple axes) ones

proc test_transvec2 {vector} {
    set transv1 [vecnorm $vector]
    set transm1 [transvec $transv1]
    return [test_compare $transv1 [coordtrans $transm1 {1 0 0}]]
}

proc test_transvec3 {} {
    if [string compare [test_transvec2 {1 2 3}] Ok] {return Bad1}
    if [string compare [test_transvec2 {-1 2 3}] Ok] {return Bad2}
    if [string compare [test_transvec2 {1 -2 3}] Ok] {return Bad3}
    if [string compare [test_transvec2 {1 2 -3}] Ok] {return Bad4}
    if [string compare [test_transvec2 {-1 -2 3}] Ok] {return Bad5}
    if [string compare [test_transvec2 {1 -2 -3}] Ok] {return Bad6}
    if [string compare [test_transvec2 {-1 2 -3}] Ok] {return Bad7}
    if [string compare [test_transvec2 {-1 -2 -3}] Ok] {return Bad8}
    return Ok
}
proc test_transvecinv2 {vector} {
    set transv1 [vecnorm $vector]
    set transm1 [transvecinv $transv1]
    return [test_compare {1 0 0} [coordtrans $transm1 $transv1]]
}

proc test_transvecinv3 {} {
    if [string compare [test_transvecinv2 {1 2 3}] Ok] {return Bad1}
    if [string compare [test_transvecinv2 {-1 2 3}] Ok] {return Bad2}
    if [string compare [test_transvecinv2 {1 -2 3}] Ok] {return Bad3}
    if [string compare [test_transvecinv2 {1 2 -3}] Ok] {return Bad4}
    if [string compare [test_transvecinv2 {-1 -2 3}] Ok] {return Bad5}
    if [string compare [test_transvecinv2 {1 -2 -3}] Ok] {return Bad6}
    if [string compare [test_transvecinv2 {-1 2 -3}] Ok] {return Bad7}
    if [string compare [test_transvecinv2 {-1 -2 -3}] Ok] {return Bad8}
    return Ok
}


puts [list Transvec 4: [test_transvec3]]
puts [list Transvecinv 4: [test_transvecinv3]]


set offset0 {.3 -7.8 65.43}
set offset1 [transoffset $offset0]
set offsetv2 [coordtrans $offset1 $v2]
test "Transoffset 1: " $offsetv2 {10.1 -.3 71.83}
set offset2 [transoffset [vecinvert $offset0]]
test_mat "Transoffset 2: " [transidentity] [transmult $offset1 $offset2]

#### testing 'trans_about'

set tabout1 [transabout {1 0 0} 52 deg]
test "Transabout 1: " [coordtrans $tabout1 $v2] [coordtrans [transaxis x 52 deg] $v2]

set tabout2 [transabout {0 1 0} 90 deg]
test "Transabout 2: " [coordtrans $tabout2 {0 0 1}] [coordtrans [transaxis y 90 deg] {0 0 1}]

set tabout3 [transabout {0 0 1} 28 deg]
test "Transabout 3: " [coordtrans $tabout3 $v2] [coordtrans [transaxis z 28 deg] $v2]


# along y-z plane
set vec4 {0 1 1}
# should take y to z
set rot4 [transabout $vec4 180 deg]
set normy {0 1 0}
set vec5 [vectrans $rot4 $normy]
test "Transabout 4: " $vec5 {0 0 1}

# When I wrote this test-package 
# I hadn't yet determined the name for this function
set t trans

# generate the identity matrix
set tmat1 [$t]
test_mat "$t 1: " $mat1 $tmat1

set tmat2 [$t axis z [expr $M_PI/4.0] rad]
test_mat "$t 2: " $mat2 $tmat2

set tmat3 [$t axis y -10 deg]
test_mat "$t 3: " $mat2 $tmat2

# use {2 3 4} as the origin
# rotate -10 about y
set offset4 {2 3 4}
set tvec4 {2 3 5}
set tmat4 [$t axis y -10 deg center $offset4]
test "$t 4: " [coordtrans $tmat4 $tvec4] [
	vecadd [vectrans [transaxis y -10 deg] [vecsub $tvec4 $offset4]] \
	$offset4]

set tmat5 [$t origin $offset4]
test_mat "$t 5: " $tmat5 [transoffset [vecinvert $offset4]]

set tmat6 [$t axis y -10 deg origin $offset4]
test_mat "$t 6: " $tmat6 [transmult [transaxis y -10 deg] \
	      [transoffset [vecinvert $offset4]] ]


set tmat7 [$t axis y -10 deg offset $offset4]
test "$t 7: " [coordtrans $tmat7 $tvec4] [
	  vecadd [vectrans [transaxis y -10 deg] $tvec4] $offset4
	 ]

#set tmat7 [$t axis y -10 deg center $offset4]
#test "$t 7: " [coordtrans $tmat7 $tvec4] [
#	vecadd [vectrans [transaxis y -10 deg] [vecsub $tvec4 $offset4]] \
	$offset4]
