/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


/*
 * NOTE: 
 *
 * All integer values are stored as big endian in class file.
 *    We thus have to use DataInputStream() to read the values.
 *
 * This class has a main(). You can test this class file parser by;
 *     java horb.horbc.ClassFile classfilename.class
 */

package horb.horbc.ClassFile;

import java.io.IOException;
import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.PrintStream;

/**
 * ClassFile parses a class file. When an instance is instantiated,
 * it parses a class file and keep info about the class file in it.
 * You can examine contents of the class file through public instance
 * variables of the instance. <p>
 * 
 * This class has a main(). You can test this class file parser by;
 * <pre>
 *     java horb.horbc.ClassFile classfilename.class
 * </pre>
 *
 * Otherwise, use this class like this;
 * <pre>
 *     ClassFile c = new ClassFile("horb.horbc.foo.class");
 *     c.print();		// print contents
 *     System.out.print(c.flag.toString()); // print flag part
 * </pre>
 * @version	$Version$
 *
 */
public class ClassFile {
  DataInputStream ins;
  String classFileName;

  short current_major_version = 45;
  boolean debug = false;

  /**
   * class file magic number. 0xCafeBabe
   */
  public int magic;

  /**
   * Minor version number of this class file.
   */
  public short minor_version;

  /**
   * Major version number of this class file.
   */
  public short major_version;

  /**
   * Number of items in the constant pool.
   */
  public short numConstantPool;

  /**
   * Constant pool. The string table.
   */
  public ConstantPool cp;

  /**
   * encoded form of access flags such as public, static...
   */
  public short access_flags;

  /**
   * Decoded form of access flag of this class.
   */
  public AccessFlag flag;

  /**
   * index of constant pool for the name of this class.
   */
  public short this_class_index;

  /**
   * String form of the name of this class  (package.class)
   */
  public String this_class;	// package.class

  /**
   * String form of the name of this package
   */
  public String packageName;	// package

  /**
   * String form of the name of this class (only class portion)
   */
  public String className;		// class

  /**
   * index of constant pool for the name of the super class.
   */
  public short super_class_index;

  /**
   * String form of the name of the super class. java.lang.Object is the default.
   */
  public String super_class;

  /**
   * true if this class has a superclass other than java.lang.Object.
   */
  public boolean hasSuperClass;

  /**
   * Number of interfaces this class has.
   */
  public short interfaces_count;

  /**
   * Indecies to constant pool of interfaces.
   */
  public short interfaces_index[];

  /**
   * String form of interfaces. In case of interface, this array includes
   * extends of the interface.
   */
  public String interfaces[];

  /**
   * Number of fields (class/instance variables) in this class.
   */
  public short fields_count;

  /**
   * Array of info about fields.
   */
  public FieldInfo fields[];

  /**
   * Number of methods in this class.
   */
  public short methods_count;

  /**
   * Array of info about methods.
   */
  public MethodInfo methods[];

  /**
   * Number of attributes this class has.
   */
  public short attributes_count;

  /**
   * Attributes of this class. <p>
   * 1) source file name
   */
  public AttributeInfo attributes[];

  /**
   * make an object while parse a class file.
   * @param file	filename of the class to be parsed
   * @exception 	ClassFileException If unknown file format.
   * @exception		IOException If I/O error occured.
   */
  public ClassFile(String file) throws ClassFileException, IOException {
    parseClassFile(file);
  }

  /**
   * parse a class file. (for test)<p>
   * <pre>
   *     usage:
   *       java horb.horbc.ClassFile classfilename.class
   * </pre>
   * @param file	filename of the class to be parsed
   */
  public static void main(String argv[]) {
    try {
      ClassFile c = new ClassFile(argv[0]);
      c.print();
    } catch (FileNotFoundException e) {
      System.err.println("file '"+argv[0]+"' not found");
      System.exit(1);
    } catch (IOException eio) {
      System.err.println("file '"+argv[0]+"' I/O error");
      System.exit(1);
    } catch (ClassFileException f) {
      System.err.println(f);
      System.exit(1);
    }      
  }

  /**
   * parse a class file.
   * @param file	filename of the class to be parsed
   * @exception 	ClassFileException If unknown file format.
   * @exception		IOException If I/O error occured.
   */
  public void parseClassFile(String file) throws IOException, ClassFileException {
    classFileName = file;
    ins = new DataInputStream(new FileInputStream(classFileName));
    magic = ins.readInt();
    if (magic != 0xcafebabe)
      throw(new ClassFileException("Invalid class file format"));
  
    minor_version = ins.readShort();
    major_version = ins.readShort();
    if (major_version != current_major_version)
      throw(new ClassFileException("Old classfile version "+major_version));
    if (debug)
      System.out.println("// compiled by javac version "+
			 major_version+"."+minor_version);
    
    numConstantPool = ins.readShort(); // constant pool count
    cp = new ConstantPool(ins, numConstantPool);

    if (debug)
      cp.print();

    access_flags = ins.readShort();
    flag = new AccessFlag(access_flags);
    if (debug)
      System.out.print(flag.toString());

    this_class_index = ins.readShort();
    this_class = cp.toString(this_class_index);
    this_class = this_class.replace('/', '.'); // package/class -> package.class
    int at;
    int len = this_class.length();
    if ((at = this_class.lastIndexOf('.')) == -1) { 
      className = this_class;	// no package
      packageName = "";
    } else {
      className = this_class.substring(at+1, len);
      packageName = this_class.substring(0, at);
    }

    if (debug) {
      System.out.println("package "+packageName+";");
      if (flag.isInterface())
	System.out.print("interface "+className);
      else
	System.out.print("class "+className);
    }

    super_class_index = ins.readShort();
    super_class = cp.toString(super_class_index);
    super_class = super_class.replace('/', '.'); // package/class -> package.class
    if (super_class.equals("java.lang.Object") == false)
      hasSuperClass = true;

    if (debug)
      System.out.print(" extends "+super_class);

    interfaces_count = ins.readShort();
    if (interfaces_count > 0) {
      interfaces_index = new short[interfaces_count];
      interfaces = new String[interfaces_count];
      if (debug)
	System.out.print(" implements ");
      for (int i = 0; i < interfaces_count; i++) {
	interfaces_index[i] = ins.readShort();
	interfaces[i] = cp.toString(interfaces_index[i]);
	interfaces[i] = interfaces[i].replace('/', '.'); // package/class -> package.class
	if (debug)
	  System.out.print(interfaces[i]);
      }
    }
    if (debug)
      System.out.println(" {");
    
    fields_count = ins.readShort();
    if (debug)
      System.out.println("  // num field = "+fields_count);
    if (fields_count > 0) {
      fields = new FieldInfo[fields_count];
      for (int i = 0; i < fields_count; i++) {
	try {
	  fields[i] = new FieldInfo(ins, cp);
	} catch (ClassFileException ef) {
	  throw new ClassFileException(ef.getMessage()+" in class/interface "+className);
	}
	if (debug)
	  fields[i].print(cp);
      }
      if (debug)
	System.out.println("");
    }

    methods_count = ins.readShort();
    if (debug)
      System.out.println("  // num method = "+methods_count);

    if (methods_count > 0) {
      methods = new MethodInfo[methods_count];
      for (int i = 0; i < methods_count; i++) {
	try {
	  methods[i] = new MethodInfo(ins, cp);
	} catch (ClassFileException e) {
	  throw new ClassFileException(e.getMessage()+" in class/interface "+className);
	}
	if (debug)
	  methods[i].print(cp);
      }
    }
    
    attributes_count = ins.readShort();
    if (debug)
      System.out.println("  // num attribute = "+attributes_count);
    if (attributes_count > 0) {
      attributes = new AttributeInfo[attributes_count];
      for (int i = 0; i < attributes_count; i++) {
	attributes[i] = new AttributeInfo(ins, cp);
	if (debug)
	  attributes[i].print(cp);
      }
    }
    if (debug)
      System.out.print("}");

    ins.close();
  }

  /**
   * print contents of the class file. Call this like;
   * <pre>
   *     ClassFile c = new ClassFile("horb.horbc.foo.class");
   *     c.print();		// print contents
   * </pre>
   */
  public void print() {
    //
    // print version number and attributes
    //
    System.out.println("// compiled by javac version "+
		       major_version+"."+minor_version);
    if (attributes_count > 0) {
      for (int i = 0; i < attributes_count; i++) {
	System.out.print("// ");
	attributes[i].print(cp);
      }
    }

    System.out.println("package "+packageName+";");
    //
    // class foo extends super implements intrf1, intrf2 {
    //
    if (flag.isInterface())
      System.out.print("interface "+className);
    else
      System.out.print("class "+className);
    System.out.print(" extends "+super_class);
    if (interfaces_count > 0) {
      System.out.print(" implements ");
      for (int i = 0; i < interfaces_count; i++) {
	System.out.print(interfaces[i]);
	if (i < interfaces_count-1)
	  System.out.print(", ");
      }
    }
    System.out.println(" {");

    //
    // print class/instance variables
    //
    if (fields_count > 0) {
      for (int i = 0; i < fields_count; i++) {
	fields[i].print(cp);
      }
      System.out.println("");
    }

    //
    // print methods
    //
    if (methods_count > 0) {
      for (int i = 0; i < methods_count; i++) {
	methods[i].print(cp);
      }
    }

    System.out.println("}");

  }
}

