/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.horbc.ClassFile;

import java.io.DataInputStream;
import java.io.IOException;

/**
 * Parse and print constant pool. 
 * This file also includes classes defining entries of 
 *		constant pool.<p>
 *
 * NOTE: All integer values are stored as big endian in class file.
 *	We thus have to use DataInputStream() to read the values.<p>
 *
 * NOTE2: The ducument says that Constant Pool[0] is never used by
 *	convention. However, the item is used in actual class files.
 */
class ConstantPool {
  static final byte CONSTANT_Unknown = 0;
  static final byte CONSTANT_Utf8 = 1;
  static final byte CONSTANT_Unicode = 2;
  static final byte CONSTANT_Integer = 3;
  static final byte CONSTANT_Float = 4;
  static final byte CONSTANT_Long = 5;
  static final byte CONSTANT_Double = 6;
  static final byte CONSTANT_Class = 7;
  static final byte CONSTANT_String = 8;
  static final byte CONSTANT_Fieldref = 9;
  static final byte CONSTANT_Methodref = 10;
  static final byte CONSTANT_InterfaceMethodref = 11;
  static final byte CONSTANT_NameAndType = 12;

  protected CPEntry items[];
  protected int numItem;
  public int ConstantValue_index = 0;
  public int Code_index = 0;
  public int SourceFile_index = 0;
  public int LineNumberTable_index = 0;
  public int LocalVariableTable_index = 0;
  public int Exceptions_index = 0;

  ConstantPool(DataInputStream ins, int numItem) throws IOException {
    this.numItem = numItem;
    items = new CPEntry[numItem];
//    System.err.println("num items in constant pool = " + numItem);

    items[0] = new CONSTANT_Unknown_info(ins);

    for (int i = 1; i < numItem; i++) {
      byte tag = ins.readByte();
//      System.err.println("cp["+i+"] tag "+tag);

      /*
       * why can't I write this as;
       * 	switch { case CONSTANT_Utf8: ... }
       */
      switch (tag) {
      case CONSTANT_Utf8:
	items[i] = new CONSTANT_Utf8_info(ins);
	break;
      case CONSTANT_Unicode:
	items[i] = new CONSTANT_Unicode_info(ins);
	break;
      case CONSTANT_Integer:
	items[i] = new CONSTANT_Integer_info(ins); 
	break;
      case CONSTANT_Float:
	items[i] = new CONSTANT_Float_info(ins); 
	break;
      case CONSTANT_Long:
	items[i] = new CONSTANT_Long_info(ins); 
	i++;			// long takes two spots in the CP
	break;
      case CONSTANT_Double:
	items[i] = new CONSTANT_Double_info(ins); 
	i++;			// double takes two spots in the CP
	break;
      case CONSTANT_Class:
	items[i] = new CONSTANT_Class_info(ins); 
	break;
      case CONSTANT_String:
	items[i] = new CONSTANT_String_info(ins); 
	break;
      case CONSTANT_Fieldref:
	items[i] = new CONSTANT_Fieldref_info(ins); 
	break;
      case CONSTANT_Methodref:
	items[i] = new CONSTANT_Methodref_info(ins); 
	break;
      case CONSTANT_InterfaceMethodref:
	items[i] = new CONSTANT_InterfaceMethodref_info(ins); 
	break;
      case CONSTANT_NameAndType:
	items[i] = new CONSTANT_NameAndType_info(ins); 
	break;
      default:
	items[i] = new CONSTANT_Unknown_info(ins); 
	items[i].tag = tag;
	System.err.println("Warning: Unknown tag "+tag+" in constant pool");
	break;
      }
    }

    //
    // register often used strings
    //
    ConstantValue_index = lookupString("ConstantValue");
    Code_index = lookupString("Code");
    SourceFile_index = lookupString("SourceFile");
    LineNumberTable_index = lookupString("LineNumberTable");
    LocalVariableTable_index = lookupString("LocalVariableTable");
    Exceptions_index = lookupString("Exceptions");
  }

  public int lookupString(String s) {
    for (int i = 1; i < numItem; i++) {
//      System.out.println("cp["+i+"]:");
      if (items[i] != null && items[i].tag == CONSTANT_Utf8) {
	CONSTANT_Utf8_info item = (CONSTANT_Utf8_info)items[i];
//	System.out.println("cp["+i+"]: "+item);
	if (s.compareTo(item.value) == 0)
	  return(i);
      }
    }
    return 0;
  }

  public String toString(int index) {
    try {
      return items[index].toString(this);
    } catch (Exception e) {
      return "ConstantPool::print - "+e.toString();
    }
  }

  public void print(int index) {
    try {
      if (index == 0)
	System.out.print("cp[0]");
      else
	items[index].print(this);
    } catch (Exception e) {
      System.err.println("ConstantPool::print - "+e.toString());
    }
  }

  public void print() {
    System.out.println("Constant Pool: "+(numItem-1)+" entries");
    for (int i = 1; i < numItem; i++) {
      System.out.print("cp["+i+"] ");
      items[i].print(this);
    }
  }
}


abstract class CPEntry {
  public byte     tag;
  public abstract String toString(ConstantPool cp);
  public void print(ConstantPool cp) {
    System.out.println(tag+"  "+toString(cp));
  }
}

class CONSTANT_Unknown_info extends CPEntry {

  CONSTANT_Unknown_info(DataInputStream ins) {
    tag = ConstantPool.CONSTANT_Unknown;
  }

  public String toString(ConstantPool cp) {
    return("<UNKNOWN TAG>");
  }
}

class CONSTANT_Utf8_info extends CPEntry {
  public short	length;
  public String  value;

  CONSTANT_Utf8_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Utf8;
    length = ins.readShort();
    byte buf[] = new byte[length];
    ins.readFully(buf);
    value = new String(buf, 0);
//    System.err.println("CONSTANT_Utf8: "+value);
  }

  public String toString(ConstantPool cp) {
    return(value);
  }
}

class CONSTANT_Unicode_info extends CPEntry {
  public short	length;
  public String value;

  CONSTANT_Unicode_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Unicode;
    length = ins.readShort();
    char buf[] = new char[length];
    for (int i = 0; i < length; i++) {
      buf[i] = ins.readChar();
    }
    value = new String(buf);
//    System.out.println("CONSTANT_Unicode: "+value);
  }

  public String toString(ConstantPool cp) {
    return(value);
  }
}

//Classes and Interfaces

class CONSTANT_Class_info extends CPEntry {
  public short	name_index;

  CONSTANT_Class_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Class;
    name_index = ins.readShort();
  }

  public String toString(ConstantPool cp) {
    return(cp.toString(name_index));
  }
}

// Fields and Methods

class CONSTANT_Fieldref_info extends CPEntry {
  public short	class_index;
  public short	name_and_type_index;

  CONSTANT_Fieldref_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Fieldref;
    class_index = ins.readShort();
    name_and_type_index = ins.readShort();
  }

  public String toString(ConstantPool cp) {
    return(cp.toString(class_index)+":"
	   +cp.toString(name_and_type_index));
  }
}

class CONSTANT_Methodref_info extends CPEntry {
  public short	class_index;
  public short	name_and_type_index;

  CONSTANT_Methodref_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Methodref;
    class_index = ins.readShort();
    name_and_type_index = ins.readShort();
  }

  public String toString(ConstantPool cp) {
    return(cp.toString(class_index)+":"
	   +cp.toString(name_and_type_index));
  }
}

class CONSTANT_InterfaceMethodref_info extends CPEntry {
  public short	class_index;
  public short	name_and_type_index;

  CONSTANT_InterfaceMethodref_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_InterfaceMethodref;
    class_index = ins.readShort();
    name_and_type_index = ins.readShort();
  }

  public String toString(ConstantPool cp) {
    return(cp.toString(class_index)+":"
	   +cp.toString(name_and_type_index));
  }
}

class CONSTANT_NameAndType_info extends CPEntry {
  public short	name_index;
  public short	signature_index;
  CONSTANT_NameAndType_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_NameAndType;
    name_index = ins.readShort();
    signature_index = ins.readShort();
  }

  public String toString(ConstantPool cp) {
    return(cp.toString(name_index)+"-"
	   +cp.toString(signature_index));
  }

}

// String Objects
class CONSTANT_String_info extends CPEntry {
  public short	string_index;	// pointer to UTB string

  CONSTANT_String_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_String;
    string_index = ins.readShort(); // string index
  }

  public String toString(ConstantPool cp) {
    return(cp.toString(string_index));
  }
}

// Numeric Constants

class CONSTANT_Integer_info extends CPEntry {
  public int	value;

  CONSTANT_Integer_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Integer;
    value = ins.readInt();
  }

  public String toString(ConstantPool cp) {
    return(""+value);
  }
}

class CONSTANT_Float_info extends CPEntry {
  public float	value;
  CONSTANT_Float_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Float;
    value = ins.readFloat();
  }
  public String toString(ConstantPool cp) {
    return(""+value);
  }
}

//All eight-byte constants take up two spots in the constant pool. 
// If this is the nth item in the constant pool, then the next item will be 
//numbered n+2.
class CONSTANT_Long_info extends CPEntry {
  public long	value;

  CONSTANT_Long_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Long;
    value = ins.readLong();
  }

  public String toString(ConstantPool cp) {
    return(""+value);
  }
}

//All eight-byte constants take up two spots in the constant pool. 
// If this is the nth item in the constant pool, then the next item will be 
//numbered n+2.
class CONSTANT_Double_info extends CPEntry {
  public double   value;

  CONSTANT_Double_info(DataInputStream ins) throws IOException {
    tag = ConstantPool.CONSTANT_Double;
    value = ins.readDouble();
  }

  public String toString(ConstantPool cp) {
    return(""+value);
  }
}

