/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.horbc;

import horb.horbc.ClassFile.*;
import java.io.PrintStream;
import java.util.*;

/**
 * Generate Proxy class (client stab), i.e., "class_Proxy.java"
 */
class Proxy {
  /**
   * methodNo starts from this number. range 0-199 is reserved by me.
   * You can use range 200-999.
   */
  static int initialMethodNo = 1000;

  short major;			// major version number of the proxy
  short minor;			// minor version number of the proxy
  ClassFile cf;
  PrintStream os;
  Sender send;
  Receiver recv;
  int methodNo = Proxy.initialMethodNo;
  String superProxy;
  boolean debug = false;
  boolean objectCopierOnly = false;
  boolean methodSync = true;
  String sync = "";
  Vector methodName;		// grownable array to store method names
  boolean hasReviveHook = false;
  boolean hasFreezeHook = false;
  String asyncMethodName;
  boolean hasAsyncMethod;

  /**
   * Generate proxy class (client stub).
   * @param   cf	class file object
   * @param   os        output stream for "class_Proxy.java"
   * @param   objectCopierOnly    if true, the contents of the proxy is just a object copyer.
   */
  Proxy(ClassFile cf, PrintStream os, boolean objectCopierOnly, short major, short minor, boolean methodSync) {
    this.cf = cf;
    this.os = os;
    this.objectCopierOnly = objectCopierOnly;

    // as default, all methods are synchronous so that some threads can share
    // a proxy. This is independent from synchronousness of server methods.
    this.methodSync = methodSync;
    if (methodSync)
      sync = "synchronized ";

    this.major = major;
    this.minor = minor;
    send = new Sender(cf, os);
    recv = new Receiver(cf, os);
    methodName = new Vector(10, 10); // initial 10, increment 10
  }

  Vector generate() {
    prolog();

    methods();
    if (cf.flag.isInterface())
      genImplConstructor();
    genConstructor();

    // a class needs object sender/receiver
    // not need for interface
    ObjectSender objectSender = new ObjectSender();
    objectSender.objectSender(cf, os, send, hasFreezeHook);

    ObjectReceiver objectReceiver = new ObjectReceiver();
    objectReceiver.objectReceiver(cf, os, recv, hasReviveHook);
    epilog();
    return methodName;
  }

  private void prolog() {

    // remember it is an interface or a class
    putln("// HORB");
    putln("// Class: "+cf.this_class+"_Proxy");
    putln("// Version: "+major+"."+minor);
    Date today = new Date();
    putln("// Date: "+today);
    putln("// Generator: HORBC Compiler "
	       + horbc.major_version +"."+ horbc.minor_version+"."+horbc.releaseLevel);
    putln("//");

    //
    // package and import
    //
    if (cf.packageName.equals("") == false)
      putln("package "+cf.packageName+";");
    putln("import horb.orb.*;");
    putln("import java.io.*;");
    putln();

    //
    // class name
    //
    String abst = "";
    if (cf.flag.isAbstract())
      abst = "abstract";
    put("public "+abst+" class "+cf.className+"_Proxy ");
    
    //
    // inheritance
    //
    if (cf.flag.isInterface()) {		// interface
      if (horbc.ignoreSuper == false && cf.hasSuperClass) {
	superProxy = cf.super_class+"_Proxy";
	put("extends "+ superProxy);
      }
    } else {			// class
      if (horbc.ignoreSuper == false && cf.hasSuperClass) {
	superProxy = cf.super_class+"_Proxy";
	put("extends "+ superProxy);
      }
    }
      
    //
    // implements
    //
    put(" implements Proxy ");
    for (int i = 0; i < cf.interfaces_count; i++) {
      put(", ");
      put(cf.interfaces[i]);
    }
    // if this proxy is for an interface, the Skeleton implements the interface
    if (cf.flag.isInterface())
      put(", "+cf.className);
    putln(" {");

    //
    // proxy's variables, Proxy interface (getIOCI(), setIOCI())
    //

    // class No in the class hierarchy. This will be set in constructor.
    putln("  protected short classNo;");
    // variables, accessors
    putln("  public final static short major = "+major+";");
    putln("  public final static short minor = "+minor+";");
    putln("  public short _getMajorVersion() { return major;}");
    putln("  public short _getMinorVersion() { return minor;}");
    putln("  HorbURL url;");
    putln("  private IOCI ioci;");
    putln("  public IOCI _getIOCI() { return ioci; }");
    putln("  public void _setIOCI(IOCI ioci) { this.ioci = ioci; }");
    putln("  public HorbURL _getObjectURL() { return url; }");
    putln("  public String _getThreadName() { return ioci.getThreadName(); }");
  }

  /**
   * note: all methods of an interface are marked as abstract.
   */
  private void methods() {
    boolean async;
    for (int methNo = 0; methNo < cf.methods_count; methNo++) {
      MethodInfo mi = cf.methods[methNo];
      oneMethod(mi, false);
      if (mi.methodName.endsWith("_Async")
	  && (cf.flag.isInterface() || mi.flag.isAbstract() == false)
	  && mi.flag.isPrivate() == false) {  
	async = true;
	hasAsyncMethod = true;
	int len = mi.methodName.length();
	asyncMethodName = mi.methodName.substring(0, len-"_Async".length());
	oneMethod(mi, true);
	asyncReceive(mi);
	async = false;
      }
    }
  }

  /**
   * <pre>
   * Method types:
   *    <clinit>()  (static initializer)  ingore
   *    private method()               ignore
   *    abstract method()              generage signature
   *    method_Local()  (local method) generate null method
   *    _reviveHook()                  generate hook caller in _recvInstance()
   *    _freezeHook()                   generate hook caller in _sendInstance()
   *    <init>()  (initializer)        generate void classname()
   *    <init>()  (abstract class)     ignore
   *    method_Async (asynchronous method) generate method_Request
   *    finalize() (finalizer)         rename to _finalize()
   * </pre>
   */
  private void oneMethod(MethodInfo mi, boolean async) {
    int numArg = mi.sig.numArg;
    boolean isConstructor = false;
    boolean isLocal = false;

    if (mi.methodName.equals("<clinit>")) // static initializer
      // ignore static initializer, i.e.,
      //    class foo { static { ... }}
      return;
    else if (mi.flag.isPrivate())
      return;
    else if (mi.methodName.equals("<init>")) {  // constructor
      isConstructor = true;
      if (cf.flag.isAbstract())
	return;
    } else if (mi.methodName.endsWith("_Local")) // local method
      isLocal = true;

    if (mi.methodName.equals("_reviveHook")) // has hook
      hasReviveHook = true;
    if (mi.methodName.equals("_freezeHook")) // has hook
      hasFreezeHook = true;

    put("  "+AccessFlagToString(mi.flag));	// public, abstract
    if (isConstructor) {			// constructor
      put(sync+"void "+cf.className+"(");
    } else if (async) {			    // asynchronous method
      put(sync+"void "+asyncMethodName+"_Request(");
    } else {
      // void foo(
      put(sync+mi.sig.toStringReturnType()+" ");
      if (mi.methodName.equals("finalize")) { // finalizer
	System.err.println("Warning: "+cf.className+".finalize() is renamed to _finalize().");
	put("_finalize(");
      } else			// usual method
	put(mi.methodName+"(");
    }

    // int arg0, char arg1[], ...)
    for (int a = 0; a < mi.sig.numArg; a++) {
      put(mi.sig.args[a].typeName);
      put(" arg"+a);
      for (int d = 0; d < mi.sig.args[a].arrayDim; d++)
	put("[]");
      if (a < mi.sig.numArg-1)
	put(", ");
    }
    put(") ");

    // put throws clause
    putThrowsClause(mi, isLocal);

    // put method body
    if (cf.flag.isInterface() == false && mi.flag.isAbstract()) {
      putln(";");		// abstract method
      return;
    } else if (isLocal) {	// local method
      //  fake return value to satisfy interface
      if (mi.sig.retType.type == JavaType.T_VOID)
	putln(" {}");
      else if (mi.sig.retType.arrayDim > 0
	       || mi.sig.retType.type == JavaType.T_CLASS)
	putln(" { return null; }");
      else if (mi.sig.retType.type == JavaType.T_BOOL)
	putln(" { return false; }");
      else
	putln(" { return 0; }");
      return;
    } else {			// not abstract, not local
      putln(" {");
    }

    //
    // generate stub
    //

    // grab the current ioci from 'this' object, since we can't
    // use ioci variable in the super classes
    putln("    IOCI io = _getIOCI();"); 
    putln("    try {");		// } catch IOException

    // io.selectMethod(int classNo, int methodNo, int numArg);
    if (async == false) {
      putln("    io.selectMethod(classNo, "+methodNo+", "+numArg+");");
      methodNo++;
    } else {
      putln("    if (inProgress)");
      putln("       throw new HORBException(\"async method in progress\");");
      putln("    io.selectMethod(classNo, "+(methodNo-1)+", "+numArg+");");
    }

    // remember the name for generating server side
    methodName.addElement(mi.methodName); 

    // send arguments
    for (int argNo = 0; argNo < mi.sig.numArg; argNo++)
      send.send("arg"+argNo, mi.sig.args[argNo], true);
    
    // kick arguments to start the remote method
    putln("    io.request();");

    if (async) {
      putln("    io.startAsyncHandler();");
      putln("    inProgress = true;");
    } else {
      recvStatus(mi);
      if (isConstructor == false)
	recvReturn(mi);		// no return value for constructors
    }
    putln("    } catch (IOException ioe) {");
    putln("      throw new NetException(ioe);");
    putln("    }");
    putln("  }");
  }

  private void putThrowsClause(MethodInfo mi, boolean isLocal) {
    boolean putThrows = false;
    boolean putHORBException = false;
    boolean putIOException = false;

    if (mi.exceptions != null && mi.exceptions.value != null) {
      ExceptionAttribute ea = (ExceptionAttribute)mi.exceptions.value;
      for (int i = 0; i < ea.numExceptions; i++) {
	if (ea.exceptions[i] != null) {
	if (putThrows == false) {
	  put("throws ");
	  putThrows = true;
	} else
	  put(", ");
	put(ea.exceptions[i]);
/* 1.2.2 we don't need to add "throw HORBException and IOExcetion".
	if ("horb.orb.HORBException".equals(ea.exceptions[i]))
	  putHORBException = true;
	else if ("java.io.IOException".equals(ea.exceptions[i]))
	  putIOException = true;
*/
	}
      }
    }
/* 
    if (isLocal == false) {
      if (putThrows == false) {
	put("throws ");
	putThrows = true;
      } else if (putHORBException == false || putIOException == false)
	put(", ");
      if (putHORBException == false) {
	put("HORBException");
	if (putIOException == false)
	  put(", ");
      }
      if (putIOException == false)
	put("IOException ");
    }
*/
  }

  /**
   * generate method_Receive() for an asynchonous method.
   */
  private void asyncReceive(MethodInfo mi) {
    put("  "+AccessFlagToString(mi.flag));	// public
    put("synchronized "+mi.sig.toStringReturnType()+" ");
    putln(asyncMethodName+"_Receive(long timeout) throws HORBException, InterruptedException {");
    putln("    IOCI io = _getIOCI();"); 
    putln("    try {");		// for IOException
    putln("    io.waitReceive(timeout);");
    putln("    inProgress = false;");
    recvReturn(mi);		// no return value for constructors
    putln("    } catch (IOException ioe) {");
    putln("      throw new NetException(ioe);");
    putln("    }");
    putln("  }");
  }

  /**
   * generate constructor caller for interface
   */
  void genImplConstructor() {
    putln("  public "+sync+"void "+cf.className+"_Impl() throws HORBException {");
    putln("    IOCI io = _getIOCI();"); 
    putln("    try {");
    putln("      io.selectMethod(classNo, "+methodNo+", "+0+");");
    methodNo++;
    putln("      io.request();");
    putln("      short stat = io.recvStatus();");
    putln("    } catch (IOException ioe) {");
    putln("      throw new NetException(ioe);");
    putln("    }");
    putln("  }");
  }

  /**
   * Generate constructors.
   * In the constructors, we should set classNo.
   */
  private void genConstructor() {
    putln("  public "+cf.className+"_Proxy() {");
    if (horbc.ignoreSuper ||  cf.hasSuperClass == false) // top class
      putln("    classNo = 1;");
    else
      putln("    classNo = (short)(super.classNo + 1);");
    putln("  }");
    
    putln("  public "+cf.className+"_Proxy(HorbURL url) throws HORBException {");
    putln("    this();");	// to set classNo
    putln("    _connect(url, null, null);");
    putln("  }");
    putln("  public "+cf.className+"_Proxy(String surl) throws HORBException {");
    putln("    this();");	// to set classNo
    putln("    _connect(new HorbURL(surl), null, null);");
    putln("  }");
    putln("  public "+cf.className+"_Proxy(HorbURL url, String user, String passwd) throws HORBException {");
    putln("    this();");	// to set classNo
    putln("    _connect(url, user, passwd);");
    putln("  }");
    putln("  public synchronized void _connect(HorbURL url, String user, String passwd) throws HORBException {");
    putln("    try {");
    putln("      if (ioci == null)");
    putln("        ioci = (IOCI)HORB.getIOCIClass().newInstance();");
    putln("    } catch (Exception e) {");
    putln("      throw new HORBException(\"can't create IOCI\");");
    putln("    }");
    putln("    this.url = ioci.connect(url, \""+cf.this_class+"\", major, minor, user, passwd);");
    putln("  }");
    putln("  public void _release() { ioci.release(); }");
  }

  private void recvStatus(MethodInfo mi) {
    putln("    short stat = io.recvStatus();");
  }

  private void recvReturn(MethodInfo mi) {
    // TODO
    if (mi.sig.retType.type == JavaType.T_VOID) {
      return;
    }
    put("    "+mi.sig.toStringReturnType()+" retValue");
    if (mi.sig.retType.arrayDim > 0)
      putln(" = null;");
    else
      putln(";");
    recv.recv("retValue", mi.sig.retType, true);
    putln("    return retValue;");
  }

  String AccessFlagToString(AccessFlag flag) {
    String s = "";
// no private
// no protected
// no static
// no native
    if (flag.isProtected())
      s += "protected ";
    if (flag.isPublic())
      s += "public ";
    if (flag.isFinal())
      s += "final ";
//    if (flag.isSynchronized())
//      s += "synchronized ";
    if (flag.isThreadsafe())
      s += "threadsafe ";
    if (flag.isTransient())
      s += "transient ";
    if (cf.flag.isInterface() == false && flag.isAbstract())
      s += "abstract ";
    return s;
  }


  private void epilog() {
    putln("  public void _invite(HorbURL clientURL, String username, String passwd) throws HORBException {");
    putln("    _getIOCI().invite(url, clientURL, username, passwd);");
    putln("  }");
    putln("  public void _invited(IOCI ioci, HorbURL url, String username, String passwd) throws HORBException, IOException {");
    putln("    this.ioci = ioci;");
    putln("    this.url = url;");
    putln("    ioci.invited(\""+cf.this_class+"\", url, major, minor, username, passwd);");
    putln("  }");

    if (hasAsyncMethod) {
      putln("  protected boolean inProgress;");
      putln("  public boolean _available() {");
      putln("    return _getIOCI().isAsyncMethodEnd();");
      putln("  }");
      putln("  public synchronized boolean _wait(long timeout) throws InterruptedException {");
      putln("    return _getIOCI().waitNoReceive(timeout);");
      putln("  }");
      putln("  public void _setHandler(AsyncMethodHandler handler, int tag) {");
      putln("    _getIOCI().setHandler(handler, tag);");
      putln("  }");

    } else {			// no async methods
      putln("  public boolean _available() { return false; }");
      putln("  public synchronized boolean _wait(long timeout) throws InterruptedException { return false; }");
      putln("  public void _setHandler(AsyncMethodHandler handler, int tag) {}");
    }
    putln("}");
  }

  private final void putln(String s) {
    os.println(s);
  }

  private final void put(String s) {
    os.print(s);
  }

  private final void putln() {
    os.println();
  }
}
