/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.horbc;

import horb.horbc.ClassFile.*;
import java.io.PrintStream;
import java.util.*;

/**
 * Generate Skeleton Class (server stub).
 */
class Skeleton {
  short major;			// major version number of the proxy
  short minor;			// minor version number of the proxy
  ClassFile cf;
  PrintStream os;
  Sender send;
  Receiver recv;
  int methodNo = Proxy.initialMethodNo;
  boolean debug = false;
  private String superSkeleton;
  boolean hasThreadEndHook = false;

 /**
  * Generate Skeleton Class (server stub).
   * @param   cf	class file object
   * @param   os        output stream for "class_Skeleton.java"
   */
  Skeleton(ClassFile cf, PrintStream os, short major, short minor) {
    this.cf = cf;
    this.os = os;
    this.major = major;
    this.minor = minor;
    send = new Sender(cf, os);
    recv = new Receiver(cf, os);
  }

  void generate() {
    prolog();
    genConstructor();
    genObjectHolder();
    methods();
    if (cf.flag.isInterface())
      genImplConstructor();
    genDispatch();	// genDispatch() should be after methods()
    epilog();
  }

  private void prolog() {
    putln("// HORB");
    putln("// Class: "+cf.this_class+"_Skeleton");
    putln("// Version: "+major+"."+minor);
    Date today = new Date();
    putln("// Date: "+today);
    putln("// Generator: HORBC Compiler "
	       + horbc.major_version +"."+ horbc.minor_version+"."+horbc.releaseLevel);
    putln("//");
    if (cf.packageName.equals("") == false)
      putln("package "+cf.packageName+";");
    putln("import horb.orb.*;");
    putln("import java.io.*;");
    putln();
    String defaultSuper = "java.lang.Object";
    put("public class "+cf.className+"_Skeleton ");
    if (horbc.ignoreSuper == false && cf.hasSuperClass) {
      superSkeleton = cf.super_class+"_Skeleton";
      put("extends "+ superSkeleton);
    }
    
    put(" implements Skeleton ");
/*
    for (int i = 0; i < cf.interfaces_count; i++) {
      put(", ");
      put(cf.interfaces[i]);
    }

    // if this skeleton is for an interface, the skeleton implements the interface
    if (cf.flag.isInterface())
      put(", "+cf.className);
*/

    putln(" {");

    putln("  public final static short major = "+major+";");
    putln("  public final static short minor = "+minor+";");
    putln("  public short getMajorVersion() { return major;}");
    putln("  public short getMinorVersion() { return minor;}");
    // class No in the class hierarchy
    putln("  protected short classNo;");
    // variables, accessors
    putln("  private boolean constructorFail;");
  }

  /**
   * object place holder and its accessors.
   */
  private void genObjectHolder() {
    putln("  "+cf.className+" o;");

    putln("  public Object getObject() throws ClassNotFoundException, InstantiationException, IllegalAccessException {");
    putln("    if (o == null)");
    putln("      constructor();");
    putln("    return o;");
    putln("  }");

    putln("  public void setObject(Object o) throws HORBException {");
    putln("    if (this.o == null)");
    putln("      this.o = ("+cf.className+")o;");
    putln("  }");

  }

  /**
   * Generate constructors.
   * In the constructors, we have to initialize classNo that keeps 
   * the position in the class hierarchy.
   * Note that I wanted to write;
   *   protected short classNo = (short)(package.superClass.classNo + 1);
   * However, javac doesn't allow to use fully qualified class name
   * in the field initializer.
   */
  private void genConstructor() {
    putln("  public "+cf.className+"_Skeleton() {");
    if (horbc.ignoreSuper || cf.hasSuperClass == false) // top class
      putln("    classNo = 1;");
    else
      putln("    classNo = (short)(super.classNo + 1);");
    putln("  }");
  }

  private void genDispatch() {
    // dispatch returns false if constructor fails
    putln("  public boolean dispatch(IOCI ioci, short classNo, short methodNo) throws IOException {");

    // method searcher for remote inheritance
    if (horbc.ignoreSuper == false && cf.hasSuperClass) {
      // if we have super classes
      putln("    if (classNo < this.classNo) {");
      putln("      return super.dispatch(ioci, classNo, methodNo);");
      putln("    }");
    }
    
    putln("    constructorFail = false;");
    putln("    switch (methodNo) {");
    for (int i = Proxy.initialMethodNo; i < methodNo; i++) {
      putln("    case "+i+":");
      putln("      stub"+i+"(ioci); break;");
    }
    putln("    default:");
    putln("      ioci.sendStatus(ioci.STAT_NO_SUCH_METHOD);");
    putln("      ioci.kick();");
    putln("      break;");
    putln("    }");
    putln("    return constructorFail;");
    putln("  }");
  }

  /**
   * Generate epilog of a skeleton class.
   * We use newInstance instead of new so that this interface may not
   * have the class Foo_Impl.
   */
  private void epilog() {
    putln("  protected void constructor() throws ClassNotFoundException, InstantiationException, IllegalAccessException {");
    putln("    constructorFail = true;");
    if (cf.flag.isInterface()) {
      // this is not really used. genImplConstructor() is used.
      putln("      o = ("+cf.className+")Class.forName(\""+cf.this_class+"_Impl\").newInstance();");
//      putln("      o = ("+cf.className+") new "+cf.this_class+"_Impl();");
    } else			// usual case
      if (cf.flag.isAbstract() == false)
	putln("    o = new "+cf.this_class+"();");
    putln("    constructorFail = false;");
    putln("  }");

    // run finalize() method of the HORB object for this thread.
    // the finalize() is called once again the HORB object is GCed.
    putln("  public synchronized void threadEndHook() {");
    if (hasThreadEndHook) {
      putln("    try {");
      putln("      if (o != null)");
      putln("        o._threadEndHook();");
      putln("    } catch (Throwable t) {}");
    }
    putln("  }");
    putln("  Proxy invitor;");
    putln("  public synchronized void invited(Proxy proxy) {");
    putln("    invitor = proxy;");
    putln("    notify();");
    putln("  }");
    putln("  public synchronized Proxy accept(long millis) throws InterruptedException {");
    putln("    Proxy invitor = this.invitor;");
    putln("    if (invitor != null) {");
    putln("      this.invitor = null;");
    putln("      return invitor;");
    putln("    }");
    putln("    wait(millis);");
    putln("    invitor = this.invitor;");
    putln("    this.invitor = null;");
    putln("    return invitor;");
    putln("  }");
    putln("  public synchronized boolean isInvited() {");
    putln("    return (invitor != null);");
    putln("  }");

    putln("}");
  }

  private void methods() {
    for (int methNo = 0; methNo < cf.methods_count; methNo++)
      oneMethod(cf.methods[methNo]);
  }

  /**
   * <pre>
   * method types:
   *    <clinit>() (static initializer) ignore
   *    private method() 		ignore
   *    abstract method()               ignore
   *    method_Local() (local method)   ignore
   *    <init>()   (initializer)        create a new instance
   *    <init>()  (abstract class)      ignore
   * </pre>
   */
  private void oneMethod(MethodInfo mi) {

    int numArg = mi.sig.numArg;
    boolean isConstructor = false;
    String retType = mi.sig.toStringReturnType();
    boolean voidRetType = (mi.sig.retType.type == JavaType.T_VOID);

    if (mi.methodName.equals("<clinit>")) // static initializer
      // ignore static initializer, i.e.,
      //    class foo { static { ... }}
      return;
    else if (mi.flag.isPrivate())             // private method
      return;
    else if (cf.flag.isInterface() == false && mi.flag.isAbstract())
      return;			  	      // abstract method
    else if (mi.methodName.endsWith("_Local")) // local method
      return;

/*
    //
    // sanity check by comparing method names with names generated
    // in client side
    //
    if (mi.methodName.equals(methodName.elementAt(methodNo)) == false) {
      horbc.err("Sanity check! method name mismatch. GenClient: "
			 +methodName.elementAt(methodNo)+" Skeleton: "
			 +mi.methodName);
      // never returns here
    }
*/

    //
    // method stub
    //
    if (mi.methodName.equals("<init>")) { // constructor
      isConstructor = true;
      if (cf.flag.isAbstract())
	return;
    }
    if (mi.methodName.equals("_threadEndHook")) { // thread end hook
      hasThreadEndHook = true;
    }

    putln("  private void stub"+methodNo+"(IOCI io) throws IOException {");
    methodNo++;

    //
    // generate stub
    //

    if (isConstructor)
      putln("    constructorFail = true;"); // default fail
    putln("    short stat = io.STAT_EXCEPTION_IN_ARGUMENT;");
    putln("    try {");
    putln("      int numArgs = io.getNumArgs();");
    

    // recv arguments
    for (int argNo = 0; argNo < mi.sig.numArg; argNo++) {
      put("      "+mi.sig.args[argNo].typeName);
      for (int d = 0; d < mi.sig.args[argNo].arrayDim; d++)
	put("[]");
      put(" arg"+argNo);
      if (mi.sig.args[argNo].arrayDim > 0)
	putln(" = null;");
      else
	putln(";");
      recv.recv(" arg"+argNo, mi.sig.args[argNo], true);
    }

    putln("      stat = io.STAT_EXCEPTION_IN_METHOD;");

    if (isConstructor) {
      // create an object by a constructor
      if (cf.flag.isInterface()) {
	// this is not really used. genImplConstructor() is used.
	putln("      o = ("+cf.className+") new "+cf.this_class+"_Impl(");
      } else			// usual case
	put("      o = new "+cf.className+"(");
    } else {
      // call a method of the remote object
      if (voidRetType)
	put("      (("+cf.className+")(getObject()))."+mi.methodName+"(");
      else
	put("      "+retType+" retValue = (("
	       +cf.className+")(getObject()))."+mi.methodName+"(");
    }
    for (int argNo = 0; argNo < mi.sig.numArg; argNo++) {
      put("arg"+argNo);
      if (argNo < mi.sig.numArg-1)
	put(", ");
    }
    putln(");");
    putln("      stat = io.STAT_NO_ERROR;");

    // catch exception during returning the result
    putln("      try {");
    putln("        io.sendStatus(stat);");
    if (voidRetType == false && isConstructor == false)
      send.send("retValue", mi.sig.retType, true);
    putln("        io.kick();");
    if (isConstructor)
      putln("        constructorFail = false;");
    putln("      } catch (Exception e) { e.printStackTrace();}");

    // catch exception during the execution of the method or catching arguments
    putln("    } catch (Exception f) {");
    putln("      if (HORBServer.debugging()) {");
//    putln("        System.err.println(f);");
    putln("        f.printStackTrace();");
    putln("      }");
    // TODO: resync here
    putln("      io.sendStatus(stat);");
    putln("//      if (stat == io.STAT_EXCEPTION_IN_METHOD || stat == io.STAT_EXCEPTION_IN_ARGUMENT)");
    putln("        io.sendString(f.toString());");
    putln("      io.kick();");
    putln("    }");

    putln("  }");
  }

  /**
   * generate constructor caller for interface.
   * We use newInstance instead of new so that this interface may not
   * have the class Foo_Impl.
   */
  void genImplConstructor() {
    putln("  private void stub"+methodNo+"(IOCI io) throws IOException {");
    methodNo++;
    putln("    constructorFail = true;"); // default fail
    putln("    int numArgs = io.getNumArgs();");
    putln("    short stat = io.STAT_EXCEPTION_IN_METHOD;");
    putln("    try {");
    putln("      o = ("+cf.className+")Class.forName(\""+cf.this_class+"_Impl\").newInstance();");
    putln("      stat = io.STAT_NO_ERROR;");
    putln("      try {");
    putln("        io.sendStatus(stat);");
    putln("        io.kick();");
    putln("        constructorFail = false;");
    putln("      } catch (Exception e) {}");
    putln("    } catch (Exception f) {");
    putln("      if (HORBServer.debugging()) {");
    putln("        f.printStackTrace();");
    putln("      }");
    // TODO: resync here
    putln("      io.sendStatus(stat);");
    putln("      io.kick();");
    putln("    }");
    putln("  }");
  }

  private final void putln(String s) {
    os.println(s);
  }

  private final void put(String s) {
    os.print(s);
  }

  private final void putln() {
    os.println();
  }
}
