/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.horbc;

import horb.horbc.ClassFile.ClassFile;
import horb.horbc.ClassFile.ClassFileException;
import java.io.DataInputStream;
import java.io.PrintStream;
import java.io.InputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.File;
import java.util.Vector;
import java.util.Enumeration;
import sun.tools.javac.*;

/**
 * The Horbc compiler. <p>
 * This class generates a proxy class (client side stub) and a skeleton
 * class (server side stub) for a class or a interface.
 */
public class horbc {
  static int major_version = 1;
  static int minor_version = 3;
  static String releaseLevel = "b1";

  static boolean generateClientStubFlag = true;
  static boolean generateServerStubFlag = true;
  static boolean compileClientStub = true;
  static boolean compileServerStub = true;
  static boolean noproxy = false;
  static boolean ignoreSuper = false;
  static String classPath;
  static String classDir;
  static boolean deleteSource = false;
  static boolean warn = false;
  static boolean debug = false;
  static boolean optim = false;
  static boolean debugProxy = false;
  static boolean verboseProxy = false;
  static boolean methodSync = true;
  static short major;
  static short minor;

  ClassFile cf;

  /**
   * Print error message and exit.
   */
  public static void err(String msg) {
    System.err.println(msg);
    System.exit(1);
  }

  /**
   * Entry point of proxy generator.
   * <pre>
   *    horbc [options] classnames...
   *  or
   *    java horb.horbc.horbc  [options] classname...
   * </pre>
   */
  public static void main(String argv[]) {

    boolean classFileShouldBeMoved = false;
    Vector sources = new Vector();

    // parse command line
    for (int i = 0; i < argv.length; i++) {
      if (argv[i].equals("-h") || argv[i].equals("-?") || argv[i].equals("-help")) {
	usage();
	System.exit(0);
      } else if (argv[i].equals("-d") || argv[i].equals("-dir")) {
	if ((i+1) < argv.length) {
	  i++;
	  classDir = argv[i];
	} else {
	  System.err.println("-d requires directory name");
	  System.exit(1);
	}
      } else if (argv[i].equals("-classpath")) {
	if ((i+1) < argv.length) {
	  i++;
	  classPath = argv[i];
	} else {
	  System.err.println("-classpath requires path");
	  System.exit(1);
	}
      } else if (argv[i].equals("-v") || argv[i].equals("-verbose")) {
	debug = true;
      } else if (argv[i].equals("-O")) {
	optim = true;
      } else if (argv[i].equals("-g")) {
	debugProxy = true;
      } else if (argv[i].equals("-verboseproxy")) {
	verboseProxy = true;
      } else if (argv[i].equals("-ignoresuper")) {
	ignoreSuper = true;
      } else if (argv[i].equals("-c")) {
	noproxy = true;
      } else if (argv[i].equals("-proxyonly")) {
	generateServerStubFlag = false;
      } else if (argv[i].equals("-skeletononly")) {
	generateClientStubFlag = false;
      } else if (argv[i].equals("-delete")) {
	deleteSource = true;
      } else if (argv[i].equals("-warn")) {
	warn = true;
      } else if (argv[i].equals("-nocompile")) {
	compileClientStub = false;
	compileServerStub = false;
      } else if (argv[i].equals("-nomethodsync")) {
	methodSync = false;
      } else if (argv[i].equals("-version")) {
	System.out.println("HORB Compiler Version "+major_version
			   +"."+minor_version+"."+releaseLevel);
	System.out.println("Copyright 1995, 1996 HIRANO Satoshi");
	System.exit(0);
      } else if (argv[i].equals("-major")) {
	if ((i+1) < argv.length) {
	  i++;
	  try {
	    major = (short)Integer.parseInt(argv[i]);
	  } catch (Exception e) {
	    System.err.println("-major num");
	    System.exit(1);
	  }
	} else {
	  System.err.println("-major num");
	  System.exit(1);
	}
      } else if (argv[i].equals("-minor")) {
	if ((i+1) < argv.length) {
	  i++;
	  try {
	    minor = (short)Integer.parseInt(argv[i]);
	  } catch (Exception e) {
	    System.err.println("-major num");
	    System.exit(1);
	  }
	} else {
	  System.err.println("-minor num");
	  System.exit(1);
	}
      } else if (argv[i].startsWith("-")) {
	System.err.println("invalid flag: " + argv[i]);
	System.exit(1);
      } else {
	sources.addElement(argv[i]);
      }
    }
    if (sources.size() == 0) {
      usage();
      System.exit(1);
    }

    //
    // generate proxy one by one
    //
    for (Enumeration e = sources.elements(); e.hasMoreElements();) {
      String source = (String)e.nextElement();
      //
      // if java source file, compile it
      //
      if (source.endsWith(".java")) {

	int n;
	if (noproxy == true) {
	  if ((n = compileIt(new File(source), true)) != 0)
	    System.exit(n);
	} else {
	  // generate class file in this directory. If -d option is specified,
	  // move the file to the proper directory later.
	  if ((n = compileIt(new File(source), false)) != 0)
	    System.exit(n);
	}
	source = source.substring(0, source.indexOf(".java"));
	classFileShouldBeMoved = true;
      }
      //
      // generate Proxy class and Skeleton class from the source class
      //
      if (noproxy == false) {
	horbc i = new horbc();
	i.generate(source, classFileShouldBeMoved);
      }
    }
  }

  /**
   * Generate proxy and skeleton class file.
   */
  void generate(String className, boolean classFileShouldBeMoved) {

    if (className.endsWith("_Proxy") || className.endsWith("_Skeleton")) {
      System.err.println("Further proxy and skeleton classes are not needed for a proxy or skeleton class "+className+".");
      return;
    }

    // store paths in classPath and sourcePath
    ClassPath cp = new ClassPath(className, classPath);
    if (cp.exists() == false) {
      System.err.println("class "+className+".class not found");
      System.exit(1);
    }

    //
    // parse class file
    //
    if (debug)
      System.err.println("parsing class file");
    try {
      cf = new ClassFile(cp.getFullPath());
    } catch (Exception eio) {
      System.err.println("Can't find "+className+" "+eio);
      System.exit(1);
    }

    // if destination directory is specified, move class file to there
    if (classFileShouldBeMoved && classDir != null)	// horbc -d <dir>
      moveClassFile(cf, cp);

    // generage client and server stubs
    if (generateClientStubFlag)
       generateClientStub(cf, cp);
    if (generateServerStubFlag)
      generateServerStub(cf, cp);
  }

  /**
   * generate a stub file "foo_Proxy.java" for client side
   */
  void generateClientStub(ClassFile cf, ClassPath cp) {
    PrintStream outs = null;
    File clientStubFile;
    if (cp.getDirectory().equals("."))
      clientStubFile = new File(cp.getClassName()+"_Proxy.java");
    else
      clientStubFile = new File(cp.getDirectory()+File.separator
	+cp.getClassName()+"_Proxy.java");

    System.err.println("\tgenerating "+clientStubFile);
    try {
      outs = new PrintStream(new FileOutputStream(clientStubFile));
    } catch (IOException outse)  {
      System.err.println("Can't create file '"+clientStubFile+": "+outse);
      System.exit(1);
    }

    Proxy gc = new Proxy(cf, outs, false, major, minor, methodSync);
    gc.generate();
    outs.close();
    if (compileClientStub) {
      int n;
      if ((n = compileIt(clientStubFile, true)) != 0) {
	System.exit(n);
      }
    }
    if (deleteSource)
      clientStubFile.delete();
  }

  /**
   * generate a stub file "foo_Skeleton.java" for server side
   */
  void generateServerStub(ClassFile cf, ClassPath cp) {
    PrintStream outs = null;
    File serverStubFile;
    if (cp.getDirectory().equals("."))
      serverStubFile = new File(cp.getClassName()+"_Skeleton.java");
    else
      serverStubFile = new File(cp.getDirectory()+File.separator
	+cp.getClassName()+"_Skeleton.java");

    System.err.println("\tgenerating "+serverStubFile);
    try {
      outs = new PrintStream(new FileOutputStream(serverStubFile));
    } catch (IOException gse)  {
      System.err.println("Can't create file '"+serverStubFile+": "+gse);
      System.exit(1);
    }

    Skeleton gs = new Skeleton(cf, outs, major, minor);
    gs.generate();
    outs.close();
    if (compileServerStub) {
      int n;
      if ((n = compileIt(serverStubFile, true)) != 0) {
	System.exit(n);
      }
    }
    if (deleteSource)
      serverStubFile.delete();
  }

  static void usage() {
    System.err.println("usage: horbc [options] source.java... class...");
    System.err.println("options:");
    System.err.println("    -? -h -help   show this message");
    System.err.println("    -v -verbose   verbose");
    System.err.println("    -version      show HORBC version");
    System.err.println("    -verboseproxy generate debug msg in Proxy");
    System.err.println("    -c            don't generate Proxy and Skeleton");
    System.err.println("    -major <num>  set major version number");
    System.err.println("    -minor <num>  set minor version number");
    System.err.println("    -ignoresuper  ignore super class");
    System.err.println("    -nomethodsync don't make methods synchronous");
    System.err.println("    -proxyonly    generate Proxy but not Skeleton");
    System.err.println("    -skeletononly generate Skeleton but not Proxy");
    System.err.println("    -nocompile    don't compile _Proxy.java and _Skeleton.java");
    System.err.println("    -delete       delete _Proxy.java and _Skeleton.java");
    System.err.println("    -O            optimizer On");
    System.err.println("    -g            generate debug symbols");
    System.err.println("    -warn         generate warning msg in javac");
    System.err.println("    -d <dir> | -dir <dir>  generate class file to dir");
    System.err.println("    -classpath <paths> set class file search path");
  }

  /**
   * Compile a class file by invoking javac.
   */
  static int compileIt(File file, boolean useClassDir) {
    int n;
    byte buf[] = new byte[512];
    Process proc = null;
    String option = " ";
    int status = 0;
    Vector vec = new Vector(5, 3);

    if (warn == false)
      vec.addElement("-nowarn");
    if (debugProxy)
      vec.addElement("-g");
    if (optim)
      vec.addElement("-O");
    if (useClassDir && classDir != null) {
      vec.addElement("-d");
      vec.addElement(classDir);
    }
    if (debug)
      vec.addElement("-verbose");

    vec.addElement(file.getPath());
    String argv[] = new String[vec.size()];
    int i = 0;
    for (Enumeration e = vec.elements(); e.hasMoreElements();)
      argv[i++] = (String)e.nextElement();

    System.err.println("\tcompiling "+file);
    try {
      sun.tools.javac.Main compiler = new sun.tools.javac.Main(System.out, "horbc+javac");
      status = compiler.compile(argv) ? 0 : 1;
    } catch (Exception ine) {
      System.err.println("Compilation error: "+ine);
      System.exit(1);
    }
    return status;
  }

  void moveClassFile(ClassFile cf, ClassPath cp) {
    if (classDir != null) {
      // create destination directory
      String packageDir = cf.packageName.replace('.', File.separatorChar);
      String dir = classDir+File.separator+packageDir;
      File destDir = new File(dir);
      if (destDir.isDirectory() == false)
	if (destDir.mkdirs() == false) {
	  System.err.println("Can't make dir "+destDir);
	  System.exit(1);
	}

      // move class file to there
      File srcFile = new File(cp.getFullPath());
      File srcDir = new File(srcFile.getParent());
      if (srcDir.equals(destDir) == false) {
	File destFile = new File(destDir+File.separator+cf.className+".class");
//	System.out.println("moving file from "+ srcFile + " to "+destFile);
	destFile.delete();
	if (srcFile.renameTo(destFile) == false) {
	  System.err.println("fail to move "+ srcFile + " to "+destFile);
	  System.exit(1);
	}
      }
    }
  }
}
