/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.io.*;
import java.util.*;

/**
 * Interface to HORB distributed access control list. You can specify hosts and
 * users that are allowed to access to a class or a method.
 *
 * <pre>
 * Syntax of ACL file
 *	name=acl_name
 *	className.host=hostname networkaddress domain...
 *	className.host_exclude=hostname networkaddress domain...
 *	className.user=username1 username2...
 *	className.user_exclude=username1 username2...
 *	className.creatable=false
 *	className2....
 *	className3....
 *	username1.password: lklaskjdf
 *	username2....
 *	username3....
 *
 * </pre>
 * Example:
 * <pre>
 *	name=etl_acl
 *	horb.orb.HORBAgent.host=bungo.etl.go.jp 192.31.99.23
 *	WClock.Test.host=etl.go.jp 192.31.*.*
 *	WCLock.Test.host_exclude=gate.etl.go.jp ftp.etl.go.jp
 *	WCLock.Test.user=hirano connelly guest
 *	WCLock.Test.user_exclude=hashimoto anonymous
 *	hirano.password=LKU&232ZC
 *	guest.password=*
 *	default.host=*
 *	default.host_exclude=gate.etl.go.jp
 *	default.user=*
 *	default.user_exclude=anonymous
 *
 *
 * </pre>
 * If ACL system is used in a HORB server, client must pass the following
 * access control.
 * <OL>
 * <LI>Client's host name or network address must be included in the class'
 * host include list or default host include list if class' host include
 * list is not exists.
 * <LI>Clien't host name or network address must not be included in the
 * class' host exclude list or default host exclude list if class' host
 * exclude list is not exists.
 * <LI>Client's username must be included in the class'
 * user include list or default user include list if class' user include
 * list is not exists.
 * <LI>Client's password must match with user's password.
 * <LI>Clien't username must not be included in the
 * class' user exclude list or default user exclude list if class' user
 * exclude list is not exists.
 * </OL>
 *
 * String matching is case insensitive. Thus Bungo.Etl.Go.Jp matches
 * with etl.go.jp. As default, access control is performed for
 * classes. However, if you need, you can control accesses for
 * objects, methods or other in your program. See examples/accessControl.
 *
 *
 * <p>
 * Components of an ACL file:
 * <DL>
 * <DT>ACL Name
 * <DD>
 * name is the name of the ACL file. It is used to distinguish ACL file
 * when another machine donwload the ACL file from this machine.<p>
 * Hostname list is a space separated list of 
 * hostnames and/or network addresses.
 * <DT>Host Include List
 * <DD>
 * A hostname is either a domain name or a host name. Actually string
 * matching is performed to compare client's hostname with this name.
 * If client's hostname ends with this name, the matching succeeds.
 * For example, bungo.etl.go.jp matches with etl.go.jp in the list.
 * Note that abcetl.go.jp also matches with etl.go.jp. If
 * client has the name of bungo instead of bungo.etl.go.jp, it does
 * not match with etl.go.jp. You can use "localhost" to represent
 * the local hostname.
 * <br>
 * Dot separated numbers represent network addresses. Each number
 * must be in the range 0 to 255.
 * You can use * character 
 * to represent "any number". For example, 192.31.22.* allows access from
 * 192.31.22.1 to 192.31.22.255. 192.31.22.* is equivalent to 192.31.22.0
 * internally. <em>Use *.*.*.* or * to represent any
 * host.</em> ACL does not depends on IP address. That is, you can use
 * arbitrary length of dot separated byte sequences.
 * <DT>Host Exclude List
 * <DD>
 * A list with a key suffixed by ".host_exclude" is called a host exclude list.
 * Hosts appear in this list are not allowed to access this class.
 * <DT>User Include List
 * <DD>
 * A list with a key suffiexed by ".user" is called a host include list.
 * Clien't username must appear in this list to access the class. Use "*" to
 * represent any user. If client does not give a username, the user is
 * treated as "anonymous".
 * <DT>User Exclude List
 * <DD>
 * Users that appear in this list are not allowed to access this class.
 * <DT>User Password
 * <DD>
 * Each user appears in the user include list must have password entry.
 * A password entry is a non-encrypted string. Encryption will be supported
 * in the future release. If you need an encrypted password, do encryption
 * in your program. If you don't need password, give "*" as password.
 * It allows any password.
 * <DT>Creatable Flag
 * <DD>
 * If there exist a line like "className.creatable=false", the class
 * is not allowed to create remotely. Clients are allowed only to
 * connect the object. If this flag does not exists, clients can
 * create objects of the class.
 * 
 * <DT>Default Host Include List
 * <DD>
 * If requested class has no host include list in ACL, "default.host" is
 * checked if it exists. Default host include list is optional.
 * <DT>Default Host Exclude List
 * <DD>
 * If requested class has no host exclude list in ACL, "default.host_exclude" is
 * checked if it exists. Default host exclude list is optional.
 * <DT>Default User Include List
 * <DD>
 * If requested class has no user include list in ACL, "default.user" is
 * checked if it exists. Default user include list is optional.
 * <DT>Default User Exclude List
 * <DD>
 * If requested class has no user exclude list in ACL, "default.user_exclude" is
 * checked if it exists. Default user exclude list is optional.
 * </DL>
 *
 * How to access to ACL, refresh ACL for example:
 * <pre>
 *  Local:
 *     ACL acl = HORBServer.getSystemACL();
 *     acl.refresh();
 *  Remote:
 *     HorbURL url = new HorbURL("horb://hostname/SystemACLMaster");
 *     ACL acl = new ACL_Proxy(url);
 *     acl.refresh();
 * </pre>
 * If you want to access the system ACL remotely, set ACLMaster=true in 
 * a configuration file to start the SystemACLMaster object or
 * call HORBServer.startSystemACLMaster().
 */
public interface ACL {
  public static final short ACL_OK = 0;
  public static final short ACL_NOT_CHANGED = 1;
  public static final short ACL_NOT_REDISTRIBUTABLE = 2;
  public static final short ACL_NOT_FOUND = 3;
  public static final short ACL_NO_PERMISSION = 4;

  public static final String SYSTEM_ACL_MASTER = "SystemACLMaster";

  /**
   * check access permission for host and user.
   * returns true if className.host and className.user matches with the
   * client bound to ioci. <p>
   * local method.
   *
   * @param className class name or classname+methodname.
   * @param ioci IOCI of client
   * @return true if the list includes the host and the user, or no ACL
   */
  public boolean checkHostUser_Local(String className, IOCIService ioci);
    
  /**
   * check access permission for a host.
   * returns true if className.host matches with the
   * client bound to ioci. <p>
   * local method.
   *
   * @param className class name or classname+methodname.
   * @param ioci IOCI of client
   * @return true if the list includes the host, or no ACL
   */
  public boolean checkHost_Local(String className, IOCIService ioci);

  /**
   * check access permission for a user.
   * returns true if className.user matches with the
   * client bound to ioci. <p>
   * local method.
   *
   * @param className class name or classname+methodname.
   * @param ioci IOCI of client
   * @return true if the list includes the user, or no ACL
   */
  public boolean checkUser_Local(String className, IOCIService ioci);
    
  /**
   * return value of the key. only the first item of the value is returned.
   * For example, there is a line like;
   *     classname.creatable=false true
   * getValue("classname.creatable") returns "false".
   *
   * @return null if no ACL or the key is not found.
   */
  public String getValue_Local(String key);

  /**
   * (Internal) get an ACL file. ACLpair[] is an array of key, value pair.
   * To access this method, HORB server must have SystemACLMaster
   * entry in the system ACL.
   * 
   * @param name ACL name.
   * @param lastCheckTime 
   * @param force if true, returns list even if ACL has not changed since
   *			last download.
   * @return ACLpair object. status contains ACL_* flag. ACL_NOT_CHANGED if 
   *			requested ACL has not chaned since last download.
   */
  public ACLpair getList(String name, long lastCheckTime, boolean force) throws HORBException, IOException;

  /** 
   * refresh ACL. Reload ACL files that were modified since last refresh.
   */
  void refresh() throws HORBException, IOException;

  /** 
   * start a autorefresher thread. This thread reloads ACL files.
   * @param refresh refresh interval.
   */
  void startRefresher(int refresh) throws HORBException, IOException;

  /** stop the autorefresher thread */
  void stopRefresher() throws HORBException, IOException;

  /** start ACL Master as specified objectID. local method. */
  void startACLMaster_Local(String objectName) throws HORBException, IOException;
  /** stop ACL Master. local method. */
  void stopACLMaster_Local() throws HORBException, IOException;
}

