/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.io.*;
import java.util.*;
import java.net.*;

/**
 * The implementation of the HORB distributed access control list.
 *
 * <pre>
 *      name=acl_name
 *	className.host=hostname_sequence
 *	className.host_exclude=hostname_sequence
 *      className.user=username_sequence
 *      className.user_exclude=username_sequence
 *      username.password: lklaskjdf
 *
 * </pre>
 * For example;
 * <pre>
 *      name=etl_acl
 *      horb.orb.HORBAgent.host=bungo.etl.go.jp 192.31.99.23
 *      WClock.Test.host=etl.go.jp 192.31.*.*
 *	WCLock.Test.host_exclude=gate.etl.go.jp ftp.etl.go.jp
 *      WCLock.Test.user=hirano connelly guest
 *	WCLock.Test.user_exclude=halkja
 *      hirano.password=LKU&232ZC
 *	guest.password=*
 *
 * </pre>
 * name is the name of the ACL file. It is used when another machine
 * donwload the ACL file from this machine.<br>
 * hostname_sequence is a space separated hostnames or network addresses.
 * A hostname is either a domain name or a host name. If it is a host name,
 * accessing host must have the exact matching hostname. If it is a domain
 * name, accessing host is allowed to access the class if it has
 * the domain name as a substring. For example, a domain name, etl.go.jp, 
 * allows access by bungo.etl.go.jp.
 * Dot separated numbers represent network addresses. You can use * character
 * to represent any number. For example, 192.31.22.* allows access from
 * 192.31.22.1 to 192.31.22.255. <em>Use * to represent any host. Addresses
 * begins with * are always treated as *.*.*.*. For example, *.1.2
 * does not make sense, since it is treated as *.*.*.*</em>
 * A list with a key postpended ".host_exclude" is called a host exclude list.
 * Hosts appear in this list are not allowed to access this class.
 *
 *
 */
public class ACL_Impl extends Thread implements ACL {

  /**
   * We build this ACL from fileACL. fileACL contains entries like;
   *   <pre>
   *      name=acl_name
   *      package.class1.host=domain host ip1 ip2
   *      package.class1.host_exclude=host2
   *      package.class2.host=host1 host2 ip1 ip2
   *   </pre>
   * To speed up access check, we convert fileACL to this ACL like;
   *   <pre>
   *      package.class1.host=((domain) (host) (ip1) (ip2))
   *      package.class1.host_exclude=((host2))
   *      package.class2=((host1) (host2) (ip1) (ip2))
   *   </pre>
   * buildHash() does this conversion.
   */
  /** IP address requires at least four numbers */
  private static final int IP_LEN = 4;
  /** if false, all classes are accessible */
  private boolean check = true;	
  /** if true, no ACLs in this server. This cannot be a ACL master. */
  private boolean noACL;
  /** this is the real ACL */
  private Hashtable fastAcl;
  /** localhost name */
  private String localHostname;
  /** objectName of this ACL master. */
  private String myObjectName;

  /** auto refresher thread */
  private ACL_Impl refresher;
  private boolean stopRefresher = false;

  /** status of ACL files */
  private AclStatus[] acls;
  /** ACL files. The contents of these files are copied into fastAcl. */
  private Properties[] aclFiles;	// row acl file
  /** names of ACL files */
  private int refreshInterval;

  public ACL_Impl(AclStatus[] acls, String localHostname, int refresh) throws HORBException, IOException {
    if (acls == null) {
      check = false;
      noACL = true;
      return;
    }

    this.localHostname = localHostname;
    this.acls = acls;
    aclFiles = new Properties[acls.length];
    refresh();
    noACL = false;
    startRefresher(refresh);

//    list(System.out);
  }

  /** 
   * start a autorefresher thread. This thread reloads ACL files.
   * @param refresh refresh interval.
   */
  public synchronized void startRefresher(int refresh) {
    refreshInterval = refresh;
    stopRefresher = false;
    if (refresh > 0) {
      refresher = this;     // (ACL_Impl)new Thread(this);
      refresher.start();
    }
  }

  /** stop the autorefresher thread */
  public synchronized void stopRefresher() {
    stopRefresher = true;
  }

  public synchronized void refresh() throws HORBException, IOException {
    for (int i = 0; i < acls.length; i++) {
      try {
	AclStatus acl = acls[i];
	if (acl == null || acl.source == null)
	  continue;
	acl.refreshInterval = refreshInterval;
	if (acl.source.startsWith("horb:")) { // master's URL
	  HorbURL url = new HorbURL(acl.source);
	  load(url, i);			// download from master
	} else if (acl.source.startsWith("file:")) {
	  HorbURL url = new HorbURL(acl.source);
	  load(url.getObjectID(), i); // load from file
	} else
	  load(acl.source, i);		// load from file
      } catch (IOException e) {
	throw e;
//	System.exit(1);
      }
    }

    // rebuild fast acl
    Hashtable newFastAcl = new Hashtable();
    for (int i = 0; i < aclFiles.length; i++)
      if (aclFiles[i] != null)
	addFastAcl(newFastAcl, aclFiles[i]);
    fastAcl = newFastAcl;	// replace with the new one
  }

  private synchronized void load(String file, int aclNo) throws HORBException, IOException {
    System.out.print("loading "+file+ " as ACL "+aclNo);
    System.out.flush();
    FileInputStream fi = null;
    try {
      File fl = new File(file);
      if (fl.canRead() == false)
	throw new IOException(fl+" Not found");
      if (fl.lastModified() == acls[aclNo].lastModifiedTime) {
	System.out.println("..not changed");
	return;
      }
      acls[aclNo].lastModifiedTime = fl.lastModified();
      fi = new FileInputStream(fl);
    } catch (IOException e) {
      System.err.println("can't open ACL file: "+e);
      throw e;
    }
    Properties aclFile = new Properties();
    aclFile.load(fi);
    fixAclName(aclFile, aclNo);
    aclFiles[aclNo] = aclFile;
    acls[aclNo].lastLoadTime = System.currentTimeMillis();
    System.out.println("..loaded");
  }

  /**
   * load ACLFILES from a master ACL object.
   */
  private synchronized void load(HorbURL masterURL, int aclNo) throws HORBException, IOException {
    System.out.print("loading "+masterURL+ " as ACL"+aclNo);
    System.out.flush();
    ACL master = null;

    if (masterURL.getObjectID() == null)
      masterURL.setObjectID(ACL.SYSTEM_ACL_MASTER);

    String name = masterURL.getRef();
    if (name == null) {
      System.out.println("ACL name is required. e.g. horb://host/#ACL_ETL");
      throw new HORBException("no ACL name");
    }

    try {
      master = new ACL_Proxy(masterURL);
    } catch (Exception e) {
      System.err.println("can't connect to master ACL server " + masterURL);
      System.err.println(e);
      return;
    }

    acls[aclNo].name = name;

    ACLpair pairs = null;

    try {
      boolean force = false;
      if (acls[aclNo].lastLoadTime == 0)
	force = true;
      pairs = master.getList(name, acls[aclNo].lastLoadTime, force);
      ((ACL_Proxy)master)._release();
    } catch (Exception e) {
      throw new HORBException("ACL download fail for "+masterURL+" "+e);
    }
    if (pairs.status == ACL_NOT_REDISTRIBUTABLE)
      throw new HORBException("not redistributable "+masterURL);
    else if (pairs.status == ACL_NOT_FOUND)
      throw new HORBException("not found "+masterURL);
    else if (pairs.status == ACL_NO_PERMISSION)
      throw new HORBException("no permission "+masterURL);
    else {
      if (pairs.status != ACL_NOT_CHANGED) {
	Properties aclFile = new Properties();
	for (int i = 0; i < pairs.key.length; i++)
	  aclFile.put(pairs.key[i], pairs.value[i]);
	acls[aclNo].lastLoadTime = System.currentTimeMillis();
	fixAclName(aclFile, aclNo);
	aclFiles[aclNo] = aclFile;
	System.out.println("..loaded");
      } else 
	System.out.println("..not changed");
    }
  }

  private void fixAclName(Properties aclFile, int aclNo) throws HORBException {
    // fix ACL name
    String name;
    if ((name = (String)aclFile.get("name")) == null) {
      if (acls[aclNo].name == null)
	name = "ACL-"+aclNo;
    } else {
      if (acls[aclNo].name != null && name.equals(acls[aclNo].name) == false)
	throw new HORBException("ACL name mismatch (config="+acls[aclNo].name
			      +" ACL file="+name);
    }
    acls[aclNo].name = name;
//    System.out.println("ACL"+aclNo+"="+name);
  }


  private void addFastAcl(Hashtable hash, Properties aclFile) {
    Enumeration e = aclFile.propertyNames();
    while (e.hasMoreElements()) {
      String key = (String)e.nextElement();
      if ("name".equals(key))
	continue;
      String value = aclFile.getProperty(key);
      if (value == null)
	continue;

      try {
	StringTokenizer st = new StringTokenizer(value);
	int num = st.countTokens();
	if (num == 0)
	  continue;
//	System.out.println("num token = "+num);
	HostList hl[] = new HostList[num];
	for (int i = 0; i < num && st.hasMoreTokens(); i++) {
	  hl[i] = new HostList();
	  boolean exclude = false;
	  String tok = st.nextToken();
//	  System.out.println("token = "+tok);
	  if (Character.isDigit(tok.charAt(0)) || tok.charAt(0) == '*') {
	    hl[i].addr = strToAddr(tok);
	    hl[i].tag = HostList.ADDR;
	  } else {
	    hl[i].name = tok.toLowerCase();
	    hl[i].tag = HostList.NAME;
	    // in order to treat localhost as local hostname 
	    // normalize localhost to a real hostname
	    if ("localhost".equals(hl[i].name))
	      hl[i].name = localHostname;
	  }
	}
	hash.put(key, hl);
      } catch (Exception e2) {
	System.err.println("ACL format error: "+key+"="+value);
	e2.printStackTrace();
      }
    }
  }

  private byte[] strToAddr(String s) throws NumberFormatException {
    StringTokenizer st = new StringTokenizer(s, ".");
    int num = st.countTokens();
    if (num == 0)
      return null;
    // TODO: this depends on 4 (32bit IP)
    byte addr[] = new byte[(num < IP_LEN ? IP_LEN : num)];
    String x;
    for (int i = 0; i < num && st.hasMoreTokens(); i++) {
      x = st.nextToken();
      if (x.equals("*"))
	addr[i] = 0;
      else
	addr[i] = (byte)Integer.parseInt(x);
    }
    return addr;
  }

  /**
   * auto refresher.
   */
   public void run() {
     while (true) {
       try {
	 Thread.sleep(1000*refreshInterval);
       } catch (InterruptedException e) {}
       if (stopRefresher)
	 return;
       try {
	 refresh();
//	 list(System.out);
       } catch (Exception e) {}
     }
   }

  /**
   * check if ACL donwload is needed or allowed.
   * @param name ACL name.
   * @param lastCheckTime return ACL_NOT_CHANGED if requested ACL has not chaned since last download.
   */

  public synchronized ACLpair getList(String name, long lastLoadTime, boolean force) {
    int aclNo;

    ACLpair pairs = new ACLpair();
    if (noACL || name == null) {
      pairs.status = ACL_NOT_FOUND;
      return pairs;
    }

    // check the system ACL
    IOCIService ioci = null;
    try {
      ioci = HORBServer.getIOCIService();
    } catch (Exception e) {}

    // find ACL list having the requested name
  outer: 
    do {
      for (aclNo = 0; aclNo < acls.length; aclNo++) {
	if (name.equals(acls[aclNo].name)) {
	  if (lastLoadTime >= acls[aclNo].lastLoadTime) {
	    pairs.status = ACL.ACL_NOT_CHANGED;
	    if (force)
	      break outer;
	    return pairs;
	  }
	  if (acls[aclNo].redistributable == false) {
	    pairs.status = ACL.ACL_NOT_REDISTRIBUTABLE;
	    return pairs;
	  }
	  break outer;
	}
      }
      pairs.status = ACL_NOT_FOUND;
      return pairs;
    } while (false);

    pairs.status = ACL_OK;

    // build a list and return it
    Properties aclFile = aclFiles[aclNo];
    int size = aclFile.size();
    pairs.key = new String[size];
    pairs.value = new String[size];

    Enumeration e = aclFile.propertyNames();
    for (int i = 0; i < size && e.hasMoreElements(); i++) {
      String x = (String)e.nextElement();
      pairs.key[i] = x;
      pairs.value[i] = (String)aclFile.get(x);
    }
    return pairs;
  }

  /**
   * return value of the key. only the first item of the value is returned.
   * @return null if no ACL or the key is not found.
   */
  public String getValue_Local(String key) {
    HostList hl[];
    
    if (check == false || noACL)
      return null;
    if ((hl = (HostList[])fastAcl.get(key)) == null)
      return null;
    if (hl.length == 0)
      return null;
    if (hl[0].isAny())
      return "*";
    return hl[0].toString();
   }

  /**
   * checks host and user.
   */
  public boolean checkHostUser_Local(String className, IOCIService is) {
    if (check == false)
      return true;
    if (noACL)
      return false;

    if (checkHost_Local(className, is) == false)
      return false;
    if (checkUser_Local(className, is) == false)
      return false;
    return true;
  }


  /**
   * check user.
   */
  public boolean checkUser_Local(String className, IOCIService is) {
    HostList hl[];

    if (check == false)
      return true;
    if (noACL)
      return false;

    IOCICommon ic = (IOCICommon)is;
    String username = ic.username;
    String passwd = ic.passwd;

    if (username == null)
      username = "anonymous";
    boolean in = false;

    // get incliude list
    if ((hl = (HostList[])fastAcl.get(className+".user")) == null)
      if ((hl = (HostList[])fastAcl.get("default.user")) == null)
	return false;		// no user list

    //
    // check include list
    //
    for (int i = 0; i < hl.length; i++) {
      HostList h = hl[i];
      if (h.tag == HostList.NAME) {
//	System.out.println("include check: user "+username+" with acl "+h.name);
	if (username.equals(h.name)) {
//	  System.out.println("include match");
	  in = true;
	  break;
	}
      } else {			// address
	if (h.isAny()) {
//	  System.out.println("any user");
	  in = true;
	  break;
	}
      }
    }
    if (in == false)
      return false;

    //
    // check exclude list
    //
    if ((hl = (HostList[])fastAcl.get(className+".user_exclude")) == null)
      hl = (HostList[])fastAcl.get("default.user_exclude");

    if (hl != null) {		// exclude list exists
      for (int i = 0; i < hl.length; i++) {
	HostList h = hl[i];
	if (h.tag == HostList.NAME) {
//	  System.out.println("exclude check: user "+username+" with acl "+h.name);
	  if (username.equals(h.name)) {
//	    System.out.println("exclude match");
	    in = false;
	    break;
	  }
	} else {			// address
	  if (h.isAny()) {
//	    System.out.println("exclude any user");
	    in = false;
	    break;
	  }
	}
      }
    }
    if (in == false)
      return false;

    //
    // check password
    //
    if ((hl = (HostList[])fastAcl.get(username+".password")) != null) {
      HostList h = hl[0];
      if (h.tag == HostList.NAME) {
	if (h.name.equals(passwd) == false)
	  return false;
      } else
	if (h.isAny() == false)
	  return false;
    }
    return true;
  }

	
  /**
   * check host.
   */
  public boolean checkHost_Local(String className, IOCIService is) {

    HostList hl[];
    boolean in = false;

    if (check == false)
      return true;
    if (noACL)
      return false;

    IOCI ioci = (IOCI)is;
    String hostname = ioci.getHostName();
    byte[] addrByte = ioci.getAddress();

    if (hostname != null) {
      hostname = hostname.toLowerCase();
      if ("localhost".equals(hostname))
	hostname = localHostname; // use real hostname instead of localhost
    }

    // check incliude list
    if ((hl = (HostList[])fastAcl.get(className+".host")) == null)
      if ((hl = (HostList[])fastAcl.get("default.host")) == null)
	return false;		// default false

    // include list exists
    for (int i = 0; i < hl.length; i++) {
      HostList h = hl[i];
      if (h.tag == HostList.NAME) {
//	System.out.println("include check: host "+hostname+" with acl "+h.name);
	if (hostname != null && hostname.endsWith(h.name)) {
//	  System.out.println("include match");
	  in = true;
	  break;
	}
      } else {			// HostList.ADDR
	if (hl[i].addrMatch(addrByte)) {
	  in = true;
	  break;
	}
      }
    }
    if (in == false)
      return false;

    //
    // check exclude list
    //
    if ((hl = (HostList[])fastAcl.get(className+".host_exclude")) == null)
      if ((hl = (HostList[])fastAcl.get("default.host_exclude")) == null)
	return true;		// default false

    for (int i = 0; i < hl.length; i++) {
      HostList h = hl[i];
      if (h.tag == HostList.NAME) {
//	System.out.println("exclude check: host "+hostname+" with acl "+h.name);
	if (hostname != null && hostname.endsWith(h.name)) {
//	  System.out.println("exclude match");
	  in = false;
	  break;
	}
      } else {			// HostList.ADDR
	if (hl[i].addrMatch(addrByte)) {
	  in = false;
	  break;
	}
      }
    }
    return in;
  }

  void list(PrintStream ps) {
    Enumeration e = fastAcl.keys();
    while (e.hasMoreElements()) {
      String key = (String)e.nextElement();
      ps.print(key+"=");
      HostList[] hl = (HostList[])fastAcl.get(key);
      for (int j = 0; j < hl.length; j++)
	ps.print(hl[j] +" ");
      ps.println();
    }
  }

  public synchronized void startACLMaster_Local(String objectName) 
    throws HORBException, IOException {
    myObjectName = objectName;
    HORBServer.registerObject("horb.orb.ACL", this, objectName, 0);
  }

  public synchronized void stopACLMaster_Local() throws HORBException, IOException {
    HORBServer.unRegisterObject(myObjectName);
  }

/*
  public static void main(String argv[]) throws HORBException, IOException {
    AclStatus[] acls = new AclStatus[2];
    acls[0] = new AclStatus();
    acls[1] = new AclStatus();
    acls[0].source = "foo.acl";
    acls[1].source = "file:///bar.acl";
    ACL_Impl acl = new ACL_Impl(acls, InetAddress.getLocalHost().getHostName(), 3);
    acl.list(System.out);
    
    for (int i = 0; i < argv.length; i++)
      System.out.println(argv[i]+": "
			 +acl.access("horb.orb.HORBAgent", InetAddress.getByName(argv[i])));
  }    
*/
}
