/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.io.*;
import java.net.*;

/**
 * Basic implememtation of Inter-Object Communication Interface
 * for TCP/IP. Programmers must use IOCIService
 * interface to access IOCI related services. Don't access
 * this class directly except some specific methods.<p>
 * A subclass can inherit this class to make a new IOCI.<p>
 * This uses java.io.DataOutputStream and java.io.DataInputStream
 * for byte-order independent implementation. So this implementation
 * is not fast.
 * @see IOCIService.
 */
public class BasicIOCI extends IOCICommon implements IOCI {
  /** major version number */
  public static final short major_version = 1;
  /** minor version number */
  public static final short minor_version = 1;
  /** 0xBAC1 is the signature of BasicIOCI. Use another signature if you write
   * another IOCI.
   */
  public static final short signature = (short)0x4830;

  // instance variables
  private int listenCount = 30;
  ServerSocket serverSocket = null;
  Socket socket = null;
  public DataOutputStream os = null;
  public DataInputStream is = null;
  boolean asyncronous = false;	// not used
  private boolean isServer;
  short linkOption;

  public BasicIOCI() {}

  public short getMajorVersion() {
    return major_version; 
  }
  public short getMinorVersion() {
    return minor_version; 
  }

  public short getSignature() {
    return signature;
  }
  /**
   * set property of BasicIOCI. Currently the common property
   * requests listed in IOCIService are supported.
   */
  public void setProperty(int request, Object obj) {
    super.setProperty(request, obj);
  }

  /**
   * get property of BasicIOCI. Currently the common property
   * requests listed in IOCIService are supported.
   */
  public Object getProperty(int request, Object obj) { 
    return super.getProperty(request, obj);
  }

  /**
   * return true if connected. <em>Don't rely on
   * this method. This implementation always returns true
   * unless the connection has been opened and not released.</em>
   */
  public synchronized boolean isConnected() {
    if (socket == null)
      return false;
    // TODO: check connectivity
    return true;
  }

  public void serverInit(int port) throws IOException {
    ServerSocket s = null;
    isServer = true;
    
    for (int i = 0; i < 20; i++) {
      try {
	s = new ServerSocket(port, listenCount);
	break;
      } catch(IOException e) {
	System.err.println("waiting to create port "+port);
	try {
	  Thread.sleep(5000);
	} catch (InterruptedException ie) {}
	if (i == 0) {
	  System.err.println("failed to create server port for "+port);
	  throw e;
	}
      }
    }
    serverSocket = s;
  }

  /**
   * accept a connect request. When new request comes, make a new
   * IOCI and return it. Don't receive data from client in this
   * method since this method from a main server thread that should
   * not be blocked to receive data.
   */
  public IOCI serverAccept(int port) throws IOException {
    for (int retry = 3; retry >= 0; --retry) {
      try {
	Socket ns = serverSocket.accept();
	BasicIOCI ioci = new BasicIOCI();
	ioci.setStreams(ns);
	return (IOCI)ioci;
      } catch(IOException e) {
	try {
	  serverSocket.close();
	} catch (IOException e2) {}
	try {
	  serverSocket = new ServerSocket(port, listenCount);
	} catch (IOException e3) {
	  if (retry == 0)
	    throw e3;
	}
      }
    }
    return null;
  }

  public void connectServer(String host, int port) throws IOException {
//    socket = new ProxyingSocket(host, port);
//    is = socket.getInputStream();          // DataInputStream
//    os = socket.getOutputStream();	   // DataOutputStream
    socket = new Socket(host, port);
    setStreams(socket);
    sendIOCISignature();
  }

  /** send signature of IOCI */
  public void sendIOCISignature() throws IOException {
    sendShort(getSignature());
    sendShort(getMajorVersion());
    sendShort(getMinorVersion());
    sendShort((short)0);	// option
  }

  /** don't call this directly. */
  public boolean recvConnectServer() throws IOException {
    short sig = recvShort();
    if (getSignature() != sig) {
      sendShort(IOCI.STAT_IOCI_NOT_FOUND);
      if (debug)
	System.out.println("IOCI signature mismatch "+sig);
      return false;
    }
    short majorVersion = recvShort();
    if (getMajorVersion() != majorVersion) {
      sendShort(IOCI.STAT_IOCI_VERSION_MISMATCH);
      if (debug)
	System.out.println("IOCI major version mismatch "+majorVersion);
      return false;
    }
    short minorVersion = recvShort();
    linkOption = recvShort(); // connection option
    return true;
  }

  /** set streams. This method is used at the server side. */
  private void setStreams(Socket ns) throws IOException {
    socket = ns;
    is = new DataInputStream(new BufferedInputStream(ns.getInputStream()));
    os = new DataOutputStream(new BufferedOutputStream(ns.getOutputStream()));
  }

  public String getLocalHostName() throws IOException {
    return InetAddress.getLocalHost().getHostName();
  }
    
  public byte[] getLocalAddress() throws IOException {
    return InetAddress.getLocalHost().getAddress();
  }

  public InetAddress getLocalInetAddress() throws IOException {
    return InetAddress.getLocalHost();
  }

  public String getHostName() {
    if (host != null)
      return host;
    host = socket.getInetAddress().getHostName();
    return host;
  }

  public byte[] getAddress() {
    if (netAddress != null)
      return netAddress;
    netAddress = socket.getInetAddress().getAddress();
    return netAddress;
  }

  /**
   * get client's IP address. BasicIOCI specific.
   */
  public InetAddress getInetAddress() {
    return socket.getInetAddress();
  }

  public void release() {
//    System.out.println("****** release *****");

    // TODO: IS THIS NEEDED?
    try {
      sendShort(ORB_CLOSE);
      kick();
    } catch (Exception e) {}
    try {
      is.close();
    } catch (Exception e) {}
    try {
      os.close();
    } catch (Exception e) {}
    try {
//System.out.println("closing port "+port);
      socket.close();
    } catch (Exception e) {}
    is = null;
    os = null;
    socket = null;
    serverSocket = null;
  }

  // this finaize() is called when this proxy object disappears.
  // ensure to close connection when this proxy disappears.
  // it implies reference count in server also be decrementded
  //
  public void finalize() {
    release();
  }

  /** return number of bytes available without blocking. */
  public int available() throws IOException {
    return is.available();
  }

  public void kick() throws IOException {
    os.flush();
  }

  public void sendObject(Object o, String expectedClassName, Loopy loopy) throws HORBException, IOException, ProxyException {
    super.sendObject(o, expectedClassName, loopy, IOCI.LOC_NETWORK);
  }

  public Object recvObject(String expectedClassName, Goldberg gb) throws HORBException, IOException, ProxyException {
    return super.recvObject(expectedClassName, gb, IOCI.LOC_NETWORK);
  }

  private boolean sendNullCheck(Object value) throws IOException {
    if (value == null) {
      os.writeByte(IOCI.OBJ_NULL);
      return true;
    } else {
      os.writeByte(IOCI.OBJ_INST);
      return false;
    }
  }

  /*
   * senders for scalar types, array of scalar types and String object
   */
  public void sendBoolean(boolean value) throws IOException {
    sendByte((byte)(value ? 1 : 0));
  }
  public void sendBooleanArray(boolean value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeByte(value[i] ? 1 : 0);
  }
  public void sendByte(byte value) throws IOException {
    os.writeByte(value);
  }
  public void sendByteArray(byte value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    os.write(value, 0, value.length);
  }
  public void sendChar(char value) throws IOException {
    os.writeChar(value);
  }
  public void sendCharArray(char value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)    
      os.writeChar(value[i]);
  }
  public void sendShort(short value) throws IOException {
    os.writeShort(value);
  }
  public void sendShortArray(short value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeShort(value[i]);
  }
  public void sendInt(int value) throws IOException {
    os.writeInt(value);
  }
  public void sendIntArray(int value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeInt(value[i]);
  }
  public void sendLong(long value) throws IOException {
    os.writeLong(value);
  }
  public void sendLongArray(long value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeLong(value[i]);
  }
  public void sendFloat(float value) throws IOException {
    os.writeFloat(value);
  }
  public void sendFloatArray(float value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeFloat(value[i]);
  }
  public void sendDouble(double value) throws IOException {
    os.writeDouble(value);
  }
  public void sendDoubleArray(double value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeDouble(value[i]);
  }
  public void sendString(String value) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeUTF(value);
  }
  public void sendStringArray(String value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      sendString(value[i]);
  }


  //
  // receivers for scalar types, array of scalar types and String object
  //

  private boolean recvNullCheck() throws IOException {
    byte flag = is.readByte();
    if (flag == IOCI.OBJ_NULL)
      return true;
    else
      return false;
  }

  public boolean recvBoolean() throws IOException {
    return (is.readByte() == 0 ? false : true);
  }
  public boolean[] recvBooleanArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    boolean[] buf = new boolean[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = (is.readShort() == 0 ? false : true);
    }
    return buf;
  }
  public byte recvByte() throws IOException {
    return is.readByte();
  }
  public byte[] recvByteArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    byte[] buf = new byte[numItem];
    is.readFully(buf, 0, numItem);
    return buf;
  }
  public char recvChar() throws IOException {
    return is.readChar();
  }
  public char[] recvCharArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    char[] buf = new char[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readChar();
    }
    return buf;
  }
  public short recvShort() throws IOException {
    return is.readShort();
  }
  public short[] recvShortArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    short[] buf = new short[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readShort();
    }
    return buf;
  }
  public int recvInt() throws IOException {
    return is.readInt();
  }
  public int[] recvIntArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
//    System.out.println("recvIntArray["+numItem);
    int[] buf = new int[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readInt();
//      System.out.println("recvIntArray["+i+"]: "+buf[i]);
    }
    return buf;
  }
  public long recvLong() throws IOException {
    return is.readLong();
  }
  public long[] recvLongArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    long[] buf = new long[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readLong();
    }
    return buf;
  }
  public float recvFloat() throws IOException {
    return is.readFloat();
  }
  public float[] recvFloatArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    float[] buf = new float[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readFloat();
    }
    return buf;
  }
  public double recvDouble() throws IOException {
    return is.readDouble();
  }
  public double[] recvDoubleArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    double[] buf = new double[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readDouble();
    }
    return buf;
  }
  public String recvString() throws IOException {
    if (recvNullCheck())
      return null;
    return is.readUTF();
  }
  public String[] recvStringArray()  throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    String[] buf = new String[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = recvString();
    }
    return buf;
  }
}


