/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.net.*;
import java.io.*;
import java.util.*;

/**
 * HORB configration file parser and HORB server command line parser.
 */
class Config {
  Vector servers;
  Vector objects;
  Vector acls;
  ServerStatus[] serverStatus;
  ObjectStatus[] objectStatus;
  AclStatus[] aclStatus;
  int maxServerNo;
  int maxObjectNo;
  int maxAclNo;
  boolean debug;
  String logFile;
  boolean logging;
  int aclRefreshInterval;
  boolean aclMaster;

  Config(String argv[]) throws HORBException, IOException {
    int i;

    maxServerNo = 0;
    maxObjectNo = 0;
    maxAclNo = 0;
    servers = new Vector(3);
    servers.setSize(3);
    objects = new Vector(3);
    objects.setSize(3);
    acls = new Vector(3);
    acls.setSize(3);

    // at least one default configuration
    objects.setElementAt(new ObjectStatus(), 0);
    servers.setElementAt(new ServerStatus(), 0);
    acls.setElementAt(new AclStatus(), 0);

    // parse config file
    String configFile = findConfigFile(argv);
    if (configFile != null)
      loadConfigFile(new File(configFile));

    // copy vectors to arrays
    serverStatus = new ServerStatus[maxServerNo+1];
    Enumeration e = servers.elements();
    for (i = 0; i <= maxServerNo && e.hasMoreElements(); i++) {
      serverStatus[i] = (ServerStatus)servers.elementAt(i);
    }

    objectStatus = new ObjectStatus[maxObjectNo+1];
    e = objects.elements();
    for (i = 0; i <= maxObjectNo && e.hasMoreElements(); i++) {
      objectStatus[i] = (ObjectStatus)objects.elementAt(i);
    }

    aclStatus = new AclStatus[maxAclNo+1];
    e = acls.elements();
    for (i = 0; i <= maxAclNo && e.hasMoreElements(); i++) {
      aclStatus[i] = (AclStatus)acls.elementAt(i);
    }
    if (maxAclNo == 0 && aclStatus[0].source == null)
      aclStatus = null;		// no ACL in config file

    // override by commad line
    parseCommandLine(argv);

    HORBServer.autoStartServers = serverStatus;
    HORBServer.autoStartObjects = objectStatus;
    HORBServer.acls = aclStatus;
    HORBServer.logFile = logFile;
    HORBServer.logging = logging;
    HORBServer.debug = debug;
    HORBServer.aclMaster = aclMaster;
    HORBServer.aclRefreshInterval = aclRefreshInterval;
  }

  private void loadConfigFile(File file) throws HORBException, IOException {

    boolean error = false;

    FileInputStream fi = null;
    try {
       fi = new FileInputStream(file);
    } catch (IOException e) {
      System.err.println("can't open config file "+e);
      throw new HORBException("config file");
    }
    Properties prop = new Properties();
    prop.load(fi);

    String value;
    for (Enumeration e = prop.keys() ; e.hasMoreElements() ;) {
      String key = (String)e.nextElement();

      value = (String)prop.get(key);
      key = key.toLowerCase();
      ConfigElem keySeq[] = parse(key);

      if ("logfile".equals(keySeq[0].str))
	logFile = value;
      else if ("debug".equals(keySeq[0].str))
	debug = Boolean.valueOf(value).booleanValue();
      else if ("aclmaster".equals(keySeq[0].str))
	aclMaster = Boolean.valueOf(value).booleanValue();
      else if ("horbserver".equals(keySeq[0].str)) {
	String subkey = keySeq[1].str;
	if (subkey == null)
	  continue;
	int serverNo = keySeq[0].index;
	if (maxServerNo < serverNo)
	  maxServerNo = serverNo;
	if (serverNo >= servers.size()) // grow up vector
	  servers.setSize(serverNo+3);
	ServerStatus ss = (ServerStatus)servers.elementAt(serverNo);
	if (ss == null) {
	  ss = new ServerStatus();
	  servers.setElementAt(ss, serverNo);
	}

	if ("port".equals(subkey))
	  ss.port = parseInt(key, value);
      	else if ("name".equals(subkey))
	  ss.name = value;
      	else if ("iociclassname".equals(subkey))
	  ss.IOCIClassName = value;
	else if ("hostnamelookup".equals(subkey))
	  ss.hostnameLookup = Boolean.valueOf(value).booleanValue();
	else if ("debug".equals(subkey))
	  ss.debug = Boolean.valueOf(value).booleanValue();
	else if ("logging".equals(subkey))
	  ss.logging = Boolean.valueOf(value).booleanValue();
	else {
	  System.err.println("config file syntax error (subkey): "+key+": "+value);
	  error = true;
	}
      } else if ("object".equals(keySeq[0].str)) { // autostart objects
	String subkey = keySeq[1].str;
	if (subkey == null)
	  continue;
	int objectNo = keySeq[0].index;
	if (objectNo >= objects.size()) // grow up vector
	  objects.setSize(objectNo+3);
	if (objectNo > maxObjectNo)
	  maxObjectNo = objectNo;
	ObjectStatus os = (ObjectStatus)objects.elementAt(objectNo);
	if (os == null) {
	  os = new ObjectStatus();
	  objects.setElementAt(os, objectNo);
	}

	if ("classname".equals(subkey))
	  os.className = value;
	else if ("objectid".equals(subkey))
	  os.objectID = value;
	else if ("implementation".equals(subkey))
	  os.implementation = value;
	else if ("port".equals(subkey))
	  os.port = parseInt(key, value);
	else {
	  System.err.println("config file syntax error (subkey): "+key+": "+value);
	  error = true;
	}

      } else if ("accesscontrol".equals(keySeq[0].str)) {
	String subkey = keySeq[1].str;
	if (subkey == null)
	  continue;
	int aclNo = keySeq[0].index;
	if (aclNo >= acls.size()) // grow up vector
	  acls.setSize(aclNo+3);
	if (aclNo > maxAclNo)
	  maxAclNo = aclNo;
	AclStatus as = (AclStatus)acls.elementAt(aclNo);
	if (as == null) {
	  as = new AclStatus();
	  acls.setElementAt(as, aclNo);
	}

	if ("name".equals(subkey))
	  as.name = value;
	else if ("source".equals(subkey))
	  as.source = value;
	else if ("redistributable".equals(subkey))
	  as.redistributable = Boolean.valueOf(value).booleanValue();
	else if ("refreshinterval".equals(subkey))
	  aclRefreshInterval = parseInt(key, value);
	else {
	  System.err.println("config file syntax error (subkey):"+key+": "+value);
	  error = true;
	}
      } else {
	System.err.println("config file syntax error: "+key+": "+value);
	error = true;
      }
    }
    if (error)
      System.exit(1);
  }

  /**
   * parse string like horbserver[0].port.
   */
  protected ConfigElem[] parse(String str) {
    String x;
    StringTokenizer st = new StringTokenizer(str, ".");
    int num = st.countTokens();
    ConfigElem[] elem = new ConfigElem[num];
    for (int i = 0; i < num && st.hasMoreTokens(); i++) {
      elem[i] = new ConfigElem();
      x = st.nextToken();
      elem[i].str = x;
      int at = x.indexOf('[');
      if (at >= 0) {
	elem[i].str = x.substring(0, at);
	try {
	  String val = x.substring(at+1);
	  at = val.indexOf(']');
	  elem[i].index = Integer.parseInt(val.substring(0, at));
	} catch (Exception e) {}
      }
    }
    return elem;
  }

  private String findConfigFile(String argv[]) {
    for (int i = 0; i < argv.length; i++) {
      if ("-conf".equals(argv[i])) {
	if ((i+1) < argv.length) {
	  i++;
	  return argv[i];
	} else {
	  System.err.println("-conf configfile");
	  System.exit(1);
	}
      }
    }
    return null;
  }

  /**
   * parse command line. Users can set options for the server[0] and object[0]
   * by the command line.
   */
  private void parseCommandLine(String argv[]) {
    for (int i = 0; i < argv.length; i++) {
      if ("-h".equals(argv[i]) || "?".equals(argv[i]) || "-help".equals(argv[i])) {
	usage();
	System.exit(0);
      } else if ("-conf".equals(argv[i])) { // skip config file
	if ((i+1) < argv.length)
	  i++;
      } else if ("-p".equals(argv[i]) || "-port".equals(argv[i])) {
	if ((i+1) < argv.length) {
	  i++;
	  serverStatus[0].port = Integer.parseInt(argv[i]);
	} else {
	  System.err.println("-port portno");
	  System.exit(1);
	}
      } else if ("-ioci".equals(argv[i])) {
	if ((i+1) < argv.length) {
	  i++;
	  serverStatus[0].IOCIClassName = argv[i];
	} else {
	  System.err.println("-ioci requires IOCI class");
	  System.exit(1);
	}
      } else if ("-acl".equals(argv[i])) {
	if ((i+1) < argv.length) {
	  i++;
	  aclStatus = new AclStatus[1];
	  aclStatus[0] = new AclStatus();
	  aclStatus[0].source = argv[i];
	} else {
	  System.err.println("-acl aclfile | URL");
	  System.exit(1);
	}
      } else if ("-aclmaster".equals(argv[i])) {
	aclMaster = true;
      } else if ("-v".equals(argv[i]) || "-verbose".equals(argv[i])) {
	debug = true;
	for (int s = 0; s < serverStatus.length; s++)
	  serverStatus[s].debug = true;
      } else if ("-version".equals(argv[i])) {
	System.out.println("HORB ORB Version "+HORB.major_version
			   +"."+HORB.minor_version+"."+HORB.releaseLevel);
	System.out.println("HORB BasicIOCI Version "+
			   BasicIOCI.major_version+"."+BasicIOCI.minor_version);
	System.out.println("Copyright 1995, 1996 HIRANO Satoshi");
	System.exit(0);
      } else if ("-logfile".equals(argv[i])) {
	if ((i+1) < argv.length) {
	  i++;
	  logFile = argv[i];
	  logging = true;
	  serverStatus[0].logging = true;
	} else {
	  System.err.println("-logfile filename");
	  System.exit(1);
	}
      } else if ("-start".equals(argv[i])) {
	if ((i+2) < argv.length) {
	  objectStatus = new ObjectStatus[1];
	  objectStatus[0] = new ObjectStatus();
	  objectStatus[0].className = argv[++i];
	  objectStatus[0].objectID = argv[++i];
	} else {
	  System.err.println("-start class objectID");
	  System.exit(1);
	}
      } else if (argv[i].startsWith("-")) {
	System.err.println("invalid flag: " + argv[i]);
	System.exit(1);
      } else {
	System.err.println("invalid argument: "+argv[i]);
	usage();
	System.exit(1);
      }
    }
  }

  static void usage() {
    System.err.println("usage: horb [options]");
    System.err.println("options:");
    System.err.println("    -? -h -help      show this message");
    System.err.println("    -conf file       configuration file");
    System.err.println("    -v -verbose      verbose");
    System.err.println("    -version         show version string");
    System.err.println("    -acl file|URL    access control list");
    System.err.println("    -aclmaster       ACL master");
    System.err.println("    -port num        use port <num>");
    System.err.println("    -ioci class      IOCI classname (default horb.orb.BasicIOCI)");
    System.err.println("    -start class objectID  start class as HORB object");
    System.err.println();
  }

  private final int parseInt(String key, String value) throws NumberFormatException {
    int i = 0;
    try {
      i = Integer.parseInt(value);
      return i;
    } catch (NumberFormatException e) {
      System.err.println("Number expected "+key+"="+value);
      throw e;
    }
  }
}
