/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.io.*;
import java.net.*;

/**
 * Persistent objects support class. By using this class, you can
 * save and restore objects to/from files. Typical usage is;<p>
 * <pre>
 *  to save:
 *   FileIOCI file = new FileIOCI("foo.fof");
 *   file.save(object);
 *
 *  to restore:
 *   FileIOCI file = new FileIOCI("foo.fof");
 *   Object object = file.load();
 * </pre>
 *
 * A file saved by FileIOCI is called a foffile, meaning freezed object
 * file. You can get info of a foffile
 * by 'fofrun -info foo.fof' command.<p>
 *
 * If a saved object implements the Restartable interface, the
 * object can be executed by the fofrun command. See Restartable
 * for more detail. See examples/foffile for an example.
 *
 * Callable methods in this class are, save(), load(), getFofInfo(),
 * getMajorVersion(), getMinorVersion(), setProperty(), getProperty(),
 * and getLocalHostName().
 *
 * @see Restartable, IOCIService.
 */
public final class FileIOCI extends IOCICommon implements IOCI {
  // version
  public static final short major_version = 1;
  public static final short minor_version = 0;
  /**
   * 0x4830 is the signature of FileIOCI. Use another signature if you write
   * another IOCI.
   */
  public static final short signature = (short)0x4831;

  // instance variables
  HorbURL url;
  File file;
  FofInfo header;
  public DataOutputStream os = null;
  public DataInputStream is = null;

  // 

  public FileIOCI() {}

  /** create FileIOCI for the URL.
   * @param url this must be like "file:///filename". If it contains
   * directory, the directory should exists.
   */
  public FileIOCI(HorbURL url) throws HORBException {
    this.url = url;
    if ("file".equals(url.getProtocol()) == false)
      throw new ArgumentException("file:// required in URL");
    file = new File(url.getObjectID());
  }

  /** create FileIOCI for the File.
   * @param file File object of the target file.
   */
  public FileIOCI(File file) throws HORBException {
    this.file = file;
    url = new HorbURL("file:///"+file);
  }

  /** create FileIOCI for the filename.
   * @param filename a platform dependent local filename. If it contains
   * directory, the directory should exists.
   */
  public FileIOCI(String filename) throws HORBException {
    this(new HorbURL("file:///"+filename));
  }

  /**
   * save object. This saves all reachable object from the object into
   * the file.
   * @param o object to be saved.
   */
  public void save(Object o) throws HORBException, IOException {
    FileOutputStream fs = new FileOutputStream(file);
    os = new DataOutputStream(new BufferedOutputStream(fs));

    header = new FofInfo((IOCI)this, url.toString(), o);
    // save header
    sendObject(header, "horb.orb.FofInfo", new Loopy());
    // save the object
    sendObject(o, "java.lang.Object", new Loopy());
    kick();
    os.close();
    os = null;
  }

  /**
   * load object. The object should be saved by the save() method.
   * @return loaded object.
   */
  public Object load() throws HORBException, IOException {
    FileInputStream fs = new FileInputStream(file);
    is = new DataInputStream(new BufferedInputStream(fs));
    // load header
    header = (FofInfo)recvObject("horb.orb.FofInfo", new Goldberg());
    if (header.IOCI_major_version != getMajorVersion())
      throw new HORBException("FileIOCI version mismatch in " + file);
    // load object
    Object o = recvObject("java.lang.Object", new Goldberg());
    is.close();
    is = null;
    return o;
  }

  /**
   * return information of the fof file.
   */
  public FofInfo getFofInfo() throws HORBException, IOException {
    FileInputStream fs = new FileInputStream(file);
    is = new DataInputStream(new BufferedInputStream(fs));
    // load header
    header = (FofInfo)recvObject("horb.orb.FofInfo", new Goldberg());
    is.close();
    return header;
  }

  public short getSignature() {
    return signature;
  }
  /** returns major version of this implementation. If this number
   * differs from a foffile's major version number, this FileIOCI
   * couldn't read the foffile.
   */
  public short getMajorVersion() {
    return major_version; 
  }
  /** returns minor version of this implementation. */
  public short getMinorVersion() {
    return minor_version; 
  }

  /**
   * set property of FileIOCI. Currently the common property
   * requests listed in IOCIService are supported.
   */
  public void setProperty(int request, Object obj) {
    super.setProperty(request, obj);
  }
  /**
   * get property of FileIOCI. Currently the common property
   * requests listed in IOCIService are supported.
   */
  public Object getProperty(int request, Object obj) { 
    return super.getProperty(request, obj);
  }

  /** returns current local hostname */
  public String getLocalHostName() throws IOException { 
    return InetAddress.getLocalHost().getHostName();
  }

  /** don't call this directly. */
  public void finalize() {
    release();
  }

  /** return number of bytes available without blocking. */
  public int available() throws IOException {
    return is.available();
  }

  /**
   * release connection. You don't need to call this.
   */
  public void release() {
    try {
      is.close();
    } catch (Exception e) {};
    try {
      os.close();
    } catch (Exception e) {};
    is = null;
    os = null;
  }

  /** don't call this directly. */
  public final void kick() throws IOException {
    os.flush();
  }

  public void sendObject(Object o, String expectedClassName, Loopy loopy) throws HORBException, IOException, ProxyException {
    super.sendObject(o, expectedClassName, loopy, IOCI.LOC_FILE);
  }

  public Object recvObject(String expectedClassName, Goldberg gb) throws HORBException, IOException, ProxyException {
    return super.recvObject(expectedClassName, gb, IOCI.LOC_FILE);
  }

  /** don't call this directly. */
  private final boolean sendNullCheck(Object value) throws IOException {
    if (value == null) {
      os.writeByte(IOCI.OBJ_NULL);
      return true;
    } else {
      os.writeByte(IOCI.OBJ_INST);
      return false;
    }
  }

  /*
   * senders for scalar types, array of scalar types and String object
   */
  /** don't call this directly. */
  public final void sendBoolean(boolean value) throws IOException {
    sendByte((byte)(value ? 1 : 0));
  }
  /** don't call this directly. */
  public final void sendBooleanArray(boolean value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeByte(value[i] ? 1 : 0);
  }
  /** don't call this directly. */
  public final void sendByte(byte value) throws IOException {
    os.writeByte(value);
  }
  /** don't call this directly. */
  public final void sendByteArray(byte value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    os.write(value, 0, value.length);
  }
  /** don't call this directly. */
  public final void sendChar(char value) throws IOException {
    os.writeChar(value);
  }
  /** don't call this directly. */
  public final void sendCharArray(char value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)    
      os.writeChar(value[i]);
  }
  /** don't call this directly. */
  public final void sendShort(short value) throws IOException {
    os.writeShort(value);
  }
  /** don't call this directly. */
  public final void sendShortArray(short value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeShort(value[i]);
  }
  /** don't call this directly. */
  public final void sendInt(int value) throws IOException {
    os.writeInt(value);
  }
  /** don't call this directly. */
  public final void sendIntArray(int value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeInt(value[i]);
  }
  /** don't call this directly. */
  public final void sendLong(long value) throws IOException {
    os.writeLong(value);
  }
  /** don't call this directly. */
  public final void sendLongArray(long value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeLong(value[i]);
  }
  /** don't call this directly. */
  public final void sendFloat(float value) throws IOException {
    os.writeFloat(value);
  }
  /** don't call this directly. */
  public final void sendFloatArray(float value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeFloat(value[i]);
  }
  /** don't call this directly. */
  public final void sendDouble(double value) throws IOException {
    os.writeDouble(value);
  }
  /** don't call this directly. */
  public final void sendDoubleArray(double value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      os.writeDouble(value[i]);
  }
  /** don't call this directly. */
  public final void sendString(String value) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeUTF(value);
  }
  /** don't call this directly. */
  public final void sendStringArray(String value[]) throws IOException {
    if (sendNullCheck(value))
      return;
    os.writeInt(value.length);
    for (int i = 0; i < value.length; i++)
      sendString(value[i]);
  }


  //
  // receivers for scalar types, array of scalar types and String object
  //

  /** don't call this directly. */
  private boolean recvNullCheck() throws IOException {
    byte flag = is.readByte();
    if (flag == IOCI.OBJ_NULL)
      return true;
    else
      return false;
  }

  /** don't call this directly. */
  public final boolean recvBoolean() throws IOException {
    return (is.readByte() == 0 ? false : true);
  }
  /** don't call this directly. */
  public final boolean[] recvBooleanArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    boolean[] buf = new boolean[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = (is.readShort() == 0 ? false : true);
    }
    return buf;
  }
  /** don't call this directly. */
  public final byte recvByte() throws IOException {
    return is.readByte();
  }
  /** don't call this directly. */
  public final byte[] recvByteArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    byte[] buf = new byte[numItem];
    is.readFully(buf, 0, numItem);
    return buf;
  }
  /** don't call this directly. */
  public final char recvChar() throws IOException {
    return is.readChar();
  }
  /** don't call this directly. */
  public final char[] recvCharArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    char[] buf = new char[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readChar();
    }
    return buf;
  }
  /** don't call this directly. */
  public final short recvShort() throws IOException {
    return is.readShort();
  }
  /** don't call this directly. */
  public final short[] recvShortArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    short[] buf = new short[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readShort();
    }
    return buf;
  }
  /** don't call this directly. */
  public final int recvInt() throws IOException {
    return is.readInt();
  }
  /** don't call this directly. */
  public final int[] recvIntArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
//    System.out.println("recvIntArray["+numItem);
    int[] buf = new int[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readInt();
//      System.out.println("recvIntArray["+i+"]: "+buf[i]);
    }
    return buf;
  }
  /** don't call this directly. */
  public final long recvLong() throws IOException {
    return is.readLong();
  }
  /** don't call this directly. */
  public final long[] recvLongArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    long[] buf = new long[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readLong();
    }
    return buf;
  }
  /** don't call this directly. */
  public final float recvFloat() throws IOException {
    return is.readFloat();
  }
  /** don't call this directly. */
  public final float[] recvFloatArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    float[] buf = new float[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readFloat();
    }
    return buf;
  }
  /** don't call this directly. */
  public final double recvDouble() throws IOException {
    return is.readDouble();
  }
  /** don't call this directly. */
  public final double[] recvDoubleArray() throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    double[] buf = new double[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = is.readDouble();
    }
    return buf;
  }
  /** don't call this directly. */
  public final String recvString() throws IOException {
    if (recvNullCheck())
      return null;
    return is.readUTF();
  }
  /** don't call this directly. */
  public final String[] recvStringArray()  throws IOException {
    if (recvNullCheck())
      return null;
    int numItem = is.readInt();
    String[] buf = new String[numItem];
    for (int i = 0; i < numItem; i++) {
      buf[i] = recvString();
    }
    return buf;
  }

  /** don't call this directly. */
  public synchronized boolean isConnected() {
    return true;
  }

  /** don't call this directly. */
  public void serverInit(int port) throws IOException {
    throw new IOException("FileIOCI doesn't support serverInit()");
  }
  /** don't call this directly. */
  public IOCI serverAccept(int port) throws IOException { return null; }
  /** don't call this directly. */
  public void connectServer(String host, int port) throws IOException {}
  /** don't call this directly. */
  public void sendIOCISignature() throws IOException {}
  /** don't cll this directly. */
  public boolean recvConnectServer() throws IOException { return false; }
  /** don't call this directly. */
  public byte[] getLocalAddress() throws IOException { return null; }
  /** don't call this directly. */
  public String getHostName() { return null; }
  /** don't call this directly. */
  public byte[] getAddress() { return null; }
}
