/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.net.UnknownHostException;

/**
 * HORB main frame. This class used mainly 
 * at the client side. Server side functions are separated in class HORBServer
 * in order to keep this class small. This class is donwloaded into client
 * systems, while HORBServer is not.
 *
 * @see HORBServer
 */
public class HORB {
  public static final short major_version = 1;
  public static final short minor_version = 3;
  public static final String releaseLevel = "b1";

  static boolean running;
  static String IOCIClassName = "horb.orb.BasicIOCI";
  static Class IOCIClass;	// default IOCI Class
  static int port = 8887;	// default port

  static {
    try {
      IOCIClass = Class.forName(IOCIClassName);
    } catch (Exception e) {}
  }

  /**
   * Start Client HORB for default port number. No HORBServer.
   *
   * @throws HORBException
   */
  public HORB() throws HORBException {
    this(false, 0, null);
  }

  /**
   * Start HORB for default port. If server flag is true, start a Server HORB.
   *
   * @param  server     if true starts a HORBServer also.
   * @throws HORBException
   */
  public HORB(boolean server) throws HORBException {
    this(server, 0, null);
  }

  /**
   * Start HORB for port. If server flag is true, start a HORBServer.
   *
   * @param  server     if true starts a HORB server also.
   * @param  port       port number for communication.
   *			If 0 or -1, the default port number is used
   *			for outgoing. If -1 and the server flag is true, 
   *			a HORB server starts without port server.
   *
   * @throws HORBException
   */
  public HORB(boolean server, int port) throws HORBException {
    this(server, port, null);
  }

  /**
   * Start HORB for port. If server flag is true, start a HORBServer.
   *
   * @param  server     if true starts a HORB server.
   * @param  port       port number for communication.
   *			If 0 or -1, the default port number is used
   *			for outgoing. If -1 and the server flag is true, 
   *			a HORB server starts without port server.
   * @param  IOCIClassName class name of IOCI. 
   *			If null, the default IOCI (BasicIOCI) is used.
   *
   * @throws HORBException
   */
  public HORB(boolean server, int port, String IOCIClassName) throws HORBException {
    if (running)
      return;

    if (port == 0)		// use default port
      port = this.port;
    else if (port == -1)	// don't run port server.
      ;
    else
      this.port = port;

    if (IOCIClassName != null)
      HORB.IOCIClassName = IOCIClassName;
    try {
      HORB.IOCIClass = Class.forName(HORB.IOCIClassName);
    } catch (Exception e1) {
      throw new HORBException("can't load IOCI class: "+IOCIClassName+" - "+e1.toString());
    }

    if (server) {
      try {
	HORBServer horbServer = new HORBServer(port, null, IOCIClassName);
      } catch (Exception e1) {
	throw new HORBException("HORBServer initializatin error "+e1.toString());
      }
    }
    running = true;
  }

  /**
   * Set default port number for IOCI (Inter-Object Communication). 
   * Change is effective from the nextly created IOCI entity.
   */
  public static final void setPort(int port) throws HORBException {
    if (port == 0)
      throw new HORBException("illegal port number");
    HORB.port = port;
  }

  /**
   * Returns current default port number for IOCI.
   */
  public static final int getPort() {
    return port;
  }

  /**
   * Set default IOCI class. The class name must be a fully 
   * qualified class name (package name and class name). 
   * 
   * Default "horb.orb.BasicIOCI".
   *
   * This method affect every creation of proxy objects. If you want to 
   * set IOCI class for a proxy object, follow these steps.
   *
   * <pre>
   *     Foo_Proxy foo = new Foo_Proxy();
   *     foo._setIOCI(new yourIOCIClass());
   *     foo._connet(new HorbURL("horb://hostname/"));
   *     foo.methods();
   * </pre>
   *
   * @param newClass name of IOCI class
   * @exception HORBException 
   */
  public static final void setIOCIClassName(String newClass) throws ClassNotFoundException {
    HORB.IOCIClass = Class.forName(newClass);
    IOCIClassName = newClass;
  }

  /**
   * Get current default IOCI class.
   */
  public static final String getIOCIClassName() {
    return IOCIClassName;
  }

  /**
   * Get current default IOCI class.
   */
  public static final Class getIOCIClass() {
    return IOCIClass;
  }

  /**
   * restore system objects at recovery.
   */
  void _reviveHook(byte place) {
    try {
      IOCIClass = Class.forName(IOCIClassName);
    } catch (Exception e) {};
  }
}

