/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;
import java.net.*;

/**
 * HORB management agent.
 *
 * This object can be used either from server side objects or from
 * management clients. There are three categories, HORB Server management,
 * HORB object mangement, and HORB thread management in this management
 * agent.
 * <UL>
 * <LI>HORB server - One for each port. It Accepts a connection from a client,
 * 	then binds the client to a HORB thread of a desired HORB object.
 * <LI>HORB object - A HORB object (aka. a remote object) can have
 *      multiple HORB threads.
 * <LI>HORB thread - A HORB thread corresponds to a client. A client
 *      calls methods repeatedly in the context of a HORB thread.
 * </UL>
 *<p>
 * Pay attention to security. It is strongly recommended to set up ACL
 * for HORBAgent.
 */
public interface HORBAgent {

  /*
   * HORB server management
   */

  /**
   * returns status of all HORB servers.
   */
  public ServerStatus[] getServerStatus() throws HORBException;

  /**
   * returns status of a HORB server serving the port. null if no server 
   * is active for the port.
   *
   * @param port server port
   */
  public ServerStatus getServerStatus(int port) throws HORBException;

  /**
   * returns debugging flag of the HORB server for the current thread.
   */
  public boolean isHORBServerDebugging() throws HORBException;

  /**
   * returns debugging flag of the HORB server for the port.
   * @param port server port.
   */
  public boolean isHORBServerDebugging(int port) throws HORBException;

  /**
   * set debug switch of the HORB server for the current thread. If this
   * switch is on, all threads launced from the HORB server prints stack
   * dump when an uncaught exception occures. This switch can also be
   * turned on by 'horb -v' option.
   * @param sw true or false
   */
  public void setHORBServerDebug(boolean sw) throws HORBException;

  /**
   * set debug switch of the HORB server for the port. If this
   * switch is on, all threads launced from the HORB server prints stack
   * dump when an uncaught exception occures. This switch can also be
   * turned on by 'horb -v' option.
   * @param port server port.
   * @param sw true or false
   */
  public void setHORBServerDebug(int port, boolean sw) throws HORBException;
  
  /**
   * terminate the HORB server that launched the current thread.
   */
  public void stopHORBServer() throws HORBException;

  /**
   * terminate the HORB server for the port.
   * @param port server port.
   */
  public void stopHORBServer(int port) throws HORBException;

  /**
   * terminate the HORB process.
   *
   * @param exitCode this HORB process exit with exit code.
   */
  public void terminateHORB(int exitCode) throws HORBException;

  /*
   * HORB Object management
   */

  /**
   * returns status of HORB objects.
   */
  public ObjectStatus[] getObjectStatus() throws HORBException;

  /**
   * returns status of a specific HORB object.
   */
  public ObjectStatus getObjectStatus(String objectID) throws HORBException;

  /**
   * unregister a HORB object registered by registerObject().
   * @param objectID  object ID of the object to be unregistered.
   * @exception NoObjectException no such object found
   *
   */
  public void unRegisterObject(String objectID) throws NoObjectException, HORBException;

  /**
   * stop acceptance of connect requests to an object. If one tries
   * to connect the object, the request will fail.
   * @param objectID  object ID to be stopped.
   */
  public void stopObject(String objectID) throws HORBException;

  /**
   * resume acception of connections to an object.
   * @param objectID  object ID to be resumed.
   */
  public void resumeObject(String objectID) throws HORBException;

  /*
   * HORB thread management
   */
  
  /**
   * returns objectID of the current thread. 
   * Since one daemon object can have multiple threads for
   * requests, several threads can have the same object name.
   * This method intends to be used from a server object (HORB object).
   * At the client side, you can obtain the objectID of a remote object
   * from the proxy object, by using _getObjectURL().
   * <pre>
   *   How to get objectID at client side:
   *
   *	  Foo_Proxy remote = new Foo_Proxy(horbURL);
   *	  HorbURL url = remote._getObjectURL();
   *	  String objectID = url.getObjectID();
   *
   *   How to get objectID at server side:
   *
   *	  HORBAgent agent = new HORBAgent_Impl();
   *	  String objectID = agent.objectID();
   * </pre>
   *
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public String getObjectID()  throws HORBException;

  /**
   * returns network address of the client for this HORB thread.
   * This method intends to be used at server side. So, this does not make
   * sense when this is called remotely.
   *
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public byte[] getClientNetAddress_Local()  throws HORBException;

  /**
   * returns IP address of the client for this HORB thread.
   * This method intends to be used at server side. So, this does not make
   * sense when this is called remotely.
   *
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public InetAddress getClientInetAddress_Local()  throws HORBException;

  /**
   * returns hostname of the client for this HORB thread. 
   * If hostname lookup fails, this method returns null.
   * This method intends to be used at server side. So, this does not make
   * sense when this is called remotely.
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public String getClientHostName_Local()  throws HORBException;

  /**
   * terminate a HORB thread.
   * @param threadName HORB thread to be terminated.
   */
  void stopHORBThread(String threadName)  throws HORBException;

  /**
   * suspend a HORB thread.
   * @param threadName HORB thread to be suspended.
   */
  void suspendHORBThread(String threadName)  throws HORBException;

  /**
   * resume a HORB thread.
   * @param threadName HORB thread to be resumed.
   */
  void resumeHORBThread(String threadName)  throws HORBException;

  /**
   * set priority of a thread.
   * @param newPriority new priority. Value should be within the range
   * Thread.MIN_PRIORTY,  Thread.MAX_PRIORITY.
   */
  void setPriorityHORBThread(String threadName, int newPriority) throws IllegalArgumentException, HORBException;

  /**
   * get priority of a HORB thread.
   */
  void getPriorityHORBThread(String threadName)  throws HORBException;

  /**
   * returns status of threads of all HORB object.
   */
  public ThreadStatus[] getThreadStatus() throws HORBException;
    
  /**
   * returns status of threads of a HORB object.
   * @param objectID   return threads' status concerning to objectID.
   */
  public ThreadStatus[] getThreadStatus(String objectID) throws HORBException;

}
