/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.net.*;

/**
 * The implementation of the HORB management agent. See HORBAgent
 * for detail.
 */
public class HORBAgent_Impl implements HORBAgent {

  /*
   * HORB server management
   */

  /**
   * returns status of all HORB servers.
   */
  public ServerStatus[] getServerStatus() {
    return HORBServer.serverTable.getServerStatus();
  }

  /**
   * returns status of this HORB server. null if no server is active for port 
   */
  public ServerStatus getServerStatus(int port) {
    ServerStatus ss = null;
    try {
      ss = HORBServer.serverTable.getServerStatus(port);
    } catch (Exception e) {}
    return ss;
  }

  /**
   * returns debugging flag of the HORB server for the current thread.
   */
  public final boolean isHORBServerDebugging() {
    try {
      HORBServer server = HORBServer.getHORBServer();
      return server.ss.debug;
    } catch (Exception e){}
    return false;
  }

  /**
   * returns debugging flag of the HORB server for the port
   * @param port server port.
   */
  public final boolean isHORBServerDebugging(int port) {
    try {
      HORBServer server = HORBServer.getHORBServer(port);
      return server.ss.debug;
    } catch (Exception e){}
    return false;
  }

  /**
   * set debug switch of the HORB server for the current thread. If this
   * switch is on, all threads launced from the HORB server prints stack
   * dump when an uncaught exception occures. This switch can also be
   * turned on by 'horb -v' option.
   * @param sw true or false
   */
  public final void setHORBServerDebug(boolean sw) {
    try {
      HORBServer server = HORBServer.getHORBServer();
      server.ss.debug = sw;
    } catch (Exception e){}
  }

  /**
   * set debug switch of the HORB server for the port. If this
   * switch is on, all threads launced from the HORB server prints stack
   * dump when an uncaught exception occures. This switch can also be
   * turned on by 'horb -v' option.
   * @param port server port.
   * @param sw true or false
   */
  public final void setHORBServerDebug(int port, boolean sw) {
    try {
      HORBServer server = HORBServer.getHORBServer(port);
      server.ss.debug = sw;
    } catch (Exception e){}
  }
  
  /**
   * terminate the HORB server that launched the current thread.
   */
  public final void stopHORBServer() {
    try {
      HORBServer server = HORBServer.getHORBServer();
      server.stopServer();
    } catch (Exception e){}
  }

  /**
   * terminate the HORB server for the port.
   * @param port server port.
   */
  public final void stopHORBServer(int port) {
    try {
      HORBServer server = HORBServer.getHORBServer(port);
      server.stopServer();
    } catch (Exception e){}
  }

  /**
   * terminate the HORB process.
   *
   * @param exitCode this HORB process exit with exit code.
   */
  public final void terminateHORB(int exitCode) {
    System.exit(exitCode);
  }

  /*
   * HORB Object management
   */

  /**
   * returns status of HORB objects.
   */
  public ObjectStatus[] getObjectStatus() {
    return HORBServer.objectTable.getObjectStatus();
  }

  /**
   * returns status of a specific HORB object.
   */
  public ObjectStatus getObjectStatus(String objectID) {
    return HORBServer.objectTable.getObjectStatus(objectID);
  }

  /**
   * unregister a HORB object registered by registerObject().
   * @param objectID  object ID of the object to be unregistered.
   * @exception NoObjectException no such object found
   *
   */
  public void unRegisterObject(String objectID) throws NoObjectException {
    HORBServer.unRegisterObject(objectID);
  }

  /**
   * stop acceptance of connect requests to an object. If one tries
   * to connect the object, the request will fail.
   * @param objectID  object ID to be stopped.
   */
  public void stopObject(String objectID) {
    ObjectInfo oi = HORBServer.objectTable.get(objectID);
    oi.os.stop = true;
  }

  /**
   * resume acception of connections to an object.
   * @param objectID  object ID to be resumed.
   */
  public void resumeObject(String objectID) {
    ObjectInfo oi = HORBServer.objectTable.get(objectID);
    oi.os.stop = false;
  }

  /*
   * HORB thread management
   */
  
  /**
   * returns objectID of the current thread. 
   * Since one daemon object can have multiple threads for
   * requests, several threads can have the same object name.
   * This method intends to be used from a server object (HORB object).
   * At the client side, you can obtain the objectID of a remote object
   * from the proxy object, by using _getObjectURL().
   * <pre>
   *   How to get objectID at client side:
   *
   *	  Foo_Proxy remote = new Foo_Proxy(horbURL);
   *	  HorbURL url = remote._getObjectURL();
   *	  String objectID = url.getObjectID();
   *
   *   How to get objectID at server side:
   *
   *	  HORBAgent agent = new HORBAgent_Impl();
   *	  String objectID = agent.getObjectID();
   * </pre>
   *
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public final String getObjectID() throws HORBException {
    ThreadServer ts = currentThreadServer();
    return ts.ts.objectID;
  }

  /**
   * returns network address of the clinet for this HORB thread.
   * This method intends to be used at server side. So, this does not make
   * sense when this is called remotely.
   *
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public final byte[] getClientNetAddress_Local() throws HORBException {
    return HORBServer.getClientNetAddress();
  }

  /**
   * returns IP address of the clinet for this HORB thread.
   * This method intends to be used at server side. So, this does not make
   * sense when this is called remotely.
   *
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public final InetAddress getClientInetAddress_Local() throws HORBException {
    return HORBServer.getClientInetAddress();
  }

  /**
   * returns hostname of the clinet for this HORB thread. 
   * If hostname lookup fails, this method returns null.
   * This method intends to be used at server side. So, this does not make
   * sense when this is called remotely.
   * @exception HORBException if the current thread  is not a HORB thread.
   */
  public final String getClientHostName_Local() throws HORBException {
    return HORBServer.getClientHostName();
  }

  /**
   * terminate a HORB thread.
   * @param threadName HORB thread to be terminated.
   */
  public final void stopHORBThread(String threadName) throws HORBException {
    HORBServer.threadTable.stop(threadName);
  }

  /**
   * suspend a HORB thread.
   * @param threadName HORB thread to be suspended.
   */
  public final void suspendHORBThread(String threadName) throws HORBException {
    HORBServer.threadTable.suspend(threadName);
  }

  /**
   * resume a HORB thread.
   * @param threadName HORB thread to be resumed.
   */
  public final void resumeHORBThread(String threadName) throws HORBException {
    HORBServer.threadTable.resume(threadName);
  }

  /**
   * set priority of a thread.
   * @param newPriority new priority. Value should be within the range
   * Thread.MIN_PRIORTY,  Thread.MAX_PRIORITY.
   */
  public final void setPriorityHORBThread(String threadName, int newPriority) throws HORBException, IllegalArgumentException {
    HORBServer.threadTable.setPriority(threadName, newPriority);
  }

  /**
   * get priority of a HORB thread.
   */
  public final void getPriorityHORBThread(String threadName) throws HORBException {
    HORBServer.threadTable.getPriority(threadName);
  }

  /**
   * returns status of threads of all HORB objects.
   */
  public ThreadStatus[] getThreadStatus() {
    return HORBServer.threadTable.getThreadStatus();
  }
    
  /**
   * returns status of threads of a HORB object.
   * @param objectID   return threads' status concerning to objectID.
   */
  public ThreadStatus[] getThreadStatus(String objectID) {
    return HORBServer.threadTable.getThreadStatus(objectID);
  }

  /*
   * helper methods
   */

  private ThreadServer currentThreadServer() throws HORBException {
    ThreadServer ts = null;
    try {
      ts = (ThreadServer)Thread.currentThread();
    } catch (ClassCastException e) {
      throw new HORBException("not HORB thread");
    } 
    return ts;
  }

}
