/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.net.*;
import java.io.*;
import java.util.*;

/**
 * HORB Server. Port server and system management.<p>
 *
 * A thread of HORB Server serves a port to accept requests
 * for object creation and thread creation. Multiple ports
 * can be served with multiple HORB servers. If port number is designated
 * to -1, a HORB Server starts without port server. You can use
 * such HORB server for invitation.<p>
 * 
 * When you alter this class, be careful for multithreading with other 
 * HORBServer threads.
 * @see HORB, IOCIService, IOCI
 */
public class HORBServer extends Thread implements Cloneable {

  /*
   * class varialbes. These are common among HORB servers.
   */
  private static HORB horb;

  static boolean debug = false;
  static boolean logging = false;
  static String logFile = null;
  static boolean aclMaster = false;

  /** list of autostart servers */
  static ServerStatus[] autoStartServers;
  /** list of autostart objects */
  static ObjectStatus[] autoStartObjects;
  /** list of access control lists */
  static AclStatus[] acls;
  static int aclRefreshInterval;

  /** list of HORB servers. */
  static ServerTable serverTable; // Hashtable keyed by port

  /** object management table.  */
  static ObjectTable objectTable; // Hashtable keyed by objectID

  /** thread management table. */
  static ThreadTable threadTable; // Hashtable keyed by Thread

  /** access control list. */
  static ACL_Impl acl; 	  // Hashtable keyed by class name
  
  /*
   * per server variables
   */

  /** status and some important variables of this server. */
  ServerStatus ss;

  /** IOCI class for this server. */
  Class IOCIClass;
  /** IOCI to accept requests */
  private IOCI serverIOCI;

  static {
    // create per system object
    try {
      if (objectTable == null) {
	serverTable = new ServerTable();
	objectTable = new ObjectTable();
	threadTable = new ThreadTable();
      }
    } catch (Exception e) {}
  }

  /**
   * HORB.exe HORB Server entry point. This main() instantiates the HORB 
   * object.
   * The HORB object instantiates the  HORBServer object if server feature is
   * requested. This scheme keeps the HORB class small.
   */
  public static void main(String argv[]) throws ClassNotFoundException, 
                    HORBException, InstantiationException, 
                    IllegalAccessException, IOException {

    Config conf = new Config(argv);

    if (horb == null) {
      // initialize the HORB object for the client facility.
      horb = new HORB(false, autoStartServers[0].port, autoStartServers[0].IOCIClassName);
    }

    /*
     * start HORB servers
     */
    for (int i = 0; i < autoStartServers.length; i++) {
      if (autoStartServers[i] != null)
	HORBServer hs = new HORBServer(autoStartServers[i]);
    }

    /*
     * start ACL Master
     */
    if (aclMaster)
      startSystemACLMaster();

    /*
     * autostart
     */
    if (autoStartObjects != null) {
      for (int i = 0; i < autoStartObjects.length; i++) {
	ObjectStatus os = autoStartObjects[i];
	if (os != null && os.className != null && os.objectID != null) {
	  Object obj = null;
	  if (os.implementation != null) // has implementation class
	    obj = Class.forName(os.implementation).newInstance();
	  else
	    obj = Class.forName(os.className).newInstance();
	  registerObject(os.className, obj, os.objectID, os.port);
	  System.out.println(os.className + "("+os.objectID+") started");
	}
      }
    }
  }

  /**
   * Start a new HORB Server for the specified port. This HORB server accepts
   * connect request from clients and serves them. This is a short form
   * of HORBServer(int, String, String);
   *
   * @param port   port number to serve for. If 0, port number is
   *		   got from HORBClient.getPort() to set as default. This means
   *		   service port is same as outgoing port number. If -1,
   *		   a HORB Server is created without a port server. Such
   *		   HORB Server is intended to be used in applets. 
   */
  public HORBServer(int port) throws HORBException {
    this(port, null, null);
  }

  /**
   * Start a new HORB Server for the specified port. This HORB server accepts
   * connect request from clients and serves them.
   *
   * @param port   port number to serve for. If 0, port number is
   *		   got from HORBClient.getPort() to set as default. This means
   *		   service port is same as outgoing port number. If -1,
   *		   a HORB Server is created without a port server. Such
   *		   HORB Server is intended to be used in applets. 
   * @param name   name of this server. The thread name also uses this. 
   *		   If null, a default name is used. The default name
   *		   is "HORBServer(portNo)".
   * @param IOCIClassName classname of an implementation of IOCI. 
   *		   Default IOCI is used if null.
   * @see IOCI, BasicIOCI.
   */
  public HORBServer(int port, String name, String IOCIClassName) throws HORBException {

    ServerStatus ss = new ServerStatus();
    ss.port = port;
    ss.name = name;
    ss.IOCIClassName = IOCIClassName;
    startServer(ss);
  }

  /**
   * Starts a new HORB Server with the specified characteristics in 
   * serverStatus. Although this interface is intended to be internal use,
   * you can use this if you need. Fill in name, IOCIClassName, port, 
   * hostnameLookup, logging, versionCheck and logFile of serverStatus before
   * calling this entry point.
   *
   * @param serverStatus server control structure.
   */
  public HORBServer(ServerStatus serverStatus) throws HORBException {
    startServer(serverStatus);
  }

  private void startServer(ServerStatus serverStatus) throws HORBException {
    ss = serverStatus;

    if (ss.port == 0)
      ss.port = HORBClient.getPort();

    if (ss.name == null)
      ss.name = "HORBServer("+ss.port+")";
    setName(ss.name);		// set thread name of this HORB server

    if (ss.IOCIClassName == null)
      ss.IOCIClassName = HORB.getIOCIClassName();
    try {
      IOCIClass = Class.forName(ss.IOCIClassName);
      serverIOCI = (IOCI)IOCIClass.newInstance();
      ss.IOCI_major_version = serverIOCI.getMajorVersion();
      ss.IOCI_minor_version = serverIOCI.getMinorVersion();
    } catch (Exception e1) {
      throw new HORBException("can't load IOCI class "+ss.IOCIClassName+e1.toString());
    }

    /*
     * start ACL if not started.
     */
    try {
      startACL(acls, aclRefreshInterval);
    } catch (Exception e) {
      System.err.println("ACL initialization failure");
      System.err.println(e);
      System.exit(1);
    }
    
    // start port server thread
    if (ss.port != -1) {
      try {
	// start acceptance of requests
	serverIOCI.serverInit(ss.port);
	start();			// this calls run() in another thread
      } catch (IOException e2) {
	System.err.println("HORB Server initialization failure");
	throw new HORBException(e2);
      }
    }
    ss.startTime = System.currentTimeMillis();
    serverTable.put(new Integer(ss.port), this);
    msg(ss.name + " started");
  }

  /**
   * main loop of HORB Server. This is an internal method.<p>
   * accept connection and start ThreadServer.
   */
  final public void run() {
    Thread.currentThread().setPriority(Thread.NORM_PRIORITY+2);
    while (true) {
      try {
	IOCI ioci = serverIOCI.serverAccept(ss.port);
//System.out.println("accepted");
	if (ioci == null)
	  continue;
	ThreadServer n = new ThreadServer(this, ioci);
	n.start();
      } catch (Exception e) {
	continue;
//	return;
      }
    }
  }

  /**
   * This is an internal method.
   */
  public void invite(HorbURL serverURL, String serverThreadName, HorbURL clientURL, IOCI newIOCI, String username, String key) throws HORBException, IOException {
    // get classname of client
    ObjectInfo oi = objectTable.get(clientURL.getObjectID());
    if (oi == null)
      throw new NoObjectException(clientURL.toString());
    String clientClassName = oi.os.className;

    newIOCI.connectServer(serverURL.getHost(), serverURL.getPort());
    newIOCI.invite2(serverURL, serverThreadName, clientURL, clientClassName, username, key);
    ThreadServer n = new ThreadServer(this, newIOCI);
    n.start();			// start receiving
  }

  void msg(String s) {
    if (ss.debug)
      System.out.println(Thread.currentThread().getName()+": "+s);
  }

  /*
   * system services
   */


  /**
   * stop this HORB server
   */
  void stopServer() {
    try {
      serverIOCI.release();
      serverTable.remove(new Integer(ss.port));
    } catch (Exception e) {}
    stop();
  }

  /**
   * Register the instance of className as a HORB object. The objectID is 
   * the name of the instance. The object can be accessed from any port.<p> 
   * You must have a skeleton class of the className, for example 
   *  horb.package1.Foo_Skeleton.class.
   *
   * <pre>
   *  Example:
   *    horb.package1.Foo foo = new horb.packate1.Foo("hello");
   *    HorbURL url = HORBServer.registerObject("horb.package1.Foo", foo, "foo1");
   *    The object can be accessed from outside via url.
   * </pre>
   *
   * @param className   class name to instantiate prepended package name
   * @param object      an instance of the object to be registered.
   * @param objectID    the name of the instance
   * @return HorbURL    URL of this object.
   * @exception HORBException Illegal objectID, or skeleton class not found
   */
  public static HorbURL registerObject(String className, Object object, String objectID) throws HORBException {
    return registerObject(className, object, objectID, 0);
  }

  /**
   * Register an instance of className for port. The objectID is the name
   * of the instance.<p>
   * You must have a skeleton class of the className, for example 
   *  horb.package1.Foo_Skeleton.class.
   *
   * <pre>
   *  Example:
   *    horb.package1.Foo foo = new horb.packate1.Foo("hello");
   *    HorbURL url = HORBServer.registerObject("horb.package1.Foo", foo, "foo1", 8899);
   *    The object can be accessed from outside via url.
   * </pre>
   *
   * @param className   class name to instantiate prepended package name.
   * @param object      an instance of the object to be registered.
   * @param objectID    the name of the instance.
   * @param port	port number. if not 0, the object is accesible only
   *			from the port. If 0, the object is accessible from
   *			any port.
   * @return HorbURL    URL of this object.
   * @exception HORBException Illegal objectID, or skeleton class not found,
   *			or HORB server is not running for the port.
   */
  public static HorbURL registerObject(String className, Object object, String objectID, int port) throws HORBException {

    // build URL
    HORBServer hs = null;
    int recvPort = port;
    if (port == 0) {
      hs = serverTable.getServer(port);
      recvPort = hs.ss.port;
    }
    String hostName = "localhost";
    try {
       hostName = hs.serverIOCI.getLocalHostName();
    } catch (Exception e) {}
    HorbURL url = new HorbURL(hostName, recvPort, objectID);

    Skeleton skeleton = null;

    //
    // TODO: stop the previous instance
    //

    // unregister the previous instance
    if (objectID == null)
      throw new HORBException("Null ObjectID");
    if (objectTable.get(objectID) != null)
      throw new HORBException("objectID "+objectID+" has been registered already");

    // create instance of the skeleton and the given class
    String skeletonClassName = className+"_Skeleton"; 	// "Foo_Skeleton"
    if (debug)
      System.out.println("HORBServer: creating an instance of " + className + " for port "+port);
    try {
      skeleton = (Skeleton)Class.forName(skeletonClassName).newInstance();
    } catch (Exception e) {
      throw new HORBException(skeletonClassName + " not found. "+e);
    }
    try {
      skeleton.setObject(object);
    } catch (Exception e) {
      throw new HORBException("object is not the type of "+className+" or mismatches with skeleton's type");
    }
    // make remote object reference and register it with the skeleton
    if (debug)
      System.out.println("HORBServer: object "+objectID+" has been created");
    Thread.currentThread().setName(objectID);
    objectTable.put(className, objectID, skeleton, port, true);

    return url;
  }

  /**
   * unregister a server object registered by registerObject().
   * This does not kill the object. Just unregister from tables.
   *
   * @param objectID  object ID of the object to be unregistered.
   * @exception NoObjectException no such object found
   *
   */
  public static void unRegisterObject(String objectID) throws NoObjectException {
    objectTable.remove(objectID);
  }

  /**
   * returns HORB server for the current thread
   * @exception HORBException if the current thread  is not a HORB thread 
   *				or server's already dead.
   */
  public static HORBServer getHORBServer() throws HORBException {
    return serverTable.getServer();
  }

  /**
   * returns HORB server for the port.
   * @param port  server port 
   * @exception HORBException if the current thread  is not a HORB thread 
   *				or server's already dead.
   */
  public static HORBServer getHORBServer(int port) throws HORBException {
    return serverTable.getServer(port);
  }

  /**
   * returns the current thread's priority.
   */
  public final static int getPriority2() {
    // todo: record priority
    return Thread.currentThread().getPriority();
  }

  /**
   * set priority of currently executing thread.
   *
   * @param newPriority    new priority
   * @exception IllegalArgumentException If the priority is not within the range of MIN_PRIORITY and MAX_PRIORITY.
   */
  public final void setPriority2(int pri) throws IllegalArgumentException {
    // todo: record priority
    Thread.currentThread().setPriority(pri);
  }

  /**
   * returns debugging flag of the HORB server for the current thread.
   */
  public static final boolean debugging() {
    try {
      HORBServer server = HORBServer.getHORBServer();
      return server.ss.debug;
    } catch (Exception e){}
    return false;
  }

  private static ThreadServer currentThreadServer() throws HORBException {
    ThreadServer ts = null;
    try {
      ts = (ThreadServer)Thread.currentThread();
    } catch (ClassCastException e) {
      throw new HORBException("not HORB thread");
    } 
    return ts;
  }

  /**
   * returns skeleton object of the current server thread.
   * <pre>
   * Example:
   *   Skeleton skeleton = HORBServer.getSkeleton();
   *   skeleton.accept(0);
   * </pre>
   */
  public static Skeleton getSkeleton() throws HORBException {
    ThreadServer ts = currentThreadServer();
    return ts.skeleton;
  }

  /**
   * returns current client's hostname.
   * @return client's hostname.
   */
  public final static String getClientHostName() throws HORBException {
    ThreadServer ts = currentThreadServer();
    if (ts.ts.clientName != null)
      return ts.ts.clientName;
    return ts.ioci.getHostName();
  }

  /**
   * returns current client's IP address.
   * @return InetAddress client's IP address.
   */
  public final static InetAddress getClientInetAddress() throws HORBException {
    ThreadServer ts = currentThreadServer();
    try {
      BasicIOCI bi = (BasicIOCI)ts.ioci;
      return bi.getInetAddress();
    } catch (Throwable t) {
      return null;		// not IP
    }
  }

  public final static byte[] getClientNetAddress() throws HORBException {
    ThreadServer ts = currentThreadServer();
    return ts.ioci.getAddress();
  }

  /**
   * returns local hostname.
   * @return local hostname.
   */
  public final static String getLocalHostName() throws HORBException, IOException {
    ThreadServer ts = currentThreadServer();
    return ts.ioci.getLocalHostName();
  }

  /**
   * returns network address of the local host in byte array.
   * @return network address of the local host.
   */
  public final static byte[] getLocalNetAddress() throws HORBException, IOException {
    ThreadServer ts = currentThreadServer();
    return ts.ioci.getLocalAddress();
  }

  /**
   * returns IP address of the local host.
   * @return IP address of the local host.
   */
  public final static InetAddress getLocalInetAddress() throws HORBException {
    ThreadServer ts = currentThreadServer();
    try {
      BasicIOCI bi = (BasicIOCI)ts.ioci;
      return bi.getLocalInetAddress();
    } catch (Throwable t) {
      return null;		// not IP
    }
  }

  /**
   * return the current port connecting to the client.
   */
  public final static int getCurrentPort() throws HORBException {
    ThreadServer ts = currentThreadServer();
    return ts.ts.port;
  }

  /**
   * return username of the client. null means anonymous access.
   */
  public final static String getUsername() throws HORBException {
    ThreadServer ts = currentThreadServer();
    return ts.ts.userName;
  }

  /**
   * return the current IOCI object of the current thread.
   */
  public final static IOCIService getIOCIService() throws HORBException {
    ThreadServer ts = currentThreadServer();
    return (IOCIService)ts.ioci;
  }

  /**
   * start an ACL server for this system. This method works only once.
   * Making an instance of HORBServer calls this method internally.
   * If your program runs not from the HORB command and create
   * HORB Server(s) and you need ACL, call this method before creating
   * the HORB Servers. If you don't call this interface, HORB server
   * starts a "pass through" ACL server. If you want access control,
   * fill in the source field of at least one AclStatus and pass them
   * to this method. Call HORBServer.startSystemACLMaster() if you want to
   * make this ACL a master ACL.
   * <pre>
   *	AclStatus acl = new AclStatus();
   *	acl.source = "client.acl";
   *	AclStatus[] acls = new AclStatus[1];
   *	acls[0] = acl;
   *	HORBServer.startACL(acls, 0);
   *	HORBServer hs = new HORBServer(8886, null, null);
   *	// HORBServer.startSystemACLMaster(); // if you want
   * </pre>
   *
   * @param acls array of ACL statuses. The source fields are mandatory.
   * @param aclRefreshInterval ACL refresh interval in seconds. If 0, no
   *				refresh occurs.
   * @exception HORBException Exception occured while loading the ACL files.
   * @exception IOException I/O Exception occured while loading the ACL files.
   * @see HORBServer#startACLMaster
   */
  public static final void startACL(AclStatus[] acls, int aclRefreshInterval) throws HORBException, IOException {
    if (acl == null) {
      String hostname = "localhost";
      try {
	 hostname = InetAddress.getLocalHost().getHostName();
      } catch (Exception e) {}
      acl = new ACL_Impl(acls, hostname, aclRefreshInterval);
    }
  }

  /**
   * return sytem ACL server.
   */
  public static final ACL getSystemACL() {
    return acl;
  }

  /**
   * start ACL master server.
   */
  public final static void startSystemACLMaster() throws HORBException, IOException {
    acl.startACLMaster_Local(ACL.SYSTEM_ACL_MASTER);
  }

  /**
   * return registered object having the requested objectID.
   *
   * @param objectID objectID of the object.
   * @exception NoObjectException 
   */
  public final static Object getObject(String objectID) throws HORBException {
    ObjectInfo oi = objectTable.get(objectID);
    //  oi may be null.
    //    if (oi == null)
    //      throw new NoObjectException("object not registered");
    try {
      return oi.skeleton.getObject();
    } catch (Exception e) {
      throw new NoObjectException("object not registered");
    }
  }
}
