/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

/**
 * URL (Universal Resource Locator) for HORB. HorbURL is used to represent a HORB object.
 */
public class HorbURL {
  protected String protocol = "horb";
  protected String host = "localhost";
  protected String objectID;
  protected String ref;
  protected int port = 0;

  /**
   * This constructor creates default URL. That is "horb://localhost/".
   */
  public HorbURL() {
  }

  /**
   * Create an URL for an existing or an unexisting HORB object.
   *
   * @param     host    host name of the remote object. If null or "" or "-", localhost is assumed.
   * @param	port	TCP/IP port number at the remote host.
   * @param	objectID  objectID of the remote object. This can be null
   *			when creates a new remote object. ref can be appended, 
   *			for example object#ref1.
   */
  public HorbURL(String host, int port, String objectID) {
    if (host == null || host.equals("-") || host.length() == 0)
      host = "localhost";
    this.host = host;
    this.port = port;
    parseObjectID(objectID);
  }

  /**
   * Create an URL for an existing or an unexisting HORB object.
   * Default port number is used.
   *
   * @param     host    host name of the remote object. If null or "" or "-", localhost is assumed.
   * @param	objectID  objectID of the remote object. This can be null
   *			when creates a new remote object. ref can be appended, 
   *			for example object#ref1.
   */
  public HorbURL(String host, String objectID) {
    if (host == null || host.equals("-") || host.length() == 0)
      host = "localhost";
    this.host = host;
    parseObjectID(objectID);
  }

  /**
   * Create an URL for an existing or an unexisting HORB object.
   * Note that url should include "protocol://" in it,
   * for example "file:///dir/file". "file:/dir/file" is not allowed.
   * <pre>
   *   URL syntax:
   *      <protocol>://[<hostname>[:<portNo>]] [/] [<objectID> [#<ref>]]
   * </pre>
   *
   *      If <hostname> is '-' or omitted, localhost is assumed.
   *      If <portNo> is omitted, default port number is assumed.
   *      If <objectID> is omitted, HORB assumes the user wants to create a new instance. getObjectID() returns null.
   * <pre>
   *   Examples:
   *      horb://www.etl.go.jp:8887/directory
   *      horb://www.etl.go.jp/directory
   *      horb://www.etl.go.jp/
   *      horb://-/directory
   *	  horb://-/
   * </pre>
   * default values are;<p>
   * <pre>
   *      protocol = "horb"
   *      host = "localhost"
   *      port = 0
   *      objectID = null
   *      ref = null
   * </pre>
   *
   * @param     url       URL of the object.
   */
  public HorbURL(String url) throws HORBException {
    setURL(url);
  }

  /**
   * create a new URL from existing URL.
   *
   * @param    url	 a HORB url. hostname and portNo are used.
   */
  public HorbURL(HorbURL url) {
    this.protocol = url.getProtocol();
    this.host = url.getHost();
    this.port = url.getPort();
    this.objectID = url.getObjectID();
    this.ref = url.getRef();
  }

  /**
   * create a new URL from existing URL and an objectID.
   *
   * @param    url	 a HORB url. hostname and portNo are used.
   * @param    objectID  objectID for this URL. ref can be appended, 
   *			for example object#ref1.
   */
  public HorbURL(HorbURL url, String objectID) {
    this.protocol = url.getProtocol();
    this.host = url.getHost();
    this.port = url.getPort();
    parseObjectID(objectID);
  }

  /**
   * Returns true if two URLs are equal.
   *
   * @param    url	 a HORB url. 
   */
  public boolean equals(HorbURL url) {
    return (this.host == url.getHost() &&
	    this.protocol == url.getProtocol() &&
	    this.port == url.getPort() &&
	    this.objectID == url.getObjectID() &&
	    this.ref == url.getRef()
	    ) ? true : false;
  }

  /**
   * Returns Protcol.
   */
  public final String getProtocol() {
    return protocol;
  }
  /**
   * Set Protcol name.
   * @param protocol  protocol name to set.
   */
  public final void getProtocol(String protocol) {
    this.protocol = protocol;
  }

  /**
   * Returns HostName.
   */
  public final String getHost() {
    return host;
  }
  /**
   * Set HostName.
   * @param host   hostname
   */
  public final void setHost(String host) {
    this.host = host;
  }

  /**
   * Return objectID.
   */
  public final String getObjectID() {
    return objectID;
  }
  /**
   * Set objectID.
   * @param objectID   objectID to set.
   */
  public final void setObjectID(String objectID) {
    this.objectID = objectID;
  }

  /**
   * Return ref.
   */
  public final String getRef() {
    return ref;
  }
  /**
   * Set ref.
   * @param ref   ref to set.
   */
  public final void setRef(String ref) {
    this.ref = ref;
  }
  /**
   * Returns PortNo.
   */
  public final int getPort() {
    return port;
  }
  /**
   * set PortNo.
   */
  public final void setPort(int port) {
    this.port = port;
  }
  /**
   * Returns string representation of this URL. This is equivalent to toString().
   */
  public final String getURL() {
    return toString();
  }
 
  /**
   * Returns string representation of this URL.
   */
  public String toString() {
    String oid = (objectID == null) ? "" : objectID;
    String p = "file".equals(protocol) ? "" : ":"+port;
    String o = (port == 0) ? "" : ""+port;
    String r = (ref == null) ? "" : ("#"+ref);
    String h = (host == null) ? "" : host;
    return(protocol+"://"+h+p+"/"+oid+r);
  }

  /**
   * parse and set URL.<p> Note that url should include "protocol://" in it,
   * for example "file:///dir/file". "file:/dir/file" is not allowed.
   *
   * default values are;<p>
   * <pre>
   *    protocol = "horb"
   *    host = "localhost"
   *    port = 0
   *    objectID = null
   *	ref = null
   * </pre>
   * @param url   URL to set.
   */
  public void setURL(String url) throws HORBException {
    port = 0;
    host = "localhost";
    protocol = "horb";
    objectID = null;
    ref = null;

    int i, j;
    String t;

    if ((i = url.indexOf("://")) < 0)
      throw new HORBException("invalid URL (no protocol://)" );
    protocol = url.substring(0, i);
    if ((j = url.indexOf('/', i+3)) >= 0) { // host/objectID
      t = url.substring(i+3, j);
      if (url.length() > j+1) {
	objectID = url.substring(j+1);
	if ((j = objectID.indexOf('#')) >= 0) {// oid#ref
	  ref = objectID.substring(j+1);
	  if (j > 0)
	    objectID = objectID.substring(0, j);
	  else
	    objectID = null;
	}
      }
    } else
      t = url.substring(i+3);
    
    if ((i = t.indexOf(':')) >= 0) {
      host = t.substring(0, i);
      port = Integer.parseInt(t.substring(i+1));
    } else
      host = t;
    if (host.length() == 0 || host.equals("-"))
      host = "localhost";
  }

  void parseObjectID(String oid) {
    int j;
    objectID = oid;
    if (oid == null)
      return;
    if ((j = oid.indexOf('#')) >= 0) {// oid#ref
      ref = oid.substring(j+1);
      if (j > 0)
	objectID = oid.substring(0, j);
      else
	objectID = null;
    }
  }

/*
  public static void main(String argv[]) {
    HorbURL url = new HorbURL(argv[0]);
    System.out.println(url);
  }
*/
}

