/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.io.DataOutputStream;
import java.io.DataInputStream;
import java.io.IOException;

/**
 * Definition of Inter-Object Communication Interface. <em>User accessible
 * methods are separated in IOCIService. If you are a HORB programmer,
 * you don't concern with this interface.</em><p>
 * This interface defines APIs between the lower transport layer and the
 * upper application layer (i.e. HORB). Thus this interface
 * is responsible for the session management and presentation of data
 * on network.<p>
 * APIs can be separated into two parts. One is common to every
 * IOCI implementations, such as upper level connection creation and
 * object passing. This is protocol independent. 
 * The other is protocol dependent and protocol specific. For example,
 * making new network connection, sending a integer, and obtaining
 * IP address.<p>
 * The common part is separated in the class IOCICommon. The protocol
 * dependent part may inherit IOCICommon easily to comply to the
 * IOCI. But you don't need to do so, if you don't want to.<p>
 * APIs in the IOCI are also categolized into server side, client side,
 * both sides. A HORB server serves a port. The meaning of port
 * depends on the implementations of IOCI. BasicIOCI, for example,
 * is an implementation for TCP/IP and the socket interface, thus
 * the port means socket number. You can use port number as other meanings,
 * such like ATM channel number.
 */
public interface IOCI extends IOCIService {

  /*
   * constants for create instance
   */

  /** create a new object. */
  public static final short ORB_CREATE_INSTANCE = 0;
  /** connect to existent object. */
  public static final short ORB_CONNECT_TO_INSTANCE = 1;
  /** disconnect to the object. */
  public static final short ORB_CLOSE = 2;
  /** invite a server object to access a client object. */
  public static final short ORB_INVITE_TO_INSTANCE = 3;

  /*
   * preamble
   */

  /** preamble for each remote method call */
  public static final short PREAMBLE = (short)/*P*/0xeace;

  /*
   * status codes
   */

  /** method call ended with no error */
  public static final short STAT_NO_ERROR = 0;
  /** requested object or class not found */
  public static final short STAT_NO_SUCH_OBJECT = 1;
  /** requested method not found */
  public static final short STAT_NO_SUCH_METHOD = 2;
  /** exception occured during execution of the method */
  public static final short STAT_EXCEPTION_IN_METHOD = 3;
  /** exception occured during sending arguments */
  public static final short STAT_EXCEPTION_IN_ARGUMENT = 4;
  /** connection is not syncronized. reconnect! */
  public static final short STAT_PREAMBLE_MISMATCH = 5;
  /** access is denied */
  public static final short STAT_PERMISSION_ERROR = 6;
  /** IOCI major version mismatch  */
  public static final short STAT_IOCI_VERSION_MISMATCH = 7;
  /** requsted IOCI class not found */
  public static final short STAT_IOCI_NOT_FOUND = 8;
  /** class major version mismatch  */
  public static final short STAT_CLASS_VERSION_MISMATCH = 9;

  /*
   * constants for object passing
   */

  /** null instance is passed. */
  public static final byte OBJ_NULL = 0;
  /**  remote object reference is passed. */
  public static final byte OBJ_REF = 1; 
  /**  usual instance is passed. */
  public static final byte OBJ_INST = 2;
  /**  usual instance, but casted to something is passed. */
  public static final byte OBJ_CAST = 3;
  /** looped object is passed. this object was passed once */
  public static final byte OBJ_LOOPY = 4;
  /** no proxy class was found. object was ignored */
  public static final byte OBJ_NO_PROXY = 5;
  /** String is passed. */
  public static final byte OBJ_STRING = 6;

  /*
   * source/destication of object passing.
   */
  /** object is transfered to/from network. */
  public static final byte LOC_NETWORK = 0;
  /** object is transfered to/from file. */
  public static final byte LOC_FILE = 1;

  /*
   * functions for server side
   */
  /** initialize IOCI for server port.
   * @param port port to serve by this HORB server.
   */
  void serverInit(int port) throws IOException;
  /** 
   * accept new connection request.
   * @param port port to listen
   * @return new IOCI object.
   */
  IOCI serverAccept(int port) throws IOException;
  /** receive connect request from client. This is the counter part of connectServer(). */
  boolean recvConnectServer() throws IOException;
  /** receive a preamble */
  short recvPreamble() throws IOException ;
  /** receive a class number */
  short acceptClassNo() throws IOException;
  /** receive a method number */
  short acceptMethod() throws IOException;
  /** receive the number of arguments for a method */
  short getNumArgs() throws IOException;
  /** send status of method call. */
  void sendStatus(short status)  throws IOException;

  /*
   * functions for client side
   */
  /**
   * connect to an object.
   * @param major major version number. Mismatch causes exception.
   * @param minor minor version number.
   * @param username username. null if anonymous user.
   * @param passwd password.
   */
  HorbURL connect(HorbURL url, String className, short major, short minor, String username, String passwd) throws HORBException, NetException;

  /**
   * create a new remote object.
   * @param url   HorbURL of the remote machine. This may not contain
   *		  objectID.
   * @param className class name of the object
   * @param major major version number. Mismatch causes exception.
   * @param minor minor version number.
   * @param username username. null if anonymous user.
   * @param passwd password.
   */
  HorbURL createObject(HorbURL url, String className, short major, short minor, String username, String passwd)
    throws HORBException, NetException;

  /**
   * receiver part of createObject().
   */
  NetIOCIInfo recvCreateObject() throws IOException;

  /**
   * connect to an existing remote object.
   *
   * @param url   HorbURL of the remote machine. objectID must be contained.
   * @param className class name of the object
   * @param major major version number. Mismatch causes exception.
   * @param minor minor version number.
   * @param username username. null if anonymous user.
   * @param passwd password.
   */
  HorbURL connectObject(HorbURL url, String className, short major, short minor, String username, String passwd)
    throws NetException, HORBException;

  /**
   * receiver part of connectObject().
   */
  NetIOCIInfo recvConnectObject() throws IOException;

  /** select a remote method */
  void selectMethod(int classNo, int methodNo, int numArgs)
    throws IOException ;
  /** receive a status of execution */
  short recvStatus()  throws IOException, HORBException;
  /** receive a status of execution without status check */
  short recvStatusNoCheck()  throws IOException;
  
  /**
   * raise exception according to status 
   * @param status status to check. This must be one of IOCI.STAT_*.
   * @param str this string is prepended to an exception message.
   */
  void raiseIfException(short status, String str) throws HORBException;

  /*
   * functions for both server and client
   */


  /** return signature of this IOCI. range 0 to 0x7fff are reserved.
   * If an IOCI is compatible with BasicIOCI, you can use the same signature.
   * subtypes can be distinguished by options.
   */
  public short getSignature();
  /** return major version number of this IOCI. */
  public short getMajorVersion();
  /** return minor version number of this IOCI. */
  public short getMinorVersion();
  /** connect to a server. */
  void connectServer(String host, int port) throws IOException;
  /** send signature of IOCI */
  void sendIOCISignature() throws IOException;
  /** invite a server object to a client object. */
  void invite(HorbURL serverURL, HorbURL clientURL, String username, String passwd) throws HORBException, NetException;
  /** invite a server object to a client object, second part.*/
  void invite2(HorbURL serverURL, String serverThreadName, HorbURL clientURL, String clientClassName, String username, String passwd) throws HORBException, IOException;
  /** connect to the client. */
  void invited(String className, HorbURL clientURL, short major, short minor, String username, String passwd) throws HORBException, NetException;
  /** receiver part of invite(). */
  NetIOCIInfo recvInvite() throws IOException;

  void startAsyncHandler();
  void waitReceive(long timeout) throws InterruptedException, HORBException;
  boolean isAsyncMethodEnd();
  boolean waitNoReceive(long timeout) throws InterruptedException;
  void setHandler(AsyncMethodHandler handler, int tag);

  /** issue a request. This is called after sending arguments. */
  void request()  throws IOException;
  /** flush the output buffer to cause actual send */
  void kick()  throws IOException;

  /**
   * Sends a object. The type of object is one of null object, remote object
   * reference, or  casted(narrowed) object.
   * Returns true if the true class of 'o' is equal to 'expectedClass',
   * Otherwise returns false. If true, the caller must send object
   * by itself, otherwise need not.
   *
   * @param o		Object to be passed.
   * @param expectedClassName Name of expected class to be passed. 
   * If this class is equal to the class of parameter 'o', _sendInstance() 
   * of proxy class of 'expectedClassName' is used.
   * @param loopy	looping object checker.
   *
   * @exception ProxyException couldn't instantiate a proxy object
   * @exception HORBException null object reference
   * @exception IOException network error
   */
  public void sendObject(Object o, String expectedClassName, Loopy loopy)
    throws HORBException, IOException;

  /**
   * Receives an object. First receive what type of object (e.g. null object,
   * object reference), and then receive object. If the the class of
   * the receiving object is equal to 'expectedClassName', it's receiver
   * class (expectedClassName_Proxy) is used to pass it in.
   *
   * @param expectedClassName Name of expected class to be passed in.
   * @param gb  another side of loop checker.
   *
   * @exception ProxyException couldn't instantiate a proxy object
   * @exception HORBException illegal tag type is detected
   * @exception IOException network error
   */   
  public Object recvObject(String receiverClassName, Goldberg gb) 
    throws HORBException, IOException;

  /** send a boolean */
  void sendBoolean(boolean value)throws IOException;
  /** send an array of booleans */
  void sendBooleanArray(boolean value[])throws IOException;
  /** send a byte */
  void sendByte(byte value)throws IOException;
  /** send an array of bytes */
  void sendByteArray(byte value[])throws IOException;
  /** send a char */
  void sendChar(char value)throws IOException;
  /** send an array of chars */
  void sendCharArray(char value[])throws IOException;
  /** send a short */
  void sendShort(short value)throws IOException;
  /** send an array of shorts */
  void sendShortArray(short value[])throws IOException;
  /** send an int */
  void sendInt(int value)throws IOException;
  /** send an array of ints */
  void sendIntArray(int value[])throws IOException;
  /** send a long */
  void sendLong(long value)throws IOException;
  /** send an array of longs */
  void sendLongArray(long value[])throws IOException;
  /** send a float */
  void sendFloat(float value)throws IOException;
  /** send an array of floats */
  void sendFloatArray(float value[])throws IOException;
  /** send a double */
  void sendDouble(double value)throws IOException;
  /** send an array of doubles */
  void sendDoubleArray(double value[])throws IOException;
  /** send a string */
  void sendString(String value)throws IOException;
  /** send an array of Strings */
  void sendStringArray(String value[])throws IOException;

  /** receive a boolean */
  boolean recvBoolean() throws IOException;
  /** receive an array of booleans */
  boolean[] recvBooleanArray()throws IOException;
  /** receive a byte */
  byte recvByte() throws IOException;
  /** receive an array of bytes */
  byte[] recvByteArray()throws IOException;
  /** receive a char */
  char recvChar() throws IOException;
  /** receive an array of chars */
  char[] recvCharArray()throws IOException;
  /** receive a short */
  short recvShort() throws IOException;
  /** receive an array of shorts */
  short[] recvShortArray() throws IOException;
  /** receive an int */
  int recvInt() throws IOException;
  /** receive an array of ints */
  int[] recvIntArray() throws IOException;
  /** receive a long */
  long recvLong() throws IOException;
  /** receive an array of longs */
  long[] recvLongArray() throws IOException;
  /** receive a float */
  float recvFloat() throws IOException;
  /** receive an array of floats */
  float[] recvFloatArray() throws IOException;
  /** receive a double */
  double recvDouble() throws IOException;
  /** receive an array of doubles */
  double[] recvDoubleArray() throws IOException;
  /** receive a string */
  String recvString() throws IOException;
  /** receive an array of strings */
  String[] recvStringArray() throws IOException;
}
