/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.io.*;
import java.net.*;

/**
 * protocol independent part of IOCI. Implementations of IOCI
 * must inherit this class. <em>Programmers must use IOCIService
 * interface to access IOCI related services. Don't access
 * this class directly.</em>
 * @see IOCIService
 */
public abstract class IOCICommon implements Runnable {
  String threadName;
  protected boolean loopyCheck = true;
  protected boolean strict = true;
  protected boolean debug = false;
  protected int port;
  protected HorbURL url;
  /*
   * the following four variables are accessed from ACL_Impl.java
   */
  protected String host;
  protected byte[] netAddress;
  protected String username;
  protected String passwd;

  /* for asynchronous transfer */
  boolean come;
  short status;
  Thread asyncHandler;
  AsyncMethodHandler userAsyncHandler;
  int asyncHandlerTag;

  /**
   * connect to object.
   * @param url   HorbURL of the remote machine.
   * @param className class name of the object.
   * @param major major version number. Mismatch causes exception.
   * @param minor minor version number.
   * @param username username. null if anonymous user.
   * @param passwd password.
   * @exception HORBException object not found in remote system, or can't create IOCI
   * @exception NetException Network error
   */
  public HorbURL connect(HorbURL url, String className, short major, short minor, String username, String passwd) throws HORBException, NetException {
    if (url.getObjectID() == null)
      return createObject(url, className, major, minor, username, passwd);
    else
      return connectObject(url, className, major, minor, username, passwd);
  }
  
  /**
   * create a new remote object.
   * @param url   HorbURL of the remote machine. This may not contain
   *		  objectID.
   * @param className class name of the object
   * @param major major version number. Mismatch causes exception.
   * @param minor minor version number.
   * @param username username. null if anonymous user.
   * @param passwd password.
   */
  public HorbURL createObject(HorbURL url, String className, short major, short minor, String username, String passwd)
    throws HORBException {
    this.username = username;
    this.passwd = passwd;
    if (url.getPort() == 0)
      url.setPort(HORB.getPort());
     
    try {
      host = url.getHost();
      port = url.getPort();
      // connect to host
      connectServer(host, port);

      // create server object in the server
      sendShort(IOCI.ORB_CREATE_INSTANCE); // create instance
      sendShort((short)0);		   // option
      sendString(className);
      sendShort(major);
      sendShort(minor);
      sendString(username);
      sendString(passwd);
      kick();

      // wait for status of the bind request
      short stat = recvShort();
      //    System.out.println("status = " + stat);
      if (stat != IOCI.STAT_NO_ERROR) {
	release();
	raiseIfException(stat, "connect fail (" + className + " at "+url+") ");
	return null;
      } else {
	this.url = new HorbURL(url, recvString());
	threadName = recvString();
	return this.url;
      }
    } catch (IOException ioe) {
      release();
      throw new NetException(ioe);
    }
  }

  /**
   * receiver part of createObject().
   */
  public NetIOCIInfo recvCreateObject() throws IOException {
    NetIOCIInfo i = new NetIOCIInfo();
    i.option = recvShort();
    i.className = recvString();
    i.major_version = recvShort();
    i.minor_version = recvShort();
    i.username = recvString();
    this.username = i.username;
    i.passwd = recvString();
    this.passwd = i.passwd;
    return i;
  }

  /**
   * connect to an existing remote object.
   *
   * @param url   HorbURL of the remote machine. objectID must be contained.
   * @param className class name of the object
   * @param major major version number. Mismatch causes exception.
   * @param minor minor version number.
   * @param username username. null if anonymous user.
   * @param passwd password.
   */
  public HorbURL connectObject(HorbURL url, String className, short major, short minor, String username, String passwd) throws HORBException {

    this.username = username;
    this.passwd = passwd;

    if (url.getPort() == 0)
      url.setPort(HORB.getPort());
    try {
      // connect to host
      port = url.getPort();
      if (port == 0)
	port = HORB.getPort();
      host = url.getHost();

      connectServer(host, port);

      // create server object in the server
      sendShort(IOCI.ORB_CONNECT_TO_INSTANCE);
      sendShort((short)0);		// option
      sendString(url.getObjectID());
      sendString(className);
      sendShort(major);
      sendShort(minor);
      sendString(username);
      sendString(passwd);
      kick();

      // wait for status of the bind request
      short stat = recvShort();
      //    System.out.println("status = " + stat);
      if (stat != IOCI.STAT_NO_ERROR) {
	release();
	raiseIfException(stat, "connect fail ("+url+") ");
	return null;
      }
      threadName = recvString();
      this.url = url;
      return url;
    } catch (IOException ioe) {
      release();
      throw new NetException(ioe);
    }
  }

  /**
   * receiver part of connectObject().
   */
  public NetIOCIInfo recvConnectObject() throws IOException {
    NetIOCIInfo i = new NetIOCIInfo();
    i.option = recvShort();
    i.objectID = recvString();
    i.className = recvString();
//System.out.println(i.className);
    i.major_version = recvShort();
    i.minor_version = recvShort();
    i.username = recvString();
//System.out.println(i.username);
    this.username = i.username;
    i.passwd = recvString();
//System.out.println(i.passwd);
    this.passwd = i.passwd;
    return i;
  }

  /** 
   * invite a server object to a client object. In order to prevent
   * someone to use invitation as a covert channel, here is key checking, too.
   */
  public void invite(HorbURL serverUrl, HorbURL clientURL, String username, String passwd) throws HORBException {
    HORBServer horbServer = HORBServer.getHORBServer(clientURL.getPort());
    IOCI newIOCI = null;
    if (username == null)
      username = this.username;
    if (passwd == null)
      passwd = this.passwd;
    try {
      newIOCI = (IOCI)this.getClass().newInstance();
    } catch (Exception e) {
      throw new IOCIException(e);
    }
    try {
      horbServer.invite(serverUrl, threadName, clientURL, newIOCI, username, passwd);
    } catch (IOException e2) {
      throw new NetException(e2);
    }
  }

  /** invite a server object to a client object, second part.*/
  public void invite2(HorbURL serverURL, String serverThreadName, HorbURL clientURL, String clientClassName, String username, String passwd) throws HORBException, IOException {
    this.username = username;
    this.passwd = passwd;
    sendShort(IOCI.ORB_INVITE_TO_INSTANCE);
    sendShort((short)0);			// option
    sendString(serverThreadName);
    sendString(clientURL.toString());
    sendString(clientClassName);
    sendString(username);
    sendString(passwd);
    kick();
/*
System.out.println("IOCIcommon.invite2 sent");
    int status = recvStatus();
System.out.println("IOCIcommon.invite2 status = "+status);
    if (status != IOCI.STAT_NO_ERROR)
      throw new NoObjectException("invite "+serverURL);
*/
  }

  
  /**
   * receiver part of invite().
   */
  public NetIOCIInfo recvInvite() throws IOException {
    NetIOCIInfo i = new NetIOCIInfo();
    i.option = recvShort();
    i.threadName = recvString(); // server's threadName
    i.clientURL = recvString(); // client's URL
    i.className = recvString();    // client's className
    i.username = recvString();
    this.username = i.username;
    i.passwd = recvString();
    this.passwd = i.passwd;
    return i;
  }

  public void invited(String className, HorbURL clientURL, short major, short minor, String username, String passwd) throws HORBException, IOException {
    this.username = username;
    this.passwd = passwd;
    try {
      sendIOCISignature();
      sendShort(IOCI.ORB_CONNECT_TO_INSTANCE);
      sendShort((short)0);		// option
      sendString(clientURL.getObjectID());
      sendString(className);
      sendShort(major);
      sendShort(minor);
      sendString(username);
      sendString(passwd);
      kick();

      // wait for status of the bind request
      short stat = recvShort();
      if (stat != IOCI.STAT_NO_ERROR) {
	release();
	throw new HORBException("connect fail ("+clientURL+") "+stat);
      }
      threadName = recvString();
    } catch (IOException ioe) {
      release();
      throw ioe;
    }
  }

  /*
   * methods for asynchronous method call
   */
  public final void startAsyncHandler() {
    come = false;
    asyncHandler = new Thread(this);
    asyncHandler.setPriority(Thread.MAX_PRIORITY);
    asyncHandler.start();
  }

  public final void waitReceive(long timeout) throws InterruptedException, HORBException {
    if (waitNoReceive(timeout) == false)
      throw new HORBException("timeout");
    come = false;
    raiseIfException(status, "");
  }

  /**
   * receive status with blocking for an asynchronous method call.
   */
  public void run() {
    try {
      status = recvStatusNoCheck();
    } catch (Exception e) {}
    waitNotify();
    if (userAsyncHandler != null)
      userAsyncHandler.run(asyncHandlerTag);
  }

  public synchronized void waitNotify() {
    come = true;
    notify();
  }

  public final boolean isAsyncMethodEnd() {
    return come;
  }

  public final synchronized boolean waitNoReceive(long timeout) throws InterruptedException {
    if (come == false)
      wait(timeout);
    return come;
  }
  
  public final void setHandler(AsyncMethodHandler handler, int tag) {
    userAsyncHandler = handler;
    asyncHandlerTag = tag;
  }
    
  /*
   * methods for sending controls, receiving status
   */

  public final void selectMethod(int classNo, int methodNo, int numArgs) throws IOException {
    sendShort(IOCI.PREAMBLE);
    sendShort((short)classNo);
    sendShort((short)methodNo);
    sendShort((short)numArgs);
  }

  public final short recvPreamble() throws IOException {
    return recvShort();
  }
  public final short acceptClassNo() throws IOException {
    return recvShort();
  }
  public final short acceptMethod() throws IOException {
    return recvShort();
  }
  public final short getNumArgs() throws IOException {
    return recvShort();
  }

  public final void request() throws IOException {
    kick();
  }

  /** send status of method call. */
  public final void sendStatus(short status) throws IOException {
    sendShort(status);
  }

  /** receive a status of execution */
  public final short recvStatus() throws IOException, HORBException {
    short status = recvShort();
    raiseIfException(status, "");
    return status;
  }

  /** receive a status of execution without status check */
  public final short recvStatusNoCheck() throws IOException {
    return recvShort();
  }

  /**
   * raise exception according to status 
   * @param status status to check. This must be one of IOCI.STAT_*.
   * @param str this string is prepended to an exception message.
   */
  public final void raiseIfException(short status, String s) throws HORBException {
    switch (status) {
    case IOCI.STAT_NO_SUCH_OBJECT:
      throw new NoObjectException(s+"no such object");
    case IOCI.STAT_NO_SUCH_METHOD:
      throw new NoMethodException(s+"no such method");
    case IOCI.STAT_EXCEPTION_IN_METHOD:
      String err = "exception in remote method";
      try {
	err = recvString();
      } catch (IOException e) {}
      throw new RemoteException(err);
    case IOCI.STAT_EXCEPTION_IN_ARGUMENT:
      err = "exception during passing arguments";
      try {
	err = recvString();
      } catch (IOException e) {}
      throw new ArgumentException(err);
    case IOCI.STAT_PREAMBLE_MISMATCH:
      throw new IOCIException(s+"preamble mismatch");
    case IOCI.STAT_PERMISSION_ERROR:
      throw new NoPermissionException(s+"no access permission");
    case IOCI.STAT_IOCI_VERSION_MISMATCH:
      throw new IOCIException(s+"IOCI major version mismatch");
    case IOCI.STAT_IOCI_NOT_FOUND:
      throw new IOCIException(s+"requested IOCI class not found");
    case IOCI.STAT_CLASS_VERSION_MISMATCH:
      throw new IOCIException(s+"class major version mismatch");
    }
  }


  /**
   * Sends a object. The type of object is one of null object, remote object
   * reference, or  casted(narrowed) object.
   * Returns true if the true class of 'o' is equal to 'expectedClass',
   * Otherwise returns false. If true, the caller must send object
   * by itself, otherwise need not.
   *
   * @param o		Object to be passed.
   * @param expectedClassName Name of expected class to be passed. 
   * If this class is equal to the class of parameter 'o', _sendInstance() 
   * of proxy class of 'expectedClassName' is used.
   * @param loopy	looping object checker.
   * @param place	IOCI.LOC_*. for example, LOC_FILE, LOC_NETWORK..
   *
   * @exception ProxyException couldn't instantiate a proxy object
   * @exception HORBException null object reference
   * @exception IOException network error
   */
  public void sendObject(Object o, String expectedClassName, Loopy loopy, byte place) throws HORBException, IOException, ProxyException {

    int num;

    //
    // check if null object
    //
    if (o == null) {
      sendByte(IOCI.OBJ_NULL);
      return;
    }

    // loopy check
    if (loopyCheck && (num = loopy.check(o)) >= 0) {
      sendByte(IOCI.OBJ_LOOPY);
      sendInt(num);
      return;
    }

    // get real classname
    Class realClass = o.getClass();
    String realClassName = realClass.getName();
    if ("java.lang.String".equals(realClassName)) { // string?
      sendByte(IOCI.OBJ_STRING);
      sendString((String)o);
      return;
//    } else if (realClassName.endsWith("_Proxy")) {	// object reference
    } else if (o instanceof Proxy) {	// object reference
      //
      // remote object reference
      //
      Proxy p = (Proxy)o;
      HorbURL url = p._getObjectURL();
      sendByte(IOCI.OBJ_REF);
      if (realClassName.equals(expectedClassName))
	sendString(null);	// optimization
      else
	sendString(realClassName);
      if (url != null)
	sendString(url.getURL());
      else
	sendString(null);	// null url
      return;
    }
    //
    // test if object is casted
    //
    Class expectedClass = null;
    try {
      expectedClass = Class.forName(expectedClassName);
    } catch (Exception e0) {
      throw new HORBException(e0);
    }
    if (realClass == expectedClass) {
      Proxy sender = null;
      try {
	sender = (Proxy)Class.forName(expectedClassName+"_Proxy").newInstance();
      } catch (Exception e) {
	if (strict)
	  throw new ProxyException(e);
	else {
	  sendByte(IOCI.OBJ_NO_PROXY);
	  if (debug)
	    System.err.println("warning: can't instantiate "+expectedClassName+"_Proxy");
	  return;
	}
      }
      sendByte(IOCI.OBJ_INST);
      sender._sendInstance((IOCI)this /* ioci */, o, loopy, place);
      return;
    } else {			// casted object
      //
      // object has been narrowed. Send the true class name.
      // instance of the class is followed.
      Proxy sender = null;

      try {
	// temporary create proxy object to call _sendInstace(),
	// because it is an instance method.
	sender = (Proxy)Class.forName(realClassName+"_Proxy").newInstance();
      } catch (Exception e3) {
	if (strict)
	  throw new ProxyException(e3);
	else {
	  sendByte(IOCI.OBJ_NO_PROXY);
	  if (debug)
	    System.err.println("warning: can't instantiate "+expectedClassName+"_Proxy");
	  return;
	}
      }
      sendByte(IOCI.OBJ_CAST);
      sendString(realClassName);
      sender._sendInstance((IOCI)this /* ioci */, o, loopy, place);
      return;
    }
  }

  /**
   * Receives an object. First receive what type of object (e.g. null object,
   * object reference), and then receive object. If the the class of
   * the receiving object is equal to 'expectedClassName', it's receiver
   * class (expectedClassName_Proxy) is used to pass it in.
   *
   * @param expectedClassName Name of expected class to be passed in.
   * @param gb  another side of loop checker.
   * @param place	IOCI.LOC_*. for example, LOC_FILE, LOC_NETWORK..
   *
   * @exception ProxyException couldn't instantiate a proxy object
   * @exception HORBException illegal tag type is detected
   * @exception IOException network error
   */   
  public Object recvObject(String expectedClassName, Goldberg gb, byte place) throws HORBException, IOException, ProxyException {
    String className;
    Object obj;

    //
    // receive what type of object is passed from sender
    //
    byte flag = recvByte();

    switch (flag) {
    case IOCI.OBJ_NULL:
      return null;
    case IOCI.OBJ_LOOPY:
      int num = recvInt();
      return gb.get(num);
    case IOCI.OBJ_NO_PROXY:
      return null;
    case IOCI.OBJ_STRING:
      obj = (Object)recvString();
      gb.put(obj);
      return obj;
    case IOCI.OBJ_REF:
      className = recvString(); // foo_Proxy
      String url = recvString(); // this can be null
      Proxy o = null;
      try {
	if (className == null)	// for optimization
	  className = expectedClassName;
	o = (Proxy)Class.forName(className).newInstance();
      } catch (Exception e) {
	throw new ProxyException(e);
      }
      if (url != null) {
	try {
	  HorbURL objectURL = new HorbURL(url);
	  o._connect(objectURL, username, passwd);
	} catch (Exception e) {
	  if (strict) {
	    System.err.println("warning: reconnection to "+url+" failed during passing an object by reference.");
//	    throw new ProxyException(e);
	  } else
	    System.err.println("warning: reconnection failed for "+url);
//!
	  e.printStackTrace();
	}
      }
      gb.put(o);
      return (Object)o;
    case IOCI.OBJ_INST:
      Proxy receiver = null;
      obj = null;
      try {
	receiver = (Proxy)Class.forName(expectedClassName+"_Proxy")
	  .newInstance();
      } catch (Exception e2) {
	throw new ProxyException(e2);
      }
      obj = receiver._recvInstance((IOCI)this /* ioci */, null, false, gb, place);
      return obj;
    case IOCI.OBJ_CAST:
      className = recvString();
      obj = null;
      //
      // temporary create proxy object to call _recvInstance,
      // because it is an instance method
      Proxy casted = null;
      try {
	casted = (Proxy)Class.forName(className+"_Proxy").newInstance();
      } catch (Exception e2) {
//	e2.printStackTrace();
	throw new ProxyException(e2);
      }
      obj = casted._recvInstance((IOCI)this /* ioci */, null, false, gb, place);
      return obj;
    default:
      throw new HORBException("Illegal data tag during receiving an object");
    }
  }

  /**
   * set propert of IOCI.
   * <pre>
   * setProperty(IOCI.PROP_DEBUG, null) turns debug flag true.
   * setProperty(IOCI.PROP_NO_DEBUG, null) turns debug flag false.
   * setProperty(IOCI.PROP_STRICT, null) turns strict flag true.
   * setProperty(IOCI.PROP_NO_STRICT, null) turns strict flag false.
   * </pre>
   */
  public void setProperty(int request, Object obj) {
    switch (request) {
    case IOCI.PROP_DEBUG:
      debug = true;
      break;
    case IOCI.PROP_NO_DEBUG:
      debug = false;
      break;
    case IOCI.PROP_STRICT:
      strict = true;
      break;
    case IOCI.PROP_NO_STRICT:
      strict = false;
      break;
    }
  }

  public Object getProperty(int request, Object obj) { 
    return null; 
  }

  public final  String getThreadName() {
    return threadName;
  }

  /**
   * return username of the client. null means anonymous access.
   */
  public final String getUsername() {
    return username;
  }

  public abstract void connectServer(String host, int port) throws IOException;
  public abstract void sendIOCISignature() throws IOException;
  public abstract void kick()  throws IOException;
  public abstract void release();
  public abstract boolean isConnected();

  public abstract void sendByte(byte value)throws IOException;
  public abstract void sendShort(short value)throws IOException;
  public abstract void sendInt(int value)throws IOException;
  public abstract void sendString(String value)throws IOException;

  public abstract byte recvByte()throws IOException;
  public abstract short recvShort()throws IOException;
  public abstract int recvInt()throws IOException;
  public abstract String recvString()throws IOException;
}
