/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import java.io.DataOutputStream;
import java.io.DataInputStream;
import java.io.IOException;

/**
 * Services provided by IOCI. Programmers can get these services
 * from IOCI.
 * <pre>
 *  Client side:
 *	Server_Proxy server = new Server_Proxy(url);
 *	IOCIService ioci = (IOCIService)server._getIOCI();
 *	String threadName = ioci.getThreadName();
 *  Server side:
 *      IOCIService ioci = (IOCIService)HORBServer.getIOCI();
 *	ioci.setProperty(PROP_DEBUG, null); // debug flag of the IOCI
 *  </pre>
 *	
 */
public interface IOCIService {

  /*
   * property manupilation
   */

  /**
   * turn on debug flag of the IOCI object. Default off.
   * <pre>
   *  Server side:
   *      IOCIService ioci = (IOCIService)HORBServer.getIOCI();
   *      ioci.setProperty(PROP_DEBUG, null); // debug flag of the IOCI
   *  Client side:
   *	  Server_Proxy server = new Server_Proxy(url);
   *	  IOCIService ioci = (IOCIService)server._getIOCI();
   *      ioci.setProperty(PROP_DEBUG, null); // debug flag of the IOCI
   * </pre>
   */
  public static final byte PROP_DEBUG = 0;

  /** turn off debug flag. */
  public static final byte PROP_NO_DEBUG = 1;
  /** 
   * strict check. Default is off. If on, an exception occures when:
   * <UL>
   * <LI>reconnection failed
   * at the receiver side during object passing by reference.
   * <LI>failure to allocate a proxy object. For example, 
   * Foo_Proxy.class not found during passing an instance of Foo.
   * </UL>
   */
  public static final byte PROP_STRICT = 2;
  /** no strict check */
  public static final byte PROP_NO_STRICT = 3;
  /**
   * loop check on. If on, loop or merging data structure is passed as
   * it is. If false, a program fails into an infinit loop when it
   * passes a loop structure. Loop check causes slite speed penalty.
   * Default is on.
   */
  public static final byte PROP_LOOP_CHECK = 4;
  /** no loop check. */
  public static final byte PROP_NO_LOOP_CHECK = 5;
  /** answer is needed. (This property is not used.) */
  public static final short PROP_ANSWER = 6;
  /** no answer is needed. (This property is not used.) */
  public static final short PROP_NO_ANSWER = 7;
  /**
   * users can use number from this to manipulate users IOCI.
   */
  public static final short PROP_USER = 10000;

  /*
   * functions for client side
   */
  /** returns the thread name of the remote object. */
  String getThreadName();

  /** 
   * set properties of IOCI.
   * <pre>
   * Examples:
   *  Server side:
   *      IOCIService ioci = (IOCIService)HORBServer.getIOCI();
   *      ioci.setProperty(PROP_DEBUG, null); // debug flag of the IOCI
   *  Client side:
   *	  Server_Proxy server = new Server_Proxy(url);
   *	  IOCIService ioci = (IOCIService)server._getIOCI();
   *      ioci.setProperty(PROP_DEBUG, null); // debug flag of the IOCI
   * </pre>
   * @param request request code. PROP_*
   * @param object request specific option
   */
  void setProperty(int reqest, Object object);

  /** 
   * get properties of IOCI. Currently no function is defined.
   * @param request request code. PROP_*
   * @param object request specific option
   * @return object request specific option
   */
  Object getProperty(int request, Object object);

  /**
   * get the major version number of this IOCI.
   * different numbers mean incompatibility.
   */
  short getMajorVersion();

  /**
   * get the major version number of this IOCI. upward compatible with
   * other minor version.
   */
  short getMinorVersion();

  /**
   * true if connection is connected normally. <em>Don't rely on
   * this method of BasicIOCI. </em>
   * @see BasicIOCI#isConnected.
   */
  boolean isConnected();

  /**
   * return number of bytes available without blocking. 
   */
  int available() throws IOException;

  /**
   * release the connection. If you want disconnect a connection
   * between a client, use this method. Use proxy._release() from
   * the client side.
   * <pre>
   *  Server side:
   *     IOCIService ioci = (IOCIService)HORBServer.getIOCI();
   *     ioci.release();
   *  Client side:
   *     proxy._release();
   * </pre>
   * @see Proxy#_release
   *   
   */
  void release();

  /**
   * returns the hostname of the local host.
   */
  String getLocalHostName() throws IOException;

  /**
   * returns the network address of the local host.
   * Note that BasicIOCI has getLocalInetAddress().
   * @see BasicIOCI#getLocalInetAddress.
   */
  byte[] getLocalAddress() throws IOException;

  /**
   * returns peer's hostname.
   */
  String getHostName();

  /**
   * returns peer's network address. Note that BasicIOCI has
   * getInetAddress().
   * @see BasicIOCI#getInetAddress
   */
  byte[] getAddress();

  /**
   * return username. null means anonymous access. This method makes sense
   * at the server side.
   */
  String getUsername();

}
