/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import horb.orb.*;
import java.io.*;
import java.util.*;

/**
 * ObjectStorage is an (very simple) object database. It provides
 * saving, loading objects and file locking.
 * <p>
 * If you want to have a system ObjectStorage, start it from
 * a configuration file.
 * <pre>
 *   sample.conf:
 *	object[0].className=horb.orb.ObjectStorage
 *	object[0].implementation=horb.orb.ObjectStorage_Impl
 *	object[0].objectID=ObjectStorage
 *   command line:
 *	horb -conf sample.conf
 * </pre>
 * This object storage can be accessed as;
 * <pre>
 *   local:
 *	ObjectStorage os = (ObjectStorage)HORBServer.getObject("ObjectStorage");
 *	os.saveObject(obj, "obj.fof");
 *   remote:
 *	ObjectStorage os = new ObjectStorage_Proxy(new HorbURL("horb://host/ObjectStorage"));
 *	os.saveObject(obj, "obj.fof");
 * </pre>
 */ 
public interface ObjectStorage {
  /**
   * save an object as a given filename. Caution for security.<p>
   * If a client does not give a filename, a temporary file is created
   * and the filename is returned. If a client gives a filename,
   * the filename may not include ".." in it and it may not be an
   * absolute path. Thus filenames must be relative filename, for example,
   * "data.fof" or "directory/data.fof".
   *
   * If ObjectStorage_Impl was instantiated with a directory name,
   * the directory name is prepended to the filename.
   *
   * @param fname filename.
   * @return filename.
   * @see ObjectStorage_Impl
   */
  public String saveObject(Object obj, String fname) throws HORBException, IOException;

  /**
   * returns an object from a given filename. Caution for security.<p>
   * Filename may not include ".." in it and it may not be an
   * absolute path. Thus filenames must be relative filename, for example,
   * "data.fof" or "directory/data.fof".
   * @param fname filename.
   * @return object saved in fname.
   */
  public Object loadObject(String fname) throws HORBException, IOException;

  /**
   * get lock of a file. The file may not be exist yet. <p> This locking
   * system runs on memory, that is, the locking is
   * valid while the HORB server is running.<p>
   * Here is an example of atomic update of an object.
   * <pre>
   *   HorbURL url = new HorbURL(host, "ObjectStorage");
   *   ObjectStorage_Proxy os = new ObjectStorage_Proxy(url);
   *   os.lock(file, 0);
   *   Data data = os.loadObject(file);
   *   data.update();
   *   os.saveObject(data, file);
   *   os.unlock(file);
   * </pre>
   *   
   */
  public boolean lock(String fname, long timeout) throws HORBException;

  /**
   * Unlock file. Unlocking should be done within the same session
   * (connection).
   * Once you lock a file from a proxy object of ObjectStorage,
   * the lock should be released before releasing the connection of
   * the proxy object.<p>
   *
   * If someone is waiting for the file, unlocking gives 
   * the next lock to the someone.
   *
   * @param fname filename to be unlocked.
   * @exception IllegalMonitorStateException if the current client is not
   * the owner of the lock.
   */
  public void unlock(String fname) throws IllegalMonitorStateException;

  /**
   * Returns locking condition of a file. If you need atomic test and set,
   * don't use this method. Use lock() instead of this.
   *
   * @param fname filename to be unlocked.
   * @exception IllegalMonitorStateException if the current client is not
   * the owner of the lock.
   */
  public boolean isLocked(String fname);

  /**
   * delete a file. Returns true if the file was deleted.
   */
  public boolean delete(String fname) throws HORBException, IOException;

  /**
   * rename a file. Returns true if the file could be renamed.
   */
  public boolean rename(String fname1, String fname2) throws HORBException, IOException;

  /**
   * copy a file from fname1 to fname2. Lock files before calling copy(), 
   * if needed.
   */
  public void copy(String fname1, String fname2) throws IOException;

  /**
   * returns a temporary filename.
   */
  public String makeTemporaryFileName();

  /**
   * Returns true if the file exists.
   */
  public boolean exists(String fname) throws HORBException, IOException;
}
