/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */


package horb.orb;

import horb.orb.*;
import java.io.*;
import java.util.*;

/**
 * Object storage is an (very simple) object database. This version
 * has very limitted functionality, just saving, loading objects and
 * locking.
 * @see ObjectStorage
 */ 
public class ObjectStorage_Impl implements ObjectStorage {
  Hashtable locktable;
  String dir;
  Random random = new Random(System.currentTimeMillis());

  public ObjectStorage_Impl() {
    locktable = new Hashtable();
  }

  public ObjectStorage_Impl(String dir) {
    locktable = new Hashtable();
    this.dir = dir;
  }

  /**
   * save an object as a given filename.<p>
   * If a filename is not specifed, a temporary file is created
   * and the filename is returned. If a client gives a filename,
   * the filename may not include ".." in it and it may not be an
   * absolute path. Thus filenames must be relative filename, for example,
   * "data.fof" or "directory/data.fof".
   *
   * If this ObjectStorage_Impl was instantiated with a directory name,
   * the directory name is prepended to the filename.
   *
   * @param fname filename.
   * @return filename.
   */
  public String saveObject(Object obj, String fname) throws IOException {
    File file = null;

    if (fname == null) {	// make a temporary filename
      do {
	fname = "H"+random.nextLong()+".fof";
	file = new File(fname);
      } while (file.exists());
    } else {
      file = pathCheck(fname);
    }
    FileIOCI ioci = new FileIOCI(file);
    ioci.save(obj);
    return fname;
  }

  /**
   * returns an object from a given filename. Caution for security.<p>
   * Filename may not include ".." in it and it may not be an
   * absolute path. Thus filenames must be relative filename, for example,
   * "data.fof" or "directory/data.fof".
   * @param fname filename.
   * @return object saved in fname.
   */
  public Object loadObject(String fname) throws IOException {
    File file = pathCheck(fname);
    FileIOCI ioci = new FileIOCI(file);
    return ioci.load();
  }

  /**
   * get lock of a file. The file may not be exist yet. <p> This locking
   * system runs on memory, that is, the locking is
   * valid while the HORB server is running.<p>
   * Here is an example of atomic update of an object.
   * <pre>
   *   HorbURL url = new HorbURL(host, "ObjectStorage");
   *   ObjectStorage_Proxy os = new ObjectStorage_Proxy(url);
   *   os.lock(file, 0);
   *   Data data = os.loadObject(file);
   *   data.update();
   *   os.saveObject(data, file);
   *   os.unlock(file);
   * </pre>
   *   
   */
  public boolean lock(String fname, long timeout) throws HORBException {
    Lock fl = null;
    File file = pathCheck(fname);

    synchronized (locktable) {
      if ((fl = (Lock)locktable.get(fname)) == null) {
	fl = new Lock();
	locktable.put(fname, fl);
	return true;
      }
    }
    return fl.lock(timeout);
  }

  /**
   * Unlock file. Unlocking should be done within the same session
   * (connection).
   * Once you lock a file from a proxy object of ObjectStorage,
   * the lock should be released before releasing the connection of
   * the proxy object.<p>
   *
   * If someone is waiting for the file, unlocking gives 
   * the next lock to the someone.
   *
   * @param fname filename to be unlocked.
   * @exception IllegalMonitorStateException if the current client is not
   * the owner of the lock.
   */
  public void unlock(String fname) throws IllegalMonitorStateException {
    Lock fl = null;
    if ((fl = (Lock)locktable.get(fname)) == null)
      throw new IllegalMonitorStateException(fname);
    fl.unlock();
  }

  /**
   * Returns locking condition of a file. If you need atomic test and set,
   * don't use this method. Use lock() instead of this.
   *
   * @param fname filename to be unlocked.
   * @exception IllegalMonitorStateException if the current client is not
   * the owner of the lock.
   */
  public boolean isLocked(String fname) {
    Lock fl = null;
    if ((fl = (Lock)locktable.get(fname)) == null)
      return false;
    return fl.isLocked();
  }

  /**
   * delete a file. Returns true if the file could be deleted.
   */
  public boolean delete(String fname) throws HORBException, IOException {
    File file = pathCheck(fname);
    return file.delete();
  }
    
  /**
   * Returns true if the file exists.
   */
  public boolean exists(String fname) throws HORBException, IOException {
    File file = pathCheck(fname);
    return file.exists();
  }

  /**
   * rename a file. Lock files before calling rename(), if needed.
   */
  public boolean rename(String fname1, String fname2) throws HORBException, IOException {
    File file1 = pathCheck(fname1);
    File file2 = pathCheck(fname2);
    return file1.renameTo(file2);
  }

  /**
   * copy a file. Lock files before calling rename(), if needed.
   */
  public void copy(String fname1, String fname2) throws IOException {
    File file1 = pathCheck(fname1);
    File file2 = pathCheck(fname2);
    FileInputStream is = new FileInputStream(file1);
    FileOutputStream os = new FileOutputStream(file2);
    byte buf[] = new byte[(1024*128)];
    int n;
    while ((n = is.read(buf)) >= 0)
      os.write(buf, 0, n);
    is.close();
    os.close();
  }

  /**
   * returns a temporary filename. 
   */
  public String makeTemporaryFileName() {
    String fname;
    File file;
    do {
      fname = "H"+random.nextLong()+".fof";
      file = new File(fname);
    } while (file.exists());
    return fname;
  }

  private File pathCheck(String fname) throws HORBException {
    if (fname.indexOf("..") >= 0)
      throw new HORBException("\"..\" is not allowed. "+fname);
    File file = new File(fname);
    if (file.isAbsolute())
      throw new HORBException("absolute path is not alloewd. "+fname);
    if (dir != null) {
      fname = dir + File.separator + fname;
      file = new File(fname);
    }
    return file;
  }

}
