/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */

package horb.orb;

import java.util.Hashtable;
import java.util.Enumeration;

/**
 * Object Table.
 *
 * records number of clients for every objects.
 */
class ObjectTable extends Hashtable {

  int numObject;		// system wide num object
  int numThread;		// system wide num thread
  int numConnect;		// system wide num connect

  final ObjectInfo get(String objectID) {
    return (ObjectInfo)super.get(objectID);
  }

  final synchronized ObjectInfo put(String className, String objectID, Skeleton skeleton, int port, boolean daemon) {
    ObjectInfo oi = new ObjectInfo(className, objectID, skeleton, port, daemon);
    super.put(objectID, oi);
    numObject++;
    return oi;
  }

  final synchronized void remove(String objectID) {
    super.remove(objectID);
    numObject--;
  }

  final synchronized Skeleton use(String objectID) {
    ObjectInfo e;
    if ((e = (ObjectInfo)get(objectID)) == null)
      return null;
    synchronized (e) {
      e.use();
      numThread++;
      numConnect++;
      return e.getSkeleton();
    }
  }
  
  final synchronized void use(ObjectInfo e) {
    synchronized (e) {
      e.use();
      numThread++;
      numConnect++;
    }
  }
  
  final synchronized void unuse(String objectID) {
    ObjectInfo e;
    if ((e = (ObjectInfo)get(objectID)) == null)
      return;
    synchronized (e) {
      numThread--;
      if (e.unuse() == 0 && !e.isDaemon())
	remove(objectID);
    }
  }

  final synchronized void unuse(ObjectInfo e) {
    synchronized (e) {
      numThread--;
      if (e.unuse() == 0 && !e.isDaemon())
	remove(e.os.objectID);
    }
  }

  /**
   * retern current status of objects.
   */
  final synchronized ObjectStatus[] getObjectStatus() {
    long now = System.currentTimeMillis();
    int size = size();
    Enumeration e = elements();
    ObjectStatus os[] = new ObjectStatus[size];
    for (int i = 0; i < size && e.hasMoreElements(); i++) {
      ObjectInfo oi = (ObjectInfo)e.nextElement();
      os[i] = oi.os.get();
      os[i].currentTime = now;
    }
    return os;
  }

  /**
   * retern current status of a specified object.
   */
  final synchronized ObjectStatus getObjectStatus(String objectID) {
    ObjectInfo oi;
    long now = System.currentTimeMillis();
    if ((oi = (ObjectInfo)get(objectID)) == null)
      return null;
    oi.os.currentTime = now;
    return oi.os.get();
  }
}
