/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */

package horb.orb;
import java.io.*;

/**
 * Proxy object interface. Any proxy object has these methods.
 * Proxy methods may throws HORBException.
 */
public interface Proxy {
  /**
   * Returns major version number of this proxy class.
   */
  public short _getMajorVersion();

  /**
   * Returns minor version number of this proxy class.
   */
  public short _getMinorVersion();

  /**
   * Returns IOCI object.
   */
  public IOCI _getIOCI();

  /**
   * Set new IOCI object.
   */
  void _setIOCI(IOCI ioci);

  /**
   * Returns URL of the remote object.
   */
  HorbURL _getObjectURL();

  /**
   * Returns thread name of the remote HORB thread.
   */
  String _getThreadName();

  /**
   * connect this proxy to an object that is designated by objectURL
   * with given username and password. This method can be used to
   * use arbitrary IOCI implementation.
   * <pre>
   *    Server_Proxy server = new Server_Proxy(url, username, password);
   * is equivalent to;
   *    Server_Proxy server = new Server_Proxy();
   *    IOCI ioci = new BasicIOCI();
   *    server._setIOCI(ioci);
   *    server._connect(url, username, password);
   * </pre>
   *
   * @param objectURL Object URL to be connected.
   * @param username username.
   * @param passwd password.
   * @exception HORBException object not found in remote system, or can't create IOCI
   * @exception NetException Network error
   */
  void _connect(HorbURL objectURL, String username, String passwd) throws HORBException, NetException;

  /**
   * Releases the connection to the remote object. This proxy object
   * becomes no longer usable.
   */
  void _release();

  /** 
   * invite a server object to connect to an object (invitor). If the server
   * calls accept() method of the corresponding skeleton object, the
   * server object can call methods of the object. See examples/invitation.
   * Username should be a valid username for inviting client object, or null.
   *
   * @param clientURL  invitor's URL
   * @param username	username to use the clientURL. If null, the username
   *			used when connecting to the server object is userd.
   *			Username is needed to protect the client system 
   *			against the server system 
   *			because the remote server can be a fake.
   * @param passwd	password of the username. Null means anonymous
   *			password. Password is needed to
   *			protect the client system against the server system 
   *			because the remote server can be a fake.
   * @exception NetException Network error
   */ 
  void _invite(HorbURL clientURL, String username, String passwd) throws HORBException, NetException;

  /**
   * (Internal) setup connection to invited path. Don't call this method
   * from user's program.
   */
  void _invited(IOCI ioci, HorbURL clientURL, String username, String passwd) throws HORBException, IOException;

  /**
   * return true if the previously issued asynchronous method call
   * has completed.
   * <pre>
   *   remote.method_Request();		// issue method request
   *   do {
   *       // another work
   *   } while (remote._available() == false);
   *   answer = remote.method_Receive(); // get answer
   * </pre>
   */
  boolean _available();

  /**
   * wait for completion of the previously issued asynchronous method call.
   * @param timeout milli seconds for timeout.
   * @return true if the method call completed until timeout. returns false
   *   		if timeout.
   */
  public boolean _wait(long timeout) throws InterruptedException;

  /**
   * Set asynchronous method call handler. Once a handler is set,
   * the handler is called with tag when asynchronous method call
   * completes.
   * @param handler handler to be called.
   * @param tag  arbitrary integer number.
   * @see AsyncMethodHandler
   */
  public void _setHandler(AsyncMethodHandler handler, int tag);

  /**
   * (Internal) Sends an object 'o' that is this type, to ioci. Don't
   * call from user's program.
   * This is a static method actually.
   * @param ioci ioci object
   * @param o    if superclass object to be filled, otherwise null.
   * @param loopy loop checker.
   * @param place source/destination of object passing.
   * @exception HORBException Exception occures during sending the instance.
   * @exception IOException Failure of socket connection
   */
  public void _sendInstance(IOCI ioci, Object o, Loopy loopy, byte place) throws HORBException, IOException;

  /**
   * (Internal) Receives an object of this type and returns the object. 
   * Don't call this user's program. This is a static method actually.
   *
   * @param ioci ioci object
   * @param o    if superclass object to be filled, otherwise null
   * @param superclass true if it is a superclass
   * @param Goldberg another side of loop checker.
   * @param place source/destination of object passing.
   *
   * @exception HORBException Exception occures during receiving the instance
   * @exception IOException Failure of socket connection
   */
  public Object _recvInstance(IOCI ioci, Object o, boolean superclass, Goldberg gb, byte place) throws HORBException, IOException;
}
